<%@ page contentType="text/html;charset=utf-8"%>
<%@ taglib uri="http://www.springframework.org/tags" prefix="spring" %>
////////////////////////////////////////////////////////////////////////////////
// REGULAR EXPRESSION STRINGS/GENERIC FUNCTIONS FOR PATTERN MATCHING.         //
// Since '\' character also denotes special character in a Javascript string, //
// '\\' required to express '\' for a regular expression stored in a string.  //
////////////////////////////////////////////////////////////////////////////////

// --- NUMBERS --- //

var re_number = /^\d+$/;

// An infinite length integer
var re_integer = "(\\+|\\-)?(\\d+)";

// An infinite length positive integer
var re_pos_integer = "\\+?(\\d+)";

// An infinite length negative integer
var re_neg_integer = "\\-?(\\d+)";

// Any number of zeroes
var re_zero = "(\\+|\\-)0+";


// --- STRINGS --- //

// All whitespace
var re_all_whitespace = "^\\s*$";

// Starting/ending whitespace
var re_start_end_whitespace = "^\\s*|\\s*$";

// Host/domain name (trendmicro.com, trendmicro, ...)
var re_valid_domain_chars = "^([a-zA-Z0-9]([-a-zA-Z0-9]{0,61}[a-zA-Z0-9])?\\.)*[a-zA-Z0-9][-a-zA-Z0-9]{0,61}[a-zA-Z0-9]$";

// Subdomain name (*.com, *.trend.com)
var re_valid_subdomain_chars = "^\\*\\.([a-zA-Z0-9]([-a-zA-Z0-9]{0,61}[a-zA-Z0-9])?\\.)*[a-zA-Z0-9][-a-zA-Z0-9]{0,61}[a-zA-Z0-9]$";

// Invalid usage of '.' in domain-style string (.abc, abc., a..b, etc).
// This is sufficient for checking the pattern of an IP address/domain name
var re_invalid_domain_period_usage = "(^\\.)|(\\.$)|(\\.\\.)";

// IPv4 address pattern (must separately verify values of $1, $2, $3, and $4)
var re_ipv4_pattern = "(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})";

//IPv6 address pattern
var re_ipv6_pattern = "^((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:)))$"

// E-mail address pattern
var re_email_address_pattern = "(.+)@(.+)";

// Valid e-mail address characters (user) \[^\\w<>\\.!#$%&'*+-\\/=\\?^`{\\|}~\"\\(\\)\]
// TODO: Is "*" valid here? var re_valid_email_address_chars_user = "[\\w<>.!#$%&'*+\\-/=?^`{|}~\"()]";
var re_valid_email_address_chars_user = "[\\w.!#$%&'+\\-/=?^`{|}~\"()]";

// Valid e-mail address characters (domain)
var re_valid_email_address_chars_domain = "[\\w.!#$%&'*+\\-/=?^`{|}~\"]";

// Only matches a string with all non-DBCS characters (ASCII value <= 127)
var re_non_dbcs_chars = "[\\x00-\\xEF]*";

var re_windows_path_pattern = /^[a-zA-Z]:[\\\/](.*)$/;

var re_linux_path_pattern = "^/[-a-zA-Z0-9_./]{1,255}$";

// Invalid file path characters
var re_invalid_path_chars = /[:"*?<>|]/i;

// Invalid file path patterns
var re_invalid_path_patterns = /(\\{2})|(\/{2})|(\.{2})/g;

// Contains a " character
var re_contains_double_quote = /["]/g;

var re_invalid_hypen = "(^\\-)|(\\-\\.)|(\\.\\-)|(\\-$)";

var re_invalid_number="(^[0-9])|(\\.[0-9])";
var re_invalid_input_shell="(`|#|&&|\\|\\|)"
////////////////////////////////////////////////////////////////////////////////
// PROTOTYPES                                                                 //
// Extend the functionality of existing objects.                              //
////////////////////////////////////////////////////////////////////////////////

//
// String object
//
String.prototype.removeWhitespace = _removeWhitespace;
String.prototype.removeAllWhitespace = _removeAllWhitespace;
String.prototype.isWhitespace = _isWhitespace;

// [DESC] Removes whitespace from beginning/end of string
// [IN]   None
// [OUT]  String: start/end whitespace removed
function _removeWhitespace()
{
    return this.replace(new RegExp(re_start_end_whitespace,"g"),"");
}

// [DESC] Removes all whitespace
// [IN]   None
// [OUT]  String: all whitespace removed
function _removeAllWhitespace()
{
    return this.replace(new RegExp("\\s","g"),"");
}

// [DESC] Returns true if string is all whitespace
// [IN]   str[String]: String
// [OUT]  True if string is all whitespace, else false
function _isWhitespace()
{
    return (new RegExp(re_all_whitespace)).test(this);
}


////////////////////////////////////////////////////////////////////////////////
// FORMAT CHECKING FUNCTIONS                                                  //
// These functions check the format of the input string and return true       //
// for a correct match.                                                       //
////////////////////////////////////////////////////////////////////////////////

// [DESC] Checks the format of an absolute pathname
// [IN]   str [String]: Integer
// [OUT]  Boolean
function checkAbsolutePath_Format(str)
{
    return !(re_invalid_path_chars.test(str) || re_invalid_path_patterns.test(str));
}

// [DESC] Checks the format of an integer
// [IN]   str [String]: Integer
// [OUT]  Boolean
function checkInteger_Format(str)
{
    return (new RegExp("^" + re_integer + "$")).test(str);
}

// [DESC] Checks the format of an IPv4 IP address 
// [IN]   str [String]: IPv4 IP address
// [OUT]  Boolean
function checkIPv4_Format(str)
{
    re = new RegExp("^" + re_ipv4_pattern + "$","gi");
    ipArray = re.exec(str);
    if (!ipArray)
        return false;

    for (var i = 1; i <= 4; i++)
        if (ipArray[i] < 0 || ipArray[i] > 255)
            return false;

    return true;
}

// [IN]  str [String]: IPv4 IP address
// [OUT] Integer; 0 - failed.
// added by tonylow 20061207
function convert_ipv4ToInt(str)
{
    re = new RegExp("^" + re_ipv4_pattern + "$", "gi");
    ipArray = re.exec(str);
    if (!ipArray)
        return 0;

    var ip = 0;
    var by = 3;
    for (var i = 1; i <= 4; i++)
	{
		var num = ipArray[i];
		if (num < 0 || num > 255)
			return 0;
		else
			ip += num << (by * 8);
		by--;
	}

    return ip;
}

// [DESC] Checks the format of an IPv4 IP address
// [IN]   str [String]: IPv4 IP address
// [OUT]  Boolean
// added by tonylow 20061208 to follow IPV4 standard
function valid_ipv4Addr(str)
{
    re = new RegExp("^" + re_ipv4_pattern + "$", "gi");
    ipArray = re.exec(str);
    if (!ipArray)
        return false;

    for (var i = 1; i <= 4; i++)
        if (ipArray[i] < 0 || ipArray[i] > 255)
            return false;

	ip1 = ipArray[1];
	ip2 = ipArray[2];
	ip3 = ipArray[3];
	ip4 = ipArray[4];
    if ((ip1 & 0x80) == 0)	// A class
    {
		//if (ip > 0x1000000 && ip < 0x7F000000)
        if (ip1 > 0x1 && ip1 < 0x7F)
        {
            return true;
        }
        if (ip1 == 0x1 && ip4 > 0)
        {
            return true;
        }
        return false;
    }
    if ((ip1 & 0xC0) == 0x80)	// B class
    {
		//if ((long)ip > (long)0x80000000 && (long)ip < (long)0xC0000000)
        if (ip1 > 0x80 && ip1 < 0xC0)
        {
            return true;
        }
        if (ip1 == 0x80)
        {
			if (ip2 > 0 || ip3 > 0)
			{
				return true;
			}
			if (ip4 > 0)
			{
				return true;
			}
        }
        return false;
    }
    if ((ip1 & 0xE0) == 0xC0)	// C class
    {
		//if ((long)i > (long)0xC0000000 && (long)i < (long)0xE0000000)
        if (ip1 > 0xC0 && ip1 < 0xE0)
        {
            return true;
        }
        if (ip1 == 0xC0)
        {
			if (ip2 > 0 || ip3 > 0)
			{
				return true;
			}
			if (ip4 > 0)
			{
				return true;
			}
        }
        return false;
    }
    return false;
}

// [IN]  mask [String]: IPv4 IP mask
// [OUT] Boolean
// added by tonylow 20061207
function valid_ipv4Mask(mask)
{
    re = new RegExp("^" + re_ipv4_pattern + "$", "gi");
    ipArray = re.exec(mask);
    if (!ipArray)
        return false;

    for (var i = 1; i <= 4; i++)
        if (ipArray[i] < 0 || ipArray[i] > 255)
            return false;

	var ip4 = ipArray[4];
	if (ipArray[1] != 255 || ip4 == 255)
	{
        return false;
	}

    if (ip4 != 254 && 
        ip4 != 252 && 
        ip4 != 248 && 
        ip4 != 240 && 
        ip4 != 224 && 
        ip4 != 192 && 
        ip4 != 128 && 
        ip4 != 0)
    {
        return false;
    }

	var ip2 = ipArray[2];
	var ip3 = ipArray[3];
	if (ip4 != 0)
	{
		if (ip2 != 255 || ip3 != 255)
		{
        	return false;
		}
		// now (ip1 == 255) && (ip2 == 255)&& (ip3 == 255) && (ip4 == 254~128)
		return true;
	}

	// now (ip1 == 255) && (ip4 == 0)

   	if (ip3 != 255 && 
   		ip3 != 254 && 
       	ip3 != 252 && 
       	ip3 != 248 && 
       	ip3 != 240 && 
       	ip3 != 224 && 
       	ip3 != 192 && 
       	ip3 != 128 && 
       	ip3 != 0)
   	{
       	return false;
   	}
	if (ip3 != 0)
	{
		if (ip2 != 255)
		{
       		return false;
		}
		// now (ip1 == 255) && (ip2 == 255) && (ip3 == 255~128) && (ip4 == 0)
		return true;
	}

	// now (ip1 == 255) && (ip3 == 0) && (ip4 == 0)

   	if (ip2 == 255 || 
  		ip2 == 254 || 
    	ip2 == 252 || 
    	ip2 == 248 || 
       	ip2 == 240 || 
       	ip2 == 224 || 
  	   	ip2 == 192 || 
       	ip2 == 128 || 
       	ip2 == 0)
   	{
       	return true;
   	}
	return false;
}

// [IN]  ipstr [String]: IPv4 IP address
//       maskstr [String]: IPv4 IP mask
//       gatewaystr [String]: IPv4 gateway
// [OUT] Boolean
// added by tonylow 20061207
function valid_gateway(ipstr, maskstr, gatewaystr)
{
	if (checkIPv4_Format(gatewaystr) == false)
		return false;
	if (valid_ipv4Mask(maskstr) == false)
		return false;
	if (checkIPv4_Format(ipstr) == false)
		return false;

	ip = convert_ipv4ToInt(ipstr);
	if (ip == 0)
		return false;
	mask = convert_ipv4ToInt(maskstr);
	if (mask == 0)
		return false;
	gw = convert_ipv4ToInt(gatewaystr);
	if (gw == 0)
		return false;

    net = ip & mask;
    total = 0xFFFFFFFF ^ mask;
	if ((ip == net) || (ip == (net + total)))
	{
		return false;
	}

	net_ip1 = net >>> 24;
	gw_ip1 = gw >>> 24;
	if (gw_ip1 != net_ip1)
	{
		return false;
	}
	net &= 0x00FFFFFF;
	gw &= 0x00FFFFFF;
    if ((gw <= net) || (gw >= (net + total)))
    {
        return false;
    }
    return true;
}

// [DESC] Checks the format of a Postfix transport map domain entry
// [IN]   str [String]: Postfix host/domain name
// [OUT]  Boolean
function checkPostfixDomain_Format(str)
{
    re = new RegExp("^" + re_postfix_domain + "$");
    return re.test(str);
}

// New style domain format, obey RFC 1034
// [DESC] Checks the format of a standard domain name
// [IN]   str [String]: host/domain name
// [OUT]  Boolean
function checkDomain_Format(str)
{
    if (str.length > 255) 
    	return false;   // Domain string length should be less than 255

    //SEG-TT 243357(TT196446) Cannot set "*@localhost" to "Recipients and Senders" > "Senders"
    if (str == "localhost")
        return true;

    re_chars = new RegExp(re_valid_domain_chars);   // The new revised re_valid_domain_chars also regulate invalid_domain_period_usage and invalid_hypen.
    return (re_chars.test(str));
}

// Almost same as function checkDomain_Format(), this function will support sub-domain, which looks like(*.trend.com)
function checkDomain_Subdomain_Format(str)
{
    if (str.length > 255) 
        return false;   // Domain string length should be less than 255

    //handle sub-domain
    if(str.charAt(0) == '*' && str.charAt(1) == '.')
    {
        str = str.substring(2); //start from the 2nd
    }

    //SEG-TT 243357(TT196446) Cannot set "*@localhost" to "Recipients and Senders" > "Senders"
    if (str == "localhost")
        return true;

    re_chars = new RegExp(re_valid_domain_chars);   // The new revised re_valid_domain_chars also regulate invalid_domain_period_usage and invalid_hypen.
    return (re_chars.test(str));
}

// [DESC] Checks Wildcard subdomain format
// [IN]   str [String]: subdomain name with Wildcard, likes *.trend.com
// [OUT]  Boolean
function checkWildcardSubdomain_Format(str)
{
    re_chars = new RegExp(re_valid_subdomain_chars);  
    return (re_chars.test(str));
}

// [DESC] Checks the format of a wildcard domain name
// [IN]   str [String]: host/domain name
// [OUT]  Boolean
function checkWildcardDomain_Format(str)
{
    allDomains = (str == "*");
    subDomains = checkWildcardSubdomain_Format(str);   
    singleDomain = checkDomain_Format(str);
      return (allDomains || subDomains || singleDomain);
}

// [DESC] Checks the format of a wildcard ibe domain name
// [IN]   str [String]: host/domain name
// [OUT]  Boolean
function checkWildcardIBEDomain_Format(str)
{
    allDomains = (str == "*");
   // subDomains = checkWildcardSubdomain_Format(str);
    singleDomain = checkDomain_Format(str);
      return (allDomains || singleDomain);
}


// [DESC] Checks the format of a standard user@domain e-mail address
// [IN]   str [String]: e-mail address 
// [OUT]  Boolean
function checkEmailAddress_Format(str)
{
    var re = new RegExp("^" + re_email_address_pattern + "$");
    matched_substrings = str.match(re);
    if (!matched_substrings)
        return false;

    var re_user_chars = new RegExp("^" + re_valid_email_address_chars_user + "+$");
    var re_domain_chars = new RegExp("^" + re_valid_email_address_chars_domain + "+$");
    var re_invalid_period_usage = new RegExp(re_invalid_domain_period_usage,"g");

    var userValid = re_user_chars.test(matched_substrings[1]) && 
                    !re_invalid_period_usage.test(matched_substrings[1]);
    var domainValid = re_domain_chars.test(matched_substrings[2]) &&
                      !re_invalid_period_usage.test(matched_substrings[2]);

    return (userValid && domainValid);
}


// [DESC] Checks the format of a wildcard user@domain e-mail address
//        Valid formats: *@*, *@domain.com, user@*.com, *@*.com, user@domain.com
// [IN]   str [String]: e-mail address 
// [OUT]  Boolean
function checkWildcardEmailAddress_Format(str)
{
    var re_pattern = new RegExp("^" + re_email_address_pattern + "$");
    var matched_substrings = re_pattern.exec(str);
    if (!matched_substrings)
        return false;

    var re_user_chars = new RegExp("^" + re_valid_email_address_chars_user + "+$");
    var re_domain_chars = new RegExp("^" + re_valid_email_address_chars_domain + "+$");

    var userValid = (matched_substrings[1] == "*" || re_user_chars.test(matched_substrings[1]));
    var domainValid = checkWildcardDomain_Format(matched_substrings[2]);

    return (userValid && domainValid);
                 
}

// [DESC] Checks the format of a wildcard user@domain e-mail address
//        Valid formats: *@domain.com, user@*.com, *@*.com, user@domain.com
// [IN]   str [String]: e-mail address 
// [OUT]  Boolean
function checkWildcardEmailAddress_Format_Mml(str)
{
    if(str=="*@*")
    {
        return false;
    }

    return checkWildcardEmailAddress_Format( str );
}

// [DESC] Checks the format of multiple e-mail addresses
// [IN]   str [String]: e-mail address(es)
// [OUT]  Boolean
function checkEmailAddressMultiple_Format(str)
{
    re_bad_delimiters = /(;,)|(,;)|(,{2})|(;{2})/g;
	if (re_bad_delimiters.test(str))
	    return false;

    splitString = str.split(new RegExp("[;]"));
    if (splitString.length == 0)
        return false;

    for (var i = 0; i < splitString.length; i++)
    {
        if (!checkEmailAddress_Format(splitString[i]))
            return false;
    }
    return true;
}

// [DESC] Checks string for ASCII-only characters
// [IN]   str [String]: string of ASCII characters
// [OUT]  Boolean
function checkASCII_Format(str)
{
    if (str.isWhitespace())
        return false;
    return (new RegExp("^" + re_non_dbcs_chars + "$")).test(str);
}
// [DESC] Checks string if contains not allowed string.
// [IN]   str [String]: string of ASCII characters
// [OUT]  Boolean
function checkIfContainInvalidChar_Shell(str)
{
    return (new RegExp(re_invalid_input_shell)).test(str);
}

////////////////////////////////////////////////////////////////////////////////
// INPUT MANIPULATION FUNCTIONS                                               //
// Functions for manipulating input.                                          //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

// [DESC] Removes leading zeros from str
// [IN]   str [String]: Number string
// [OUT]  If NaN, return NaN.  Else, return Number.
function trimLeadingZeroes(str)
{
    if (str.isWhitespace())
        return Number.NaN;
    return new Number(str);
}

// [DESC] Clamps the number value stored in field pointed to by obj
//        min <= obj.value <= max
// [IN]   obj [Object]: Form Text input field
//        min [Number]: Minimum value
//        max [Number]: Maximum value
// [OUT]  Boolean: If successful, return true, else return false.
function clampNumberValue(obj, min, max)
{
    if (!obj.value || isNaN(min) || isNaN(max) || min > max)
        return false;
    
    if (obj.value < min)
        obj.value = min;
    else if (obj.value > max)
        obj.value = max;

    return true;
}

// please put onblur="checkdate(this)" in fromDate and toDate textbox and
// put onclick="return doDateCheck(this.form.fromDate, this.form.toDate, this.form.fromHH, this.form.fromMM, this.form.toHH, this.form.toMM)"
// in submit button and the dates will be vaildated.
function stringTrim(strToTrim) {
	return(strToTrim.replace(/^\s+|\s+$/g, ''));
}

function checkdate(objName) {
var datefield = objName;
if (chkdate(objName) == false) {
datefield.select();
alert("<spring:message code="js.baddate"/>");
datefield.focus();
return false;
}
else {
return true;
   }
}

/*
function chkdate(objName) {
	var strDate = objName.value;
	var dd = new Date(strDate);
	if(isNaN(dd)) {
		return false;
	}
	return true;
}
*/

function chkdate(objName) {
var strDatestyle = "US"; //United States date style
var strDate;
var strDateArray;
var strDay;
var StrDay1;
var strMonth;
var strMonth1;
var strYear;
var strYear1;
var strYear2;
var strYear3;
var intday;
var intday1;
var intMonth;
var intMonth1;
var intYear;
var intYear1;
var intYear2;
var intYear3;
var booFound = false;
var datefield = objName;
var strSeparatorArray = new Array("/");
var intElementNr;
var err = 0;
var strMonthArray = new Array(12);
strMonthArray[0] = "Jan";
strMonthArray[1] = "Feb";
strMonthArray[2] = "Mar";
strMonthArray[3] = "Apr";
strMonthArray[4] = "May";
strMonthArray[5] = "Jun";
strMonthArray[6] = "Jul";
strMonthArray[7] = "Aug";
strMonthArray[8] = "Sep";
strMonthArray[9] = "Oct";
strMonthArray[10] = "Nov";
strMonthArray[11] = "Dec";
strDate = datefield.value;
if (strDate.length < 1) {
return true;
}
for (intElementNr = 0; intElementNr < strSeparatorArray.length ; intElementNr++) {
if ( strDate.indexOf( strSeparatorArray[intElementNr] ) != -1 ) {
strDateArray = strDate.split(strSeparatorArray[intElementNr]);
if (strDateArray.length < 3) {
err = 1;
return false;
}
else {
if(strDateArray[2].length==4)
{
strMonth = strDateArray[0];
strDay = strDateArray[1];
strYear = strDateArray[2];
}
else
if(strDateArray[0].length==4)
{
strYear = strDateArray[0];
strMonth = strDateArray[1];
strDay = strDateArray[2];
}
else
return false;
}
booFound = true;
   }
}
if (booFound == false) {
  return false;
}

if (strDay.length == 1) {
strDay = '0' + strDay;
}

if (strMonth.length == 1) {
strMonth = '0' + strMonth;
}

if (strYear.length == 1) {
strYear = '000' + strYear;
}
if (strYear.length == 2) {
strYear = '00' + strYear;
}
if (strYear.length == 3) {
strYear = '0' + strYear;
}
if (strYear.length != 4) {
return false;
}
// US style
/*
if (strDatestyle == "US") {
strTemp = strDay;
strDay = strMonth;
strMonth = strTemp;
}
*/
if (strDay.length != 2) {
return false;
}

if (strMonth.length != 2) {
return false;
}

strDay1 =  strDay.substring(1,2); 
strMonth1 =  strMonth.substring(1,2);
strYear1=strYear.substring(1,2);
strYear2=strYear.substring(2,3);
strYear3=strYear.substring(3,4);


intday = parseInt(strDay, 10);
intday1= parseInt(strDay1, 10);
if (isNaN(intday)|| isNaN(intday1)) {
err = 2;
return false;
}
intMonth = parseInt(strMonth, 10);
intMonth1 = parseInt(strMonth1, 10);
if (isNaN(intMonth)) {
for (i = 0;i<12;i++) {
if (strMonth.toUpperCase() == strMonthArray[i].toUpperCase()) {
intMonth = i+1;
strMonth = strMonthArray[i];
i = 12;
   }
}
if (isNaN(intMonth)|| isNaN(intMonth1)) {
err = 3;
return false;
   }
}
intYear = parseInt(strYear, 10);
intYear1 = parseInt(strYear1, 10);
intYear2 = parseInt(strYear2, 10);
intYear3 = parseInt(strYear3, 10);
if (isNaN(intYear)|| isNaN(intYear1)|| isNaN(intYear2)|| isNaN(intYear3) ) {
err = 4;
return false;
}
if (intMonth>12 || intMonth<1) {
err = 5;
return false;
}
if ((intMonth == 1 || intMonth == 3 || intMonth == 5 || intMonth == 7 || intMonth == 8 || intMonth == 10 || intMonth == 12) && (intday > 31 || intday < 1)) {
err = 6;
return false;
}
if ((intMonth == 4 || intMonth == 6 || intMonth == 9 || intMonth == 11) && (intday > 30 || intday < 1)) {
err = 7;
return false;
}
if (intMonth == 2) {
if (intday < 1) {
err = 8;
return false;
}
if (LeapYear(intYear) == true) {
if (intday > 29) {
err = 9;
return false;
}
}
else {
if (intday > 28) {
err = 10;
return false;
}
}
}
/*
if (strDatestyle == "US") {
datefield.value = strMonth + "/" + strDay + "/" + strYear;
}
else {
datefield.value = strDay + "/" + strMonth + "/" + strYear;
}
*/
return true;
}
function LeapYear(intYear) {
if (intYear % 100 == 0) {
if (intYear % 400 == 0) { return true; }
}
else {
if ((intYear % 4) == 0) { return true; }
}
return false;
}

function doDateCheck(from, to, fromHH ,fromMM, toHH, toMM ) {
if (Date.parse(from.value) < Date.parse(to.value )) {
   return true;
}
else {
if(Date.parse(from.value) == Date.parse(to.value ) )
{
if( ( toHH.value + toMM.value ) >= ( fromHH.value + fromMM.value ) )
return true;
}
if (from.value == "" || to.value == "")
{ 
alert("<spring:message code="js.bothdates"/>");
return false;
}
else 
{
alert("<spring:message code="js.badtimeorder"/>");
return false;
}
   }
}





// please put onblur="checkport(this)" in the item you want to check it is a valid port or not. Port range is from 1-65535.
function checkport(objName) {
var datefield = objName;
if ( objName.value.length==0 || objName.value < 1 || isNaN(objName.value) ||  objName.value > 65535 || objName.value.indexOf('.')!=-1) {
datefield.select();
alert("<spring:message code="js.badport"/>");
datefield.focus();
return false;
}
else {
objName.value=stringTrim(objName.value);
return true;
   }
}

function checkportstr(port)
{
    if(port.length==0 || isNaN(port) || parseInt(port, 10) < 1 || parseInt(port, 10) > 65535 || port.indexOf('.') != -1)
    {
        return false;
    }
    
    return true;
}

// please put onblur="checknumber(this)" in the item you want to check it is a valid number or not. Number range is greater or equal 0.
function checknumber(objName) {
var datefield = objName;
if (  (new RegExp(re_all_whitespace)).test(objName.value) || objName.value.length==0 || objName.value < 0 || isNaN(objName.value) || objName.value.indexOf('.')!=-1  ) {
datefield.select();
alert("<spring:message code="js.badnumber"/>");
datefield.focus();
return false;
}
else {
objName.value=stringTrim(objName.value);
return true;
   }
}
// please put onblur="checkip(this)" in the item you want to check it is a valid ip or not. 
function checkip(objName) {
var datefield = objName;
if (objName.value.length!=0 && isValidIPAddress(objName.value) == false && objName.value!="localhost") {
datefield.select();
alert("<spring:message code="js.badIP"/>");
datefield.focus();
return false;
}
else {
return true;
   }
}

function checkIpv4AndIpv6(objName) {
var datefield = objName;
if (objName.value.length!=0 && isValidIPAddress(objName.value) == false && isValidateIPv6Address(objName.value) == false && objName.value!="localhost") {
datefield.select();
alert("<spring:message code="js.badIP"/>");
datefield.focus();
return false;
}
else {
return true;
   }
}

function isValidIPAddress(ipaddr) {
   var re = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
   if (re.test(ipaddr)) {
      var parts = ipaddr.split(".");
      if (parseInt(parseFloat(parts[0])) == 0) { return false; }
      for (var i=0; i<parts.length; i++) {
         if (parseInt(parseFloat(parts[i])) > 255) { return false; }
      }
      return true;
   } else {
      return false;
   }
}



function isValidIPv6AddressSelfDefinedRange( ipaddr )
{
    if( ! ipaddr )
    {
        return false;
    }
    var splittedByStripParts = ipaddr.split( "-" );
    var standardIPv6Part = splittedByStripParts[ 0 ];
    var rightRange = splittedByStripParts[ splittedByStripParts.length - 1 ]; 

    if ( ! isValidateIPv6Address ( standardIPv6Part ) )
    {
        return false;
    }

    var leftRange ;
    //For this format 2001::10:1.2.3.4    
    if( ipaddr.indexOf( "." ) != -1 )
    {
        var splittedByDotParts = standardIPv6Part.split( "." );
        leftRange = splittedByDotParts [ splittedByDotParts.length - 1 ];
        var reRightRangePatternDecimal = /^(\d{1,3})$/;
        if ( ! reRightRangePatternDecimal.test( rightRange ) )
        {
            return false;
        }
        if( ( parseInt( leftRange , 10 ) >= parseInt( rightRange ) ) || ( parseInt( rightRange , 10 ) > 255 )  )
        {
            return false;
        }
        return true;
    }

    //For those format 2001:1:2::3-aa and 2001::-aa 
    var splittedByColonParts = standardIPv6Part.split( ":" );
    var reRightRangePatternOx = /^[A-Fa-f0-9]{1,4}$/;
    if ( ! reRightRangePatternOx.test( rightRange ) )
    {
        return false;
    }
    leftRange = splittedByColonParts[ splittedByColonParts.length - 1 ];
    
    //For this format 2001::-aa
    if( ! leftRange )
    {
        leftRange = "0";
    }
    if( parseInt( leftRange , 16 ) >= parseInt( rightRange , 16 ) )
    {
        return false;
    }

    return true;
}



function checkIPv6AddressForMML( ipaddr )
{
    if (  isValidateIPv6Address ( ipaddr )  ) 
    {
        return true;
    }
    if ( isValidIpv6WithMask ( ipaddr ) )
    {
        return true;
    }
    
    if ( isValidIPv6AddressSelfDefinedRange( ipaddr ) )
    {
        return true;
    }

    return false;
}


function isValidIPAddressRange(ipaddr) {
   ipaddr=ipaddr.trim();
   var re = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})-(\d{1,3})$/;
   if(re.test(ipaddr)) {
     var separator= /[.-]/;
     var parts = ipaddr.split(separator);
     if(parseInt(parseFloat(parts[0])) == 0) {return false; }
     for(var i=0; i<parts.length; i++) {
       if (parseInt(parseFloat(parts[i])) > 255) {return false; }
     }
     if(parseInt(parseFloat(parts[3])) >= parseInt(parseFloat(parts[4]))) {return false;}
     return true;     
   }
   return false;

}




function isValidIPAddressMask(ipaddr) {
   ipaddr=ipaddr.trim();
   var re = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})\/(\d{1,2})$/;
   if(re.test(ipaddr)) {
     var separator= /[.\/]/;
     var parts = ipaddr.split(separator);
     if(parseInt(parseFloat(parts[0])) == 0) {return false; }
     for(var i=0; i<parts.length-1; i++) {
       if (parseInt(parseFloat(parts[i])) > 255) {return false; }
     }
     if((parseInt(parseFloat(parts[4])) > 32) || (parseInt(parseFloat(parts[4])) <=0)) {return false;}
     return true;     
   }
   return false;
}




function checkSenderIP(objName) {
    var datafield=objName.value;
    if(typeof String.prototype.trim !== 'function') {
        String.prototype.trim = function() {
            return this.replace(/^\s+|\s+$/g, ''); 
        }
    }
    datafield = datafield.trim();
    if(datafield.length==0) {
        return false;
    }
    if( isValidIPAddress(datafield) ) {
        return true;
    } else if( isValidIPAddressRange(datafield) ) {
        return true;
    } else if( isValidIPAddressMask(datafield) ) {
        return true;
    } else if( checkIPv6AddressForMML(datafield)) {
        return true;
    } else {
        objName.select();
        alert("<spring:message code="js.invalidIP" />");
        objName.focus();  
        return false;
    }
}

// please put onblur="checkempty(this)" in the item you want to check it is empty or not. It can avoid the item to be empty.
function checkempty(objName) {
	return checkemptywithMsg(objName, '<spring:message code="js.emptyItem"/>');
}

function checkemptywithMsg(objName, errMsg) {
var datefield = objName;
if ( objName.value.Trim().length==0) {
datefield.select();
alert(errMsg);
datefield.focus();
return false;
}
else {
return true;
   }
}

function checkdomain(objName) {
    var datefield = objName;
    if (objName.value.length!=0)
    {
        if ( !FSfncValidateDomain(objName, true, false) || !checkDomain_Format(objName.value) )
        {
            datefield.select();
            alert("<spring:message code="js.badDomain"/>");
            datefield.focus();
            return false;
        }
    }
    return true;
}

function checkdomain_subdomain(objName) {
    var datefield = objName;
    if (objName.value.length!=0)
    {
        if ( !FSfncValidateDomain_Subdomain(objName.value, true, false) || !checkDomain_Subdomain_Format(objName.value) )
        {
            datefield.select();
            alert("<spring:message code="js.badDomain"/>");
            datefield.focus();
            return false;
        }
    }
    return true;
}

function checkIpv4OrIpv6OrDomain(objName)
{
    var datefield = objName;
    if(objName.value.length!=0 && checkDomain_Format(objName.value)==false && isValidateIPv6Address(objName.value)==false && isValidIPAddress(objName.value) == false)
    {
        datefield.select();
        alert("<spring:message code="js.badIPDomain"/>");
        datefield.focus();
        return false;
    }
    else
    {
        return true;
    }
}

function isValidateIPv6Address(ipaddr)
{
	return (new RegExp(re_ipv6_pattern)).test(ipaddr);
}
function cLength(str)
{
    var pattern=/([0-9a-f]{1,4}:)|(:[0-9a-f]{1,4})/gi;
    var temp=str.replace(pattern," ");
    return temp.length;
}
function checkIpv6WithMask(objName)
{
    var datefield = objName;
    if(objName.value.length!=0 && isValidIpv6WithMask(objName.value)==false)
    {
        datefield.select();
        alert("<spring:message code="js.badIPv6WithMask"/>");
        datefield.focus();
        return false;
    }
    else
    {
        return true;
    }
}
function isValidIpv6WithMask(ipWithMask)
{
    var location = ipWithMask.indexOf("/");
    if(location!=-1)
    {
        var ip = ipWithMask.substring(0,location); 
        var mask = ipWithMask.substring(location+1);
        if(isValidateIPv6Address(ip) == true && isValidIpv6Mask(mask) == true)
        {
            return true;
        }
    }
    return false;
}

function isValidIpv6Mask(mask)
{   
    if (isNumber(mask) == true)
    {
        if (mask >= 0 && mask <= 128)
        {
            return true;
        }
    }
    
    return false;
}

function isNumber(num)
{
    var pattern = /^\d+$/;
    if(pattern.test(num))
    {
        return true;
    }
    return false;
}

function checkipordomain(objName) {
var datefield = objName;
if (objName.value.length!=0 && checkDomain_Format(objName.value)==false && isValidIPAddress(objName.value) == false) {
datefield.select();
alert("<spring:message code="js.badIPDomain"/>");
datefield.focus();
return false;
}
else {
return true;
   }
}

function checkemail(objName) {
var datefield = objName;
if (objName.value.length==0 || FSfncValidateEmailAddress(objName.value) == false) {
datefield.select();
alert("<spring:message code="js.bademail"/>");
datefield.focus();
return false;
}
else {
return true;
   }
}

//added by jacky_zhu at 2010/3/3 to support the email format sync with IMHT
function checkPrefilterEmailAddress(objName, noAsterisk)
{
    var result = true;
    var datefield = objName;
	
    if (objName.value.length==0 ) 
    {        
        result = false;
    }
    else
    {
        var str = objName.value;        
        var re_pattern = new RegExp("^" + re_email_address_pattern + "$");
        var matched_substrings = re_pattern.exec(str);
        if (!matched_substrings)
        {
            result = false;
        }
        else
        {
            var userValid = false;
            //SEGTT-282957, according to RFC special charaters are permitted(.!#$%&'*+-/=?^_`{|}~)
            //var localpart =  /^[a-zA-Z0-9_\-]+(\.[_a-zA-Z0-9\-]+)*$/;
            //var localpart = /^[0-9a-zA-Z]+([\.\!\#\$\%\&\'\*\+\-\/\=\?\^\_\`\{\|\}\~]{0,1}[a-zA-Z0-9]+)*$/;
            var localpart = new RegExp("^" + re_valid_email_address_chars_user + "+$");
            if (noAsterisk)
            {
                userValid = localpart.test(matched_substrings[1]);
            }
            else
            {
                userValid = (matched_substrings[1] == "*" || localpart.test(matched_substrings[1]));
            }
            var domainValid = checkDomain_Format(matched_substrings[2]);
            result = userValid && domainValid;
        }
    }
    
    if (result == false)
    {
        datefield.select();
        alert("<spring:message code="js.bademail"/>");
        datefield.focus();
    }
    
    return result;
}

// validate approve sender email format.
// *@* is not valid
function checkwildcardemail(objName) 
{
    if (objName.value.length == 0 || !checkWildcardEmailAddress_Format(objName.value))
	{
	    objName.select();
           alert("<spring:message code="js.bademail"/>");
           objName.focus();
           return false;
	}
	
return true;
}

function checkwildcarddomain(objName) 
{
    if (objName.value.length == 0 || !checkWildcardDomain_Format(objName.value))
	{
	    objName.select();
           alert("<spring:message code="js.badDomain"/>");
           objName.focus();
           return false;
	}
	
return true;
}


var knownDomsPat=/^(com|net|org|edu|int|mil|gov|arpa|biz|aero|name|coop|info|pro|museum|co|uk)$/;


function FSfncValidateDomain(FormField,NoWWW,CheckTLD) {
	// NoWWW and CheckTLD are optional, both accept values of true, false, and null.
	// NoWWW is used to check that a domain name does not begin with 'www.', eg. for WHOIS lokkups.
    var validChars = '.0123456789';		
	var dots = 0;
     
    if (FormField.value.length > 255) 
    	return false;

    for (var i = 0; i < FormField.value.length; i++) {
       var chr = FormField.value.substring(i,i+1);
       if (chr == '.') {
           dots++;
       }
    }
    
    // SEG Case  125905: IMSA 7.0: SMTP domain delivery check  
    if (dots<=0)
    return false;

	DomainName=FormField.value.toLowerCase();
	
	if (CheckTLD==null) {CheckTLD=true}
	var specialChars="/\\(\\)><@,;:\\\\\\\"\\.\\[\\]";
	var validChars="\[^\\s" + specialChars + "\]";
	var atom=validChars + '+';
	var atomPat=new RegExp("^" + atom + "$");
        if(DomainName=="localhost")  { return true;}
	var domArr=DomainName.split(".");
	var len=domArr.length;
	if (len==1) { return false;}
	for (i=0;i<len;i++) {if (domArr[i].search(atomPat)==-1) {return false;}}
	if ((CheckTLD) && (domArr[domArr.length-1].length!=2) && (domArr[domArr.length-1].search(knownDomsPat)==-1)) {return false;}
	if ((NoWWW) && (DomainName.substring(0,4).toLowerCase()=="www.")) {return false;}
	return true;
}

//support sub-domain, not support top-level sub-domain, such as(*.com)
function FSfncValidateDomain_Subdomain(value,NoWWW,CheckTLD) {
    // NoWWW and CheckTLD are optional, both accept values of true, false, and null.
    // NoWWW is used to check that a domain name does not begin with 'www.', eg. for WHOIS lokkups.
    var validChars = '.0123456789';
    var dots = 0;
     
    if (value.length > 255) 
        return false;

    //handle sub-domain
    if(value.charAt(0) == '*' && value.charAt(1) == '.')
    {
        value = value.substring(2); //start from the 2nd
    }
    for (var i = 0; i < value.length; i++) {
       var chr = value.substring(i,i+1);
       if (chr == '.') {
           dots++;
       }
    }
    
    // SEG Case  125905: IMSA 7.0: SMTP domain delivery check  
    if (dots <= 0)
        return false;

    DomainName = value.toLowerCase();

    if (CheckTLD==null) {CheckTLD=true}
    var specialChars="/\\(\\)><@,;:\\\\\\\"\\.\\[\\]";
    var validChars="\[^\\s" + specialChars + "\]";
    var atom=validChars + '+';
    var atomPat=new RegExp("^" + atom + "$");
    if(DomainName=="localhost")  { return true;}
    var domArr=DomainName.split(".");
    var len=domArr.length;
    if (len==1) { return false;}
    for (i=0;i<len;i++) {if (domArr[i].search(atomPat)==-1) {return false;}}
    if ((CheckTLD) && (domArr[domArr.length-1].length!=2) && (domArr[domArr.length-1].search(knownDomsPat)==-1)) {return false;}
    if ((NoWWW) && (DomainName.substring(0,4).toLowerCase()=="www.")) {return false;}
    return true;
}

function FSfncValidateEmailAddress(FormField,CheckTLD) {
	// CheckTLD is optional, it accepts values of true, false, and null.
	emailStr = FormField.toLowerCase()
	if (CheckTLD==null) {CheckTLD=true}
	var emailPat=/^(.+)@(.+)$/;
	var specialChars="\\(\\)><@,;:\\\\\\\"\\.\\[\\]";
	var validChars="\[^\\s" + specialChars + "\]";
	var quotedUser="(\"[^\"]*\")";
	var ipDomainPat=/^\[(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})\]$/;
	var atom=validChars + '+';
	var word="(" + atom + "|" + quotedUser + ")";
	var userPat=new RegExp("^" + word + "(\\." + word + ")*$");
	var domainPat=new RegExp("^" + atom + "(\\." + atom +")*$");
	var matchArray=emailStr.match(emailPat);
	if (matchArray==null) {return false;}
	var user=matchArray[1];
	var domain=matchArray[2];
	for (i=0; i<user.length; i++) {if (user.charCodeAt(i)>127) {return false;}}
	for (i=0; i<domain.length; i++) {if (domain.charCodeAt(i)>127) {return false;}}
	if (user.match(userPat)==null) {return false;}
	var IPArray=domain.match(ipDomainPat);
	if (IPArray!=null) {for (var i=1;i<=4;i++) {if (IPArray[i]>255) {return false;}}; return true}
	var atomPat=new RegExp("^" + atom + "$");
        if(domain=="localhost") { return true;}
	var domArr=domain.split(".");
	var len=domArr.length;
	for (i=0;i<len;i++) {if (domArr[i].search(atomPat)==-1) {return false;}}
	if ((CheckTLD) && (domArr[domArr.length-1].length!=2) && (domArr[domArr.length-1].search(knownDomsPat)==-1)) {return false;}
	if (len<2) {return false;}
	return true;
	}

function checkFileName(s)
{
    var i, baseName;
    var re1, re2,re3, result1, result2,result3;
    
    baseName = s.value;
     
    if ( baseName == null || baseName.length == 0 ){
//alert("base is null");
            return false;

    }
//alert("basename: " +baseName);
//alert("File : " + s);
    //make sure base name has no \ / : * ? " < > |
    re1 = new RegExp("[^a-zA-Z0-9_\+\$\.\!\@\#\~\,\%\&\^-]");
    re2 = new RegExp("[^\b]");
	re3 = new RegExp("[*/*]");
    result1 = re1.exec(baseName);
    result2 = re2.exec(baseName);
    result3 = re3.exec(baseName);
	
    if ( result1 != null && result2 != null ){
	           if ( result3 != null ){
			            s.select();
                        alert("<spring:message code="js.badChar1"/> "+ result3[0] +" <spring:message code="js.badChar2"/>");
						s.focus();
                        return false;
         }
		 
            if ( result1 != null ){
			            s.select();
						if(result1[0]!=" ")
                        alert("<spring:message code="js.badChar1"/> "+ result1[0]+ " <spring:message code="js.badChar2"/>" );
						else
						alert("<spring:message code="js.nospace"/>" );
						s.focus();
                        return false;
        }
            if ( result2 != null ){
			            s.select();
                        alert("<spring:message code="js.badChar1"/> "+ result2[0] +" <spring:message code="js.badChar2"/>"  );
						s.focus();
                        return false;
        }
    }
    return true;
}

function checkAttachName(s)
{
       for (var i = 0; i < s.value.length; i++) {
       var chr = s.value.substring(i,i+1);
       if ( chr == '\\' || chr == '/'  || chr == '<' || chr == '>' || chr == '|' || chr== '"' )
	   {
        s.select();
        alert("<spring:message code="js.badChar1"/> "+ chr +" <spring:message code="js.badChar2"/> " );
	    s.focus();
        return false; 
       }
    }
	return true;
}

function LockForm(form)
{
    for (var i=0; i<form.elements.length; i++) 
    {
        form.elements[i].disabled = true;
    }
}

function LockFormButton(form)
{
    for (var i=0; i<form.elements.length; i++) 
    {
        if(form.elements[i].type=="button"
            ||form.elements[i].type=="submit")
            form.elements[i].disabled = true;
    }
}

function checkHostName(objName)
{
	var datafield = objName;
	if (objName.value.length==0) 
	{
		datafield.select();
		return false;
	}
	
	var strVal = objName.value;
	var hostNamePatt = /[a-zA-Z0-9_-]+/;

	if(strVal.match(hostNamePatt) )
		return true;
		
	if(checkip(objName))
		return true;

	return checkdomain(objName);
	
}

function checkHostNameStr(hostname)
{
    var hostNamePatt = /^[a-zA-Z0-9_-]+$/;
    if(hostname.match(hostNamePatt))
        return true;
    
    if(checkDomain_Format(hostname) == false)
    {
        return false;
    }
    
    return true;
}

function checkEmail70(objName, isTitle, isMultiAddr)
{
	var datafield = objName;
	if (objName.value.length==0) 
	{
		datafield.select();
		return false;
	}
	
	var strVal = objName.value;
	var emailTitlePatt = /\s*"[^""]*"\s*<[a-zA-Z0-9_-]+@[a-zA-Z0-9_-]+(\.[a-zA-Z0-9_-]+)*>\s*/;
	var emailPatt = /\s*[a-zA-Z0-9_-]+@[a-zA-Z0-9_-]+(\.[a-zA-Z0-9_-]+)*\s*/;
		
	if(isTitle)
	{
		if(isMultiAddr)
		{
		    var emailArr=strVal.split(";");
		    var len=emailArr.length;

		    for (i=0;i<len;i++)
		    {
		        if(!emailArr[i].match(emailPatt) && !emailArr[i].match(emailTitlePatt))
		            return false;
		    }
		    return true;		
		}
		else
		{
			if(strVal.match(emailPatt) || strVal.match(emailTitlePatt))
				return true;
			return false;
		}	
	}
	else
	{
		if(isMultiAddr)
		{
	          var emailArr=strVal.split(";");
		    var len=emailArr.length;

		    for (i=0;i<len;i++)
		    {
		        if(!emailArr[i].match(emailPatt))
		            return false;
		    }
		    return true;		

		}
		else
		{
			if(strVal.match(emailPatt))
				return true;
			return false;
		}	
	}

	return false;
}

function updateCheckboxGroup(clickeditem, top, items)
{
    if(clickeditem==top)
    {
        if(items==null)
            return true;
        items.checked=top.checked;
        for(i=0; i<items.length; i++)
        {
            items[i].checked=top.checked;
        }
    }
    else
    {
        top.checked=items.checked;
        if(items.length > 1)
        {
            var checkAll=true;
            for(i=0; i<items.length; i++)
            {
                if(!items[i].checked)
                {
                    checkAll=false;
                }
            }
            top.checked=checkAll;
        }
    }
}

String.prototype.Trim = function() 
{ 
    return this.replace(/(^\s*)|(\s*$)/g, ""); 
} 


//Added for Domain Sync, all subdomain should start with "*." instead of original "." in Postfix configuration on Web UI. Just for Web UI. Back end will take use "." as prefix for subdomain still.
function checkpostfix_wildcard_deliverydomain(objName) 
{
    if (objName.value.length == 0 || !checkWildcardDomain_Format(objName.value))
	{
	    objName.select();
           alert("<spring:message code="js.badDomain"/>");
           objName.focus();
           return false;
	}
	
    return true;
}

function checkpostfix_wildcard_relaydomain(objName) 
{
    if (objName.value == "*")
	{
	    objName.select();
           alert("<spring:message code="js.badDomain"/>");
           objName.focus();
           return false;
	}
	if(checkpostfix_wildcard_deliverydomain(objName))
        return true;
    else
        return false;
}

function checkLinuxPath(objName)
{
    var re_path = new RegExp(re_linux_path_pattern);
    if (!re_path.test(objName.value)) {
        objName.select();
        alert("<spring:message code="js.invalidPath"/>");
        objName.focus();
        return false;
    }
    else {
        return true;
    }
}

function check_xheader_Format(str)
{
    //xheader  name:value
   var valid_string = "!\"#$%&'()*+,-./0123456789;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ\\[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~";
					   
   var pattern = "^["+valid_string+"\\\\]+\\s*:\\s*"+"["+valid_string+"\\s]+$";
   
    return (new RegExp(pattern )).test(str);
}
function checkNameObjNameStartWith(str)
{
	var pattern = "^\\s*_IMSS_";
	return (new RegExp(pattern)).test(str);



}


//add by imsa
function DateCheck(from,to,fromHH,fromMM,toHH,toMM)
{
    if (Date.parse(from) < Date.parse(to)) 
    {
        return true;
    }
    else {
        if(Date.parse(from) == Date.parse(to) )
        {
            if(toHH.length == 1)
                toHH = "0" + toHH;
            if(toMM.length == 1)
                toMM = "0" + toMM;
            if(fromHH.length == 1)
                fromHH = "0" + fromHH;
            if(fromMM.length == 1)
                fromMM = "0" + fromMM;
            if( ( toHH + toMM ) >= ( fromHH + fromMM ) )
            return true;
        }
        if (from == "" || to == "")
        { 
            alert("<spring:message code="js.bothdates"/>");
            return false;
        }
        else 
        {
            alert("<spring:message code="js.badtimeorder"/>");
            return false;
        }
   }
}
