#!/bin/sh

# Script for IMSA patch rollbacking via Web Console.
# ini_util.sh is based on StarGate implementation, with minor bug fixes.
# Zhennan, Nov/07/2006
#
# This script is invoked by imssmgr, which is running as root
# when invoked, stdout and stderr will be redirected to log file: patchlog.txt
# for easy testing, no stdout and stderr redirection inside the script.
# this script can be executed from console directly, with correct folder structure
# 
# Naming convention
# F_  means a subroutine


export PATCH_FOLDER=${IMSS_HOME}/patch
PATCHMGMINI=${PATCH_FOLDER}/patch.ini
LOCK_FILE=${PATCH_FOLDER}/.isinprogress
ADMIN_CONF=${IMSS_HOME}/UI/php/conf/widget.conf
EUQ_CONF=${IMSS_HOME}/UI/euqUI/conf/EUQ.conf

INI_UTIL_SH=${PATCH_FOLDER}/ini_util.sh
PATCH_COMMON_SH=${PATCH_FOLDER}/patch_common.sh

PATCH_NUMBER=""
ERROR_NUMBER=0
NO_BACKUP_FILE=0

ROLLBACK_SH=""
BACKUP_TAR_GZ=""
BACKUP_FILELIST=""
APPLY_DB_SH=""
BACKUP_FOLDER=""

. ${INI_UTIL_SH}
. ${PATCH_COMMON_SH}

IS_IMSVA=0
if [ -f /etc/trend-release ]; then
    IS_IMSVA=1
fi

# for imsva, product is installed in /opt/trend/imss.
# for imss, product's installed folder is set by customer.
TARGET_FOLDER=""
if [ ${IS_IMSVA} -eq 0 ]; then
    TARGET_FOLDER=${IMSS_HOME}
fi


F_set_global_variable() {

	BACKUP_FOLDER=${PATCH_FOLDER}/patches/${PATCH_NUMBER}
	PATCH_COMMON_SH=${BACKUP_FOLDER}/patch_common.sh
	BACKUP_TAR_GZ=${BACKUP_FOLDER}/backup.tar.gz
	BACKUP_FILELIST=${BACKUP_FOLDER}/backup.filelist
	APPLY_DB_SH=${BACKUP_FOLDER}/apply_db.sh
}


F_check_all() {

    if [ ! -f "$PATCHMGMINI" ] ; then
        F_log "Can not find $PATCHMGMINI."
        ERROR_NUMBER=${ERR_INI_FILE_ACCESS}
        return 1
    fi

    # key, section, are required
    PATCH_NUMBER=`F_ini_get $PATCHMGMINI Patches Rollbackable`
    if [ -z "$PATCH_NUMBER" ] ; then
        F_log "Cannot find rollbackable patch Number in $PATCHMGMINI."
        ERROR_NUMBER=${ERR_INI_FILE_CONTENT}    
        return 1
    fi

	# the patch number has been got, then just set variables
	F_set_global_variable
    
    if [ ! -f "$BACKUP_FILELIST" ] ; then
        F_log "Can not find $BACKUP_FILELIST."
        ERROR_NUMBER=${ERR_FILE_MISSING}
        return 1
    fi

    
    #stop if the patch number can not be found in Applied
    _applied_patches=`F_ini_get $PATCHMGMINI Patches Applied`
    _found_in_applied=""
    for i in $_applied_patches 
    do
        if [ "$PATCH_NUMBER" = "$i" ] ; then
            _found_in_applied="found"
        fi
    done
    if [ -z ${_found_in_applied} ] ; then
        F_log "Cannot find rollbackable patch Number in Applied patches."
        ERROR_NUMBER=${ERR_INI_FILE_CONTENT}    
        return 1
    fi

    F_ini_exist_section ${PATCHMGMINI} "$PATCH_NUMBER"
    if [ $? -ne 0 ] ; then
        F_log "Cannot find rollback info for patch $PATCH_NUMBER."
        ERROR_NUMBER=${ERR_INI_FILE_CONTENT}    
        return 1
    fi

    NO_BACKUP_FILE=`F_ini_get $PATCHMGMINI "$PATCH_NUMBER" "NoBackupFile"`
    if [ $NO_BACKUP_FILE -eq 0 ] ; then
        if [ ! -f "$BACKUP_TAR_GZ" ] ; then
            F_log "Can not find $BACKUP_TAR_GZ."
            ERROR_NUMBER=${ERR_FILE_MISSING}
            return 1
        fi
    fi

    # all keys in the $PATCH_NUMBER section are not required.
    
    F_log "Required file and fields check OK."
    return 0
}

F_post_rollback() {
   # update patch.ini
    
    F_ini_delete_section ${PATCHMGMINI} ${PATCH_NUMBER}

    _applied_patches=`F_ini_get $PATCHMGMINI "Patches" "Applied"`
    _new_applied_patches=" "
    
    for i in $_applied_patches 
    do
        if [ "$PATCH_NUMBER" != "$i" ] ; then
            _new_applied_patches="${_new_applied_patches} $i"
        fi
    done
    
    F_ini_set $PATCHMGMINI "Patches" "Applied" "${_new_applied_patches}"
	
	next_rollback_able=`echo $_new_applied_patches | awk '{print $NF}'`
	if [ -z "$next_rollback_able" ]; then
	    F_ini_delete_key ${PATCHMGMINI} "Patches" "Rollbackable"
	else
		F_ini_set ${PATCHMGMINI} "Patches" "Rollbackable" "$next_rollback_able"
	fi
    
    # remove backup files
	rm -rf ${BACKUP_FOLDER}
	return 0
}

F_rollback() {

    IS_MASTER=0
    sh ${IMSS_HOME}/script/is_master.sh
    if [ $? = 1 ] ; then
        IS_MASTER=1
    fi

    _value=`F_ini_get ${PATCHMGMINI} ${PATCH_NUMBER} "UIInterruption"`
    if [ ! -z "$_value" ] ; then
    	if [ $_value = "1" ] ; then
    		if [ $IS_MASTER = 1 ] ; then
    		    ${IMSS_HOME}/PostgreSQL/bin/psql -U sa -d imss -c "update tb_patch_status set status=4 where status=2;" > /dev/null 2>&1
    		    sleep 10
    		    ${IMSS_HOME}/imss/PostgreSQL/bin/psql -U sa -d imss -c "update tb_patch_status set status=2 where status=4;" > /dev/null 2>&1
    		fi
    	fi
    fi
    
    #handle db rollback
    . ${APPLY_DB_SH} "rollback"  ${PATCH_NUMBER}
    
    if [ $? -ne 0 ];then
        F_log "Cannot rollback database, stop rollback patch."
        return 1
    fi
    
    _value=`F_ini_get ${PATCHMGMINI} ${PATCH_NUMBER} "CmdBeforeUpdate"`
    
    if [ ! -z "$_value" ] ; then
        F_log "Calling ${_value}"
        sh -c "${_value}"
    fi

    # no need to check return value of "rm -rf"
    # if it's failed in the middle, better keep rollbacking...
    pushd .
    cd ${TARGET_FOLDER}/
    rm -rf `cat ${BACKUP_FILELIST}`
    popd
    
    if [ $NO_BACKUP_FILE -eq 0 ] ; then
        tar -C ${TARGET_FOLDER}/ -zxf ${BACKUP_TAR_GZ}
        if [ $? -ne 0 ] ; then
            F_log "Cannot untar files."
            ERROR_NUMBER=${ERR_UNTAR_BACKUP_FILES}
            
            # although this probably won't work at this time...
            _value=`F_ini_get ${PATCHMGMINI} ${PATCH_NUMBER} "CmdAfterUpdate"`
            if [ ! -z "$_value" ] ; then
                F_log "Calling ${_value}"
                sh -c "${_value}"
            fi
            return 1
        fi  
    fi
    
    # must write result before call CmdAfterUpdate
    # otherwise, imssmgr may not able to find the result during startup
    # for rollback, result_build and patch_title are useless
    F_write_result "1" "0" "2" "0" "0" ${ERR_MSG_SUCCESS} "result_build" "patch_title"

    # Clear cache folder adminUI/ROOT/work and euqUI/ROOT/work
    rm -rf ${IMSS_HOME}/UI/adminUI/ROOT/work/
    rm -rf ${IMSS_HOME}/UI/euqUI/ROOT/work/

    #hf1368
    if [ $PATCH_NUMBER -le 1368 ] ; then
        F_log "rollback ${IMSS_HOME}/UI/adminUI/conf/server.xml file"
        sed -i "s#protocol=\"AJP/1.3\" secretRequired=\"false\"#protocol=\"AJP/1.3\"#g" ${IMSS_HOME}/UI/adminUI/conf/server.xml
        F_log "rollback ${IMSS_HOME}/UI/euqUI/conf/server.xml file"
        sed -i "s#protocol=\"AJP/1.3\" address=\"0.0.0.0\" secretRequired=\"true\" secret=\"oRcImDnErT\"#protocol=\"AJP/1.3\"#g" ${IMSS_HOME}/UI/euqUI/conf/server.xml
    fi

    hotfix_number=`grep Applied $IMSS_HOME/patch/patch.ini | head -n 1 | awk -F '=' '{print $2}' | awk '{print $(NF-1)}'`   
 
    if [ ! -n "$hotfix_number" ] ;then
        hotfix_number=1323
    fi
	
    if [ $hotfix_number -lt 1368 ] ;then
        F_log "rollback mod_mpm_prefork"
        sed -i '/mod_mpm_prefork.so/d' ${ADMIN_CONF}
        sed -i '/mod_mpm_prefork.so/d' ${EUQ_CONF}
    fi
    
    _value=`F_ini_get ${PATCHMGMINI} ${PATCH_NUMBER} "CmdAfterUpdate"`
	# call F_post_rollback before the imssmgr restart beacause 
	##  1. the complier optimizing lead to imssmgr canot load the patch.ini file timely and the imssmgr would read outdated infomation.
	##  2. F_post_rollback will remove related section then F_ini_get can not get peoperbly infomation.
	F_post_rollback

    if [ ! -z "$_value" ] ; then
        F_log "Calling ${_value}"
        sh -c "${_value}"
    fi
    
 
    return 0
}


F_main() {

    F_log "Start rollback patch."
    
    if [ -f "$LOCK_FILE" ] ; then
        F_log "Another installing/uninstalling is in progress."
        ERROR_NUMBER=${ERR_ISINPROGRESS}
        return 1
    fi

    touch ${LOCK_FILE}
    
    if [ $? -ne 0 ] ; then
        F_log "Can not create lock file."
        ERROR_NUMBER=${ERR_FAILED_TO_LOCK}
        return 1
    fi
    
    # use size to indicate type, zero is update, one is rollback
    # this is used in IMSA 1, although IMSA 7 doesn't need this, just keep it.
    echo "r" > ${LOCK_FILE}
    
    F_check_all
    
    if [ $? -ne 0 ] ; then
        rm -f ${LOCK_FILE}
        return 1
    fi
        
    F_rollback

    if [ $? -ne 0 ] ; then
        rm -f ${LOCK_FILE}
        return 1
    fi
	

    
    # Success
    F_log "Done. Success"
    rm -f ${LOCK_FILE}

    return 0
}

F_main
if [ $? -ne 0 ] ; then

    # for rollback, result_build and patch_title are useless
    F_write_result "1"  "0"  "2" "0" "1" ${ERR_MSG_FAIL} "result_build" "patch_title"
    exit ${ERROR_NUMBER}
else
    exit 0
fi
