#!/usr/bin/env python
# -*- coding: utf-8 -*-

import ConfigParser
import os
import utility
from model import Const

# Names of sections and keys in the config file.
# ++++++++++++++++++++++ The general section +++++++++++++++++++++
SECTION_GENERAL = "general"
# The version of current release, e.g., 9.0.
KEY_VERSION = "version"
# The build number of current release, e.g., 1234.
KEY_BUILD = "build"
# Display language of the installer, en or ja.
KEY_LANG = "lang"
# The install mode, 0 is interactive mode, 1 is silent (automatic). 
KEY_INSTALL_MODE = "install_mode"
# The path of the installation log.
KEY_LOG_PATH = "log_path"
# The prefix of the resource file.
KEY_RES_FILE_PREFIX = "res_file_prefix"
# The suffix of the resource file. Together with prefix and language, it 
# determines the file name of the resource file. E.g., prefix is res_, suffix is
# .ini, language is en, then the file name of the resource file is res_en.ini.
KEY_RES_FILE_SUFFIX = "res_file_suffix"
# 0 means imss, 1 means imsva.
KEY_PRODUCT = "product"
# 0 means not to check system requirement (space, memory...), 1 means to check.
KEY_CHECK_REQ = "check_requirement"
# The free disk size required to install IMSx, if the free disk size is below this, installation
# cannot continue.
KEY_MIN_FREE_DISK_SIZE = "min_free_disk_size"
# The minimum size of total disk in bytes, show show a warning if below this value. 
KEY_MIN_DISK_SIZE = "min_disk_size"
# The minimum swap size in bytes, should show a warning if below this value.
KEY_MIN_SWAP_SIZE = "min_swap_size"
# The minimum memory size in bytes, should show a warning if below this value. 
KEY_MIN_MEM_SIZE = "min_mem_size"
# The dependency packages required by IMSx
KEY_DEPENDENCIES = "dependencies"
# ---------------------- The general section --------------------

# ++++++++++++++++ The database settings section ++++++++++++++++
SECTION_DATABASE = "database"
# The type of the admin database, bundled or an external database installed by the 
# user himself, 0 means bundled, 1 means it is an external one.
KEY_ADMIN_DB_TYPE = "admin_db_type"
# The server address of the admin database.
KEY_ADMIN_DB_ADDR = "admin_db_address"
# The port of the admin database.
KEY_ADMIN_DB_PORT = "admin_db_port"
# The admin database name.
KEY_ADMIN_DB_NAME = "admin_db_name"
# The user name of the admin database.
KEY_ADMIN_DB_USERNAME = "admin_db_username"
# The password of the admin database.
KEY_ADMIN_DB_PASSWORD = "admin_db_password"

# The type of the EUQ database, bundled or an external database installed by the 
# user himself, 0 means bundled, 1 means it is an external one.
KEY_EUQ_DB_TYPE = "euq_db_type"
# The server address of the EUQ database.
KEY_EUQ_DB_ADDR = "euq_db_address"
# The port of the EUQ database.
KEY_EUQ_DB_PORT = "euq_db_port"
# The EUQ database name.
KEY_EUQ_DB_NAME = "euq_db_name"
# The user name of the EUQ database.
KEY_EUQ_DB_USERNAME = "euq_db_username"
# The password of the EUQ database.
KEY_EUQ_DB_PASSWORD = "euq_db_password"
# ----------------- The database settings section -----------------

# ++++++++++++++++ The deployment settings section ++++++++++++++++
SECTION_DEPLOYMENT = "deployment"
# The path you run the install.sh 
KEY_INSTALL_PKG_PATH = "install_pkg_path"
# Whether it is fresh install, 1 means fresh install, 0 means append install.
KEY_FRESH_INSTALL = "fresh_install"
# The file paths of all rpms that need to be installed. Multiple rpms are
# separated by ;.
KEY_RPM_PATHS = "rpm_paths"
# The rpm names to be installed, separated by ;, this list shall be one-to-one 
# mapped to the rpm paths, and the order must be the same.
KEY_RPM_NAMES = "rpm_names"
# The installation path.
KEY_INSTALL_PATH = "install_path"
# Ports needed by all IMSx components, ports are separated by ;.
KEY_PORTS_ALL = "ports_all"
# Ports needed by central controller, ports are separated by ;.
KEY_PORTS_CENTRAL_CONTROLLER = "ports_central_controller"
# Ports needed by scanner, ports are separated by ;.
KEY_PORTS_SCANNER = "ports_scanner"
# Ports needed by EUQ master web console only, ports are separated by ;.
KEY_PORTS_EUQ_MASTER = "ports_euq_master"
# Ports needed by EUQ web console, ports are separated by ;.
KEY_PORTS_EUQ = "ports_euq"
# Ports needed by foxhunter, ports are separated by ;.
KEY_PORTS_FOXHUNTER = "ports_foxhunter"
# The listening port of foxhunter.
KEY_FOXHUNTER_LISTEN_PORT = "foxhunter_listen_port"
# The domain used by foxhunter to report status.
KEY_FOXHUNTER_DOMAIN = "foxhunter_domain"
# Whether to enable ERS or not
KEY_ENABLE_ERS = "enable_ers"
# Whether to enable IP profiler or not
KEY_ENABLE_IP_PROFILER = "enable_ip_profiler"
# Whether central controller is enabled on this machine, 0 means no, 1 means yes.
KEY_ENABLE_CENTRAL_CONTROLLER = "enable_central_controller"
# Whether scanner is enabled on this machine, 0 means no, 1 means yes.
KEY_ENABLE_SCANNER = "enable_scanner"
# Whether EUQ web console is enabled on this machine, 0 means no, 1 means yes.
KEY_ENABLE_EUQ_CONSOLE = "enable_euq_console"
# Whether foxhunter is enabled on this machine, 0 means no, 1 means yes.
KEY_ENABLE_FOXHUNTER = "enable_foxhunter"
# The scanner id of the instance, if it's fresh install, it is always 1, if
# append install, the scanner id will be retrieved from the existing admin db.
KEY_SCANNER_ID = "scanner_id"
# The parent address.
KEY_PARENT_ADDRESS = "parent_address"
# The admin user name.
KEY_ADMIN_USER = "admin_user"
# The admin password.
KEY_ADMIN_PASSWORD = "admin_password"
# ----------- The deployment settings section -----------

# ++++++++++++++++ The user info section ++++++++++++++++
# There are multiple user_info_x sections, such as user_info_0, user_info_1 and
# so on. Each section contains one user's information.
SECTION_USER_INFO_PREFIX = "user_info_"
# The name of the user.
KEY_USER_NAME = "name"
# The group of the user.
KEY_USER_GROUP = "group"
# The home directory of the user, empty means not to create home directory.
KEY_USER_HOME = "home"
# The login shell of the user.
KEY_USER_SHELL = "shell"
# Whether the user exists on local machine when installing, if it exists, it 
# won't be removed during uninstallation.
KEY_USER_EXIST = "exist"
# ---------------- The user info section ----------------

# +++++++++++++++ The group info section ++++++++++++++++
# There are multiple group_info_x sections, such as group_info_0, group_info_1 and
# so on. Each section contains one group's information.
SECTION_GROUP_INFO_PREFIX = "group_info_"
# The name of the group.
KEY_GROUP_NAME = "name"
# Whether the group exists on local machine when installing, if it exists, it 
# won't be removed during uninstallation.
KEY_GROUP_EXIST = "exist"
# --------------- The group info section ----------------

# ++++++++++++ The uninstall settings section +++++++++++
SECTION_UNINSTALL = "uninstall"
# Whether remove config, log, queue and db data paths.
KEY_REMOVE_DATA = "remove_data"
# ------------ The uninstall settings section -----------

# ++++++++++++ The upgrade settings section +++++++++++
SECTION_UPGRADE = "upgrade"
# The disk which is mounted to the original root ("/").
KEY_ORIG_ROOT_DISK = "orig_root_disk"
# The file system type of the original root partition.
KEY_ORIG_ROOT_FS = "orig_root_fs"
# The mount point of the previous version's root.
KEY_ORIG_ROOT_MOUNT_POINT = "orig_root_mount_point"
# The base version needed for upgrade.
KEY_BASE_VERSION = "base_version"
# The minimum build no required for upgrade.
KEY_BASE_MIN_BUILD_NO = "base_min_build_no"
# The backup folder of admin DB.
KEY_ADMIN_DB_BACKUP_FOLDER = "admin_db_backup_folder"
# The flag file to indicate that dry run is ready.
KEY_DRYRUN_ENABLED_FLAG_FILE = "dryrun_enabled_flag_file"
# The flag file to indicate that dry run has started.
KEY_DRYRUN_STARTED_FLAG_FILE = "dryrun_started_flag_file"
# The flag file to indicate that dry run has been stopped.
KEY_DRYRUN_STOPPED_FLAG_FILE = "dryrun_stopped_flag_file"
# The length of dry run period in hours.
KEY_DRYRUN_TIME = "dryrun_time"
# The path of the upgrade sql file that is executed before configuration migration.
KEY_UPGRADE_PREV_SQL_PATH = "upgrade_prev_sql_path"
# The path of the upgrade sql file that is executed after configuration migration.
KEY_UPGRADE_POST_SQL_PATH = "upgrade_post_sql_path"
# The folder where imss logs to be stored.
KEY_LOG_BACKUP_FOLDER = "log_backup_folder"
# ------------ The upgrade settings section -----------
#--------------------IMSS UPGRADE SECTION--------------
KEY_IMSS_UPD_BASE_SP="base_sp"
KEY_IMSS_UPD_BASE_HF="base_hotfix"
KEY_IMSS_UPD_BASE_PATCH="base_patch"
KEY_IMSS_UPDBCK_OLD_FOLDER = "backup_folder"





class Config(object):
    """
    This class is used to access configurations. Some configurations are read from
    the file, while some are fixed configurations.
    """    
    
    def __init__(self):
        """
        Initialize internal variables. 
        """
        # Assign default values to configurations, basically, for one configuration,
        # we need to: 
        #    1. Give a default value here.
        #    2. Add it to loadConfig and saveConfig if necessary. 
        #    3. Provide setter and getter.
        #
        # Configurations read from the file.
        self.version = 0
        self.build = 0
        self.lang = ""
        self.installMode = 0
        self.logPath = ""
        self.resFilePrefix = ""
        self.resFileSuffix = ""
        self.product = 0
        self.checkReq = 0
        self.minFreeDiskSize = 0
        self.minDiskSize = 0
        self.minSwapSize = 0
        self.minMemSize = 0
        self.dependencies = []
        
        self.adminDbType = 0
        self.adminDbAddr = ""
        self.adminDbPort = 0
        self.adminDbName = ""
        self.adminDbUser = ""
        self.adminDbPasswd = ""
        self.euqDbType = 0
        self.euqDbAddr = ""
        self.euqDbPort = 0
        self.euqDbName = 0
        self.euqDbUser = ""
        self.euqDbPasswd = ""

        self.installPkgPath = ""
        self.isFreshInstall = 0
        self.rpmPaths = []
        self.rpmNames = []
        self.installPath = ""
        self.portsAll = []
        self.portsCentralController = []
        self.portsScanner = []
        self.portsEuqMaster = []
        self.portsEuq = []
        self.portsFoxhunter = []
        self.foxhunterListenPort = 0
        self.foxhunterDomain = ""
        self.enableErs = 0
        self.enableIpProfiler = 0
        self.enableCentralController = 0
        self.enableScanner = 0
        self.enableEuqConsole = 0
        self.enableFoxhunter = 0
        self.scannerId = 0
        self.parentAddr = ""
        self.adminUser = ""
        self.adminPasswd = ""
        
        self.removeData = 0
        
        self.origRootDisk = ""
        self.origRootFs = ""
        self.origRootMountPoint = ""
        self.baseVersion = 0.0
        self.baseMinBuildNo = ""
        self.adminDBBackupFolder = ""
        self.dryrunEnabledFlag = ""
        self.dryrunStartedFlag = ""
        self.dryrunStoppedFlag = ""
        self.dryrunTime = 0
        self.upgradePrevSqlPath = ""
        self.upgradePostSqlPath = ""
        self.logBackupFolder = ""
        
        self.userInfoList = []
        self.groupInfoList = []
        
        # Configurations not in config file.
        #
        # Service setting, a list of tuples, the tuple format is (src, dst, servicename),
        # where src is the source of the link, dst is symlink's path and 
        # servicename is the name of the service.
        # should replace install path after get path in installation UI, so set value in get method
        self.serviceSetting = []

        # Cron setting, a lit of tuples, the tuple format is (signature, content), where
        # signature is the string (pattern) that will be used to identify 
        # the existence of a cron job and content is the complete content
        # of the cron job.
        self.cronSetting = []
        
        # Current role of IMSx, 0: parent, 1: child.
        self.role = 0
        
        # postfix SMTPD port
        self.smtpd_port = 25
        # UUID of /boot
        self.boot_uuid = ""
        
        self.failRemovePaths = []
        self.isRHEL7OrLater = False

        # use in REHL7 for service
        # should replace install path after get path in installation UI, so set value in get method
        self.imssSystemdSettings = []

        self.imssUpdOldBckPath=""
        self.imssUpdDbBckPath=""
        self.pgdataFolder=""

    def loadConfig(self, path):
        """
        Load config from an ini file.
        
        path: the full file path of the config file.
        
        return: a tuple (retcode, exception). Retcode 0 means successful, in 
        this case, exception is None. Retcode -1 means failed, and exception
        is set. 
        """
        (ret, exception) = (0, None)
        parser = ConfigParser.SafeConfigParser()
        
        try:
            parser.readfp(open(path, "r"))
            
            self.version = parser.getfloat(SECTION_GENERAL, KEY_VERSION)
            self.build = parser.getint(SECTION_GENERAL, KEY_BUILD)
            self.lang = parser.get(SECTION_GENERAL, KEY_LANG)
            self.installMode = parser.getint(SECTION_GENERAL, KEY_INSTALL_MODE)
            self.logPath = parser.get(SECTION_GENERAL, KEY_LOG_PATH)
            self.resFilePrefix = parser.get(SECTION_GENERAL, KEY_RES_FILE_PREFIX)
            self.resFileSuffix = parser.get(SECTION_GENERAL, KEY_RES_FILE_SUFFIX)
            self.product = parser.getint(SECTION_GENERAL, KEY_PRODUCT)
            self.checkReq = parser.getint(SECTION_GENERAL, KEY_CHECK_REQ)
            self.minFreeDiskSize = parser.getint(SECTION_GENERAL, KEY_MIN_FREE_DISK_SIZE)
            self.minDiskSize = parser.getint(SECTION_GENERAL, KEY_MIN_DISK_SIZE)
            self.minSwapSize = parser.getint(SECTION_GENERAL, KEY_MIN_SWAP_SIZE)
            self.minMemSize = parser.getint(SECTION_GENERAL, KEY_MIN_MEM_SIZE)
            self.dependencies = parser.get(SECTION_GENERAL, KEY_DEPENDENCIES).split(';')
            
            self.adminDbType = parser.getint(SECTION_DATABASE, KEY_ADMIN_DB_TYPE)
            self.adminDbAddr = parser.get(SECTION_DATABASE, KEY_ADMIN_DB_ADDR)
            self.adminDbPort = parser.getint(SECTION_DATABASE, KEY_ADMIN_DB_PORT)
            self.adminDbName = parser.get(SECTION_DATABASE, KEY_ADMIN_DB_NAME)
            self.adminDbUser = parser.get(SECTION_DATABASE, KEY_ADMIN_DB_USERNAME)
            self.adminDbPasswd = parser.get(SECTION_DATABASE, KEY_ADMIN_DB_PASSWORD)
            self.euqDbType = parser.getint(SECTION_DATABASE, KEY_EUQ_DB_TYPE)
            self.euqDbAddr = parser.get(SECTION_DATABASE, KEY_EUQ_DB_ADDR)
            self.euqDbPort = parser.getint(SECTION_DATABASE, KEY_EUQ_DB_PORT)
            self.euqDbName = parser.get(SECTION_DATABASE, KEY_EUQ_DB_NAME)
            self.euqDbUser = parser.get(SECTION_DATABASE, KEY_EUQ_DB_USERNAME)
            self.euqDbPasswd = parser.get(SECTION_DATABASE, KEY_EUQ_DB_PASSWORD)

            self.installPkgPath = parser.get(SECTION_DEPLOYMENT, KEY_INSTALL_PKG_PATH)
            self.isFreshInstall = parser.getint(SECTION_DEPLOYMENT, KEY_FRESH_INSTALL)
            self.rpmPaths = [x for x in parser.get(SECTION_DEPLOYMENT, KEY_RPM_PATHS).split(";") if x]
            self.rpmNames = [x for x in parser.get(SECTION_DEPLOYMENT, KEY_RPM_NAMES).split(";") if x]
            self.installPath = parser.get(SECTION_DEPLOYMENT, KEY_INSTALL_PATH)
            self.portsAll = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_ALL).split(";") if x]
            self.portsCentralController = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_CENTRAL_CONTROLLER).split(";") if x]
            self.portsScanner = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_SCANNER).split(";") if x]
            self.portsEuqMaster = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_EUQ_MASTER).split(";") if x]
            self.portsEuq = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_EUQ).split(";") if x]
            self.portsFoxhunter = [int(x) for x in parser.get(SECTION_DEPLOYMENT, KEY_PORTS_FOXHUNTER).split(";") if x]
            self.foxhunterListenPort = parser.getint(SECTION_DEPLOYMENT, KEY_FOXHUNTER_LISTEN_PORT)
            self.foxhunterDomain = parser.get(SECTION_DEPLOYMENT, KEY_FOXHUNTER_DOMAIN)
            self.enableErs = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_ERS)
            self.enableIpProfiler = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_IP_PROFILER)
            self.enableCentralController = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_CENTRAL_CONTROLLER)
            self.enableScanner = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_SCANNER)
            self.enableEuqConsole = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_EUQ_CONSOLE)
            self.enableFoxhunter = parser.getint(SECTION_DEPLOYMENT, KEY_ENABLE_FOXHUNTER)
            self.scannerId = parser.getint(SECTION_DEPLOYMENT, KEY_SCANNER_ID)
            self.parentAddr = parser.get(SECTION_DEPLOYMENT, KEY_PARENT_ADDRESS)
            self.adminUser = parser.get(SECTION_DEPLOYMENT, KEY_ADMIN_USER)
            self.adminPasswd = parser.get(SECTION_DEPLOYMENT, KEY_ADMIN_PASSWORD)
            self.removeData = parser.getint(SECTION_UNINSTALL, KEY_REMOVE_DATA)
            self.upgradePrevSqlPath = parser.get(SECTION_UPGRADE, KEY_UPGRADE_PREV_SQL_PATH)
            self.upgradePostSqlPath = parser.get(SECTION_UPGRADE, KEY_UPGRADE_POST_SQL_PATH)

            #IMSVA upgrade config
            if self.product == 1:
                self.origRootDisk = parser.get(SECTION_UPGRADE, KEY_ORIG_ROOT_DISK)
                self.origRootFs = parser.get(SECTION_UPGRADE, KEY_ORIG_ROOT_FS)
                self.origRootMountPoint = parser.get(SECTION_UPGRADE, KEY_ORIG_ROOT_MOUNT_POINT)
                self.baseVersion = parser.getfloat(SECTION_UPGRADE, KEY_BASE_VERSION)
                self.baseMinBuildNo = parser.getint(SECTION_UPGRADE, KEY_BASE_MIN_BUILD_NO)
                self.adminDBBackupFolder = parser.get(SECTION_UPGRADE, KEY_ADMIN_DB_BACKUP_FOLDER)
                self.dryrunEnabledFlag = parser.get(SECTION_UPGRADE, KEY_DRYRUN_ENABLED_FLAG_FILE)
                self.dryrunStartedFlag = parser.get(SECTION_UPGRADE, KEY_DRYRUN_STARTED_FLAG_FILE)
                self.dryrunStoppedFlag = parser.get(SECTION_UPGRADE, KEY_DRYRUN_STOPPED_FLAG_FILE)
                self.dryrunTime = parser.getint(SECTION_UPGRADE, KEY_DRYRUN_TIME)
                self.logBackupFolder = parser.get(SECTION_UPGRADE, KEY_LOG_BACKUP_FOLDER)
            #IMSS upgrade config
            elif self.product == 0:
                self.imssUpdOldBckPath = parser.get(SECTION_UPGRADE, KEY_IMSS_UPDBCK_OLD_FOLDER)
                self.imssUpdDbBckPath = self.imssUpdOldBckPath
                self.imssUpdBaseHf = parser.getint(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_HF)
                self.imssUpdBasePatch = parser.getint(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_PATCH)
                self.imssUpdBaseSp = parser.getint(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_SP)
                #it's a stable value in imss
                self.pgdataFolder = "/var/imss/pgdata"

            # Parse user_info_x section.
            del self.userInfoList[:]
            for section in parser.sections():
                if section.startswith(SECTION_USER_INFO_PREFIX):
                    self.userInfoList.append((parser.get(section, KEY_USER_NAME),
                                              parser.get(section, KEY_USER_GROUP),
                                              parser.get(section, KEY_USER_HOME),
                                              parser.get(section, KEY_USER_SHELL),
                                              parser.getint(section, KEY_USER_EXIST)
                                              ))
            # Parse group_info_x section.
            del self.groupInfoList[:]
            for section in parser.sections():
                if section.startswith(SECTION_GROUP_INFO_PREFIX):
                    self.groupInfoList.append((parser.get(section, KEY_GROUP_NAME),
                                               parser.getint(section, KEY_GROUP_EXIST)
                                              ))
            

            self.setIsRHEL7OrLater(utility.Utility.isOSVer7orLatter())
            self.installPkgPath = parser.get(SECTION_DEPLOYMENT, KEY_INSTALL_PKG_PATH)
        except Exception, e:
            ret = -1
            exception = e

        return (ret, exception)
    
    def saveConfig(self, path):
        """
        Save config to a file.
        
        path: the full file path to be saved to.
        
        return: a tuple (retcode, exception). Retcode 0 means successful, in 
        this case, exception is None. Retcode -1 means failed, and exception
        is set.
        """
        (ret, exception) = (0, None)
        parser = ConfigParser.SafeConfigParser()
        
        try:
            parser.add_section(SECTION_GENERAL)
            parser.set(SECTION_GENERAL, KEY_VERSION, str(self.version))
            parser.set(SECTION_GENERAL, KEY_BUILD, str(self.build))
            parser.set(SECTION_GENERAL, KEY_LANG, self.lang)
            parser.set(SECTION_GENERAL, KEY_INSTALL_MODE, str(self.installMode))
            parser.set(SECTION_GENERAL, KEY_LOG_PATH, self.logPath)
            parser.set(SECTION_GENERAL, KEY_RES_FILE_PREFIX, self.resFilePrefix)
            parser.set(SECTION_GENERAL, KEY_RES_FILE_SUFFIX, self.resFileSuffix)
            parser.set(SECTION_GENERAL, KEY_PRODUCT, str(self.product))
            parser.set(SECTION_GENERAL, KEY_CHECK_REQ, str(self.checkReq))
            parser.set(SECTION_GENERAL, KEY_MIN_FREE_DISK_SIZE, str(self.minFreeDiskSize))
            parser.set(SECTION_GENERAL, KEY_MIN_DISK_SIZE, str(self.minDiskSize))
            parser.set(SECTION_GENERAL, KEY_MIN_SWAP_SIZE, str(self.minSwapSize))
            parser.set(SECTION_GENERAL, KEY_MIN_MEM_SIZE, str(self.minMemSize))
            parser.set(SECTION_GENERAL, KEY_DEPENDENCIES, ';'.join(self.dependencies))
            
            parser.add_section(SECTION_DATABASE)
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_TYPE, str(self.adminDbType))
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_ADDR, self.adminDbAddr)
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_PORT, str(self.adminDbPort))
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_NAME, self.adminDbName)
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_USERNAME, self.adminDbUser)
            parser.set(SECTION_DATABASE, KEY_ADMIN_DB_PASSWORD, self.adminDbPasswd)
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_TYPE, str(self.euqDbType))
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_ADDR, self.euqDbAddr)
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_PORT, str(self.euqDbPort))
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_NAME, self.euqDbName)
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_USERNAME, self.euqDbUser)
            parser.set(SECTION_DATABASE, KEY_EUQ_DB_PASSWORD, self.euqDbPasswd)
            
            parser.add_section(SECTION_DEPLOYMENT)
            parser.set(SECTION_DEPLOYMENT, KEY_FRESH_INSTALL, str(self.isFreshInstall))
            parser.set(SECTION_DEPLOYMENT, KEY_RPM_PATHS, ";".join(self.rpmPaths))
            parser.set(SECTION_DEPLOYMENT, KEY_RPM_NAMES, ";".join(self.rpmNames))
            parser.set(SECTION_DEPLOYMENT, KEY_INSTALL_PATH, self.installPath)
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_ALL, ";".join([str(x) for x in self.portsAll]))
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_CENTRAL_CONTROLLER, ";".join([str(x) for x in self.portsCentralController]))
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_SCANNER, ";".join([str(x) for x in self.portsScanner]))
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_EUQ_MASTER, ";".join([str(x) for x in self.portsEuqMaster]))
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_EUQ, ";".join([str(x) for x in self.portsEuq]))
            parser.set(SECTION_DEPLOYMENT, KEY_PORTS_FOXHUNTER, ";".join([str(x) for x in self.portsFoxhunter]))
            parser.set(SECTION_DEPLOYMENT, KEY_FOXHUNTER_LISTEN_PORT, str(self.foxhunterListenPort))
            parser.set(SECTION_DEPLOYMENT, KEY_FOXHUNTER_DOMAIN, self.foxhunterDomain)
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_ERS, str(self.enableErs))
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_IP_PROFILER, str(self.enableIpProfiler))
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_CENTRAL_CONTROLLER, str(self.enableCentralController))
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_SCANNER, str(self.enableScanner))
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_EUQ_CONSOLE, str(self.enableEuqConsole))
            parser.set(SECTION_DEPLOYMENT, KEY_ENABLE_FOXHUNTER, str(self.enableFoxhunter))
            parser.set(SECTION_DEPLOYMENT, KEY_SCANNER_ID, str(self.scannerId))
            parser.set(SECTION_DEPLOYMENT, KEY_PARENT_ADDRESS, self.parentAddr)
            parser.set(SECTION_DEPLOYMENT, KEY_ADMIN_USER, self.adminUser)
            parser.set(SECTION_DEPLOYMENT, KEY_ADMIN_PASSWORD, self.adminPasswd)
            parser.set(SECTION_DEPLOYMENT, KEY_INSTALL_PKG_PATH, self.installPkgPath)
            
            parser.add_section(SECTION_UNINSTALL)
            parser.set(SECTION_UNINSTALL, KEY_REMOVE_DATA, str(self.removeData))
            
            parser.add_section(SECTION_UPGRADE)
            parser.set(SECTION_UPGRADE, KEY_ORIG_ROOT_DISK, self.origRootDisk)
            parser.set(SECTION_UPGRADE, KEY_ORIG_ROOT_FS, self.origRootFs)
            parser.set(SECTION_UPGRADE, KEY_ORIG_ROOT_MOUNT_POINT, self.origRootMountPoint)
            parser.set(SECTION_UPGRADE, KEY_BASE_VERSION, str(self.baseVersion))
            parser.set(SECTION_UPGRADE, KEY_BASE_MIN_BUILD_NO, str(self.baseMinBuildNo))
            parser.set(SECTION_UPGRADE, KEY_ADMIN_DB_BACKUP_FOLDER, self.adminDBBackupFolder)
            parser.set(SECTION_UPGRADE, KEY_DRYRUN_ENABLED_FLAG_FILE, self.dryrunEnabledFlag)
            parser.set(SECTION_UPGRADE, KEY_DRYRUN_STARTED_FLAG_FILE, self.dryrunStartedFlag)
            parser.set(SECTION_UPGRADE, KEY_DRYRUN_STOPPED_FLAG_FILE, self.dryrunStoppedFlag)
            parser.set(SECTION_UPGRADE, KEY_DRYRUN_TIME, str(self.dryrunTime))
            parser.set(SECTION_UPGRADE, KEY_UPGRADE_PREV_SQL_PATH, self.upgradePrevSqlPath)
            parser.set(SECTION_UPGRADE, KEY_UPGRADE_POST_SQL_PATH, self.upgradePostSqlPath)
            parser.set(SECTION_UPGRADE, KEY_LOG_BACKUP_FOLDER, self.logBackupFolder)
            if self.product == 0:
                parser.set(SECTION_UPGRADE, KEY_IMSS_UPDBCK_OLD_FOLDER,self.imssUpdOldBckPath )
                parser.set(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_HF,str(self.imssUpdBaseHf))
                parser.set(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_PATCH,str(self.imssUpdBasePatch))
                parser.set(SECTION_UPGRADE, KEY_IMSS_UPD_BASE_SP,str(self.imssUpdBaseSp))

            # Save users info.
            for index, (name, group, home, shell, exist) in enumerate(self.userInfoList):
                section = "%s%d" %(SECTION_USER_INFO_PREFIX, index)
                parser.add_section(section)
                parser.set(section, KEY_USER_NAME, name)
                parser.set(section, KEY_USER_GROUP, group)
                parser.set(section, KEY_USER_HOME, home)
                parser.set(section, KEY_USER_SHELL, shell)
                parser.set(section, KEY_USER_EXIST, str(exist))
            
            # Save groups info.
            for index, (name, exist) in enumerate(self.groupInfoList):
                section = "%s%d" %(SECTION_GROUP_INFO_PREFIX, index)
                parser.add_section(section)
                parser.set(section, KEY_GROUP_NAME, name)
                parser.set(section, KEY_GROUP_EXIST, str(exist))
            
            parser.write(open(path, "w"))
        except Exception, e:
            ret = -1
            exception = e
        
        return (ret, exception)
    
    def getVersion(self):
        """
        Get version.
        """
        return self.version
    
    def setVersion(self, val):
        """
        Set version.
        """
        self.version = val
        
    def getBuild(self):
        """
        Get build.
        """
        return self.build
    
    def setBuild(self, val):
        """
        Set build.
        """
        self.build = val
    
    def getLanguage(self):
        """
        Get the language.
        """
        return self.lang
    
    def setLanguage(self, val):
        """
        Set the language.
        """
        self.lang = val
    
    def getInstallMode(self):
        """
        Get the installation mode.
        """
        return self.installMode
    
    def setInstallMode(self, val):
        """
        Set the install mode.
        """
        self.installMode = val
    
    def getLogPath(self):
        """
        Get the log path.
        """
        return self.logPath
    
    def setLogPath(self, val):
        """
        Set the log path.
        """
        self.logPath = val
    
    def getResFilePrefix(self):
        """
        Get resource file prefix.
        """
        return self.resFilePrefix
    
    def setResFilePrefix(self, val):
        """
        Set resource file prefix.
        """
        self.resFilePrefix = val
        
    def getResFileSuffix(self):
        """
        Get resource file suffix.
        """
        return self.resFileSuffix
    
    def setResFileSuffix(self, val):
        """
        Set resource file suffix.
        """
        self.resFileSuffix = val
    
    def getProduct(self):
        """
        Get product.
        """
        return self.product
    
    def setProduct(self, val):
        """
        Set product.
        """
        self.product = val
        
    def getCheckReq(self):
        """
        Get check requirement.
        """
        return self.checkReq
    
    def setCheckReq(self, val):
        """
        Set check req.
        """
        self.checkReq = val
    
    def getMinFreeDiskSize(self):
        return self.minFreeDiskSize
    
    def setMinFreeDiskSize(self, val):
        self.minFreeDiskSize = val
    
    def getMinDiskSize(self):
        """
        Get min disk size.
        """
        return self.minDiskSize
    
    def setMinDiskSize(self, val):
        """
        Set min disk size.
        """
        self.minDiskSize = val
    
    def getMinSwapSize(self):
        """
        Get min swap size.
        """
        return self.minSwapSize
    
    def setMinSwapSize(self, val):
        """
        Set min swap size.
        """
        self.minSwapSize = val
        
    def getMinMemSize(self):
        """
        Get min mem size.
        """
        return self.minMemSize
    
    def setMinMemSize(self, val):
        """
        Set min mem size.
        """
        self.minMemSize = val
    
    def getAdminDBType(self):
        """
        Get the admin DB type.
        """
        return self.adminDbType
    
    def setAdminDBType(self, val):
        """
        Set the admin DB type.
        """
        self.adminDbType = val
    
    def getAdminDBAddress(self):
        """
        Get the admin DB address.
        """
        return self.adminDbAddr
    
    def setAdminDBAddress(self, val):
        """
        Set the admin DB address.
        """
        self.adminDbAddr = val
        
    def getAdminDBPort(self):
        """
        Get admin DB port.
        """
        return self.adminDbPort
    
    def setAdminDBPort(self, val):
        """
        Set admin DB port.
        """
        self.adminDbPort = val
    
    def getAdminDBName(self):
        """
        Get admin DB name.
        """
        return self.adminDbName
    
    def setAdminDBName(self, val):
        """
        Set admin DB name.
        """
        self.adminDbName = val
    
    def getAdminDBUsername(self):
        """
        Get admin DB username.
        """
        return self.adminDbUser
    
    def setAdminDBUsername(self, val):
        """
        Set admin DB username
        """
        self.adminDbUser = val
        
    def getAdminDBPassword(self):
        """
        Get admin DB password.
        """
        return self.adminDbPasswd
    
    def setAdminDBPassword(self, val):
        """
        Set admin DB password.
        """
        self.adminDbPasswd = val
        
    def getEuqDBType(self):
        """
        Get the EUQ DB type.
        """
        return self.euqDbType
    
    def setEuqDBType(self, val):
        """
        Set the EUQ DB type.
        """
        self.euqDbType = val
    
    def getEuqDBAddress(self):
        """
        Get the EUQ DB address.
        """
        return self.euqDbAddr
    
    def setEuqDBAddress(self, val):
        """
        Set the EUQ DB address.
        """
        self.euqDbAddr = val
        
    def getEuqDBPort(self):
        """
        Get EUQ DB port.
        """
        return self.euqDbPort
    
    def setEuqDBPort(self, val):
        """
        Set EUQ DB port.
        """
        self.euqDbPort = val
    
    def getEuqDBName(self):
        """
        Get EUQ DB name.
        """
        return self.euqDbName
    
    def setEuqDBName(self, val):
        """
        Set EUQ DB name.
        """
        self.euqDbName = val
    
    def getEuqDBUsername(self):
        """
        Get EUQ DB username.
        """
        return self.euqDbUser
    
    def setEuqDBUsername(self, val):
        """
        Set EUQ DB username
        """
        self.euqDbUser = val
        
    def getEuqDBPassword(self):
        """
        Get EUQ DB password.
        """
        return self.euqDbPasswd
    
    def setEuqDBPassword(self, val):
        """
        Set EUQ DB password.
        """
        self.euqDbPasswd = val
        
    def getFreshInstall(self):
        """
        Get fresh install setting. 
        """
        return self.isFreshInstall
    
    def setFreshInstall(self, val):
        """
        Set fresh install setting.
        """
        self.isFreshInstall = val
    
    def getRpmPaths(self):
        """
        Return a list of rpm paths.
        """
        return list(self.rpmPaths)
    
    def setRpmPaths(self, val):
        """
        Set rpm paths with a list.
        """
        self.rpmPaths = list(val)
    
    def getRpmNames(self):
        """
        Return a list of rpm names.
        """
        return list(self.rpmNames)
    
    def setRpmNames(self, val):
        """
        Set rpm names with a list.
        """
        self.rpmNames = list(val)
    
    def getInstallPkgPath(self):
        return self.installPkgPath

    def setInstallPkgPath(self, val):
        self.installPkgPath = val

    def getInstallPath(self):
        """
        Get install path.
        """
        return self.installPath
    
    def setInstallPath(self, val):
        """
        Set install path.
        """
        self.installPath = val
        
    def getPortsAll(self):
        """
        Return a list of ports.
        """
        return list(self.portsAll)
    
    def setPortsAll(self, val):
        """
        Set a list of ports.
        """
        self.portsAll = list(val)
    
    def getPortsCentralController(self):
        """
        Return a list of ports.
        """
        return list(self.portsCentralController)
    
    def setPortsCentralController(self, val):
        """
        Set a list of ports.
        """
        self.portsCentralController = list(val)
    
    def getPortsScanner(self):
        """
        Return a list of ports.
        """
        return list(self.portsScanner)
    
    def setPortsScanner(self, val):
        """
        Set a list of ports.
        """
        self.portsScanner = list(val)
        
    def getPortsEuqMaster(self):
        """
        Return a list of ports.
        """
        return list(self.portsEuqMaster)
    
    def setPortsEuqMaster(self, val):
        """
        Set a list of ports.
        """
        self.portsEuqMaster = list(val)
    
    def getPortsEuq(self):
        """
        Return a list of ports.
        """
        return list(self.portsEuq)
    
    def setPortsEuq(self, val):
        """
        Set a list of ports.
        """
        self.portsEuq = list(val)
    
    def getPortsFoxhunter(self):
        """
        Return a list of ports.
        """
        return list(self.portsFoxhunter)
    
    def setPortsFoxhunter(self, val):
        """
        Set a list of ports.
        """
        self.portsFoxhunter = list(val)
    
    def getFoxhunterListenPort(self):
        """
        Get foxhunter listen port.
        """
        return self.foxhunterListenPort
    
    def setFoxhunterListenPort(self, val):
        """
        Set foxhunter listen port.
        """
        self.foxhunterListenPort = val
    
    def getFoxhunterDomain(self):
        """
        Get foxhunter domain.
        """
        return self.foxhunterDomain
    
    def setFoxhunterDomain(self, val):
        """
        Set foxhunter domain.
        """
        self.foxhunterDomain = val

    def getEnableErs(self):
        """
        Get enable status of ERS.
        """
        return self.enableErs
    
    def setEnableErs(self, val):
        """
        Set enable status of ERS.
        """
        self.enableErs = val
        
    def getEnableIpProfiler(self):
        """
        Get enable status of IP profiler.
        """
        return self.enableIpProfiler
    
    def setEnableIpProfiler(self, val):
        """
        Set enable status of IP profiler.
        """
        self.enableIpProfiler = val
    
    def getEnableCentralController(self):
        """
        Get enable status of central controller.
        """
        return self.enableCentralController
    
    def setEnableCentralController(self, val):
        """
        Set enable status of central controller.
        """
        self.enableCentralController = val
    
    def getEnableScanner(self):
        """
        Get enable status of scanner.
        """
        return self.enableScanner
    
    def setEnableScanner(self, val):
        """
        Set enable status of scanner.
        """
        self.enableScanner = val
    
    def getEnableEuqConsole(self):
        """
        Get enable status of EUQ web console.
        """
        return self.enableEuqConsole
    
    def setEnableEuqConsole(self, val):
        """
        Set enable status of EUQ web console.
        """
        self.enableEuqConsole = val
        
    def getEnableFoxhunter(self):
        """
        Get enable status of foxhunter.
        """
        return self.enableFoxhunter
    
    def setEnableFoxhunter(self, val):
        """
        Set enable status of foxhunter.
        """
        self.enableFoxhunter = val

    def getIsRHEL7OrLater(self):
        return self.isRHEL7OrLater

    def setIsRHEL7OrLater(self, val):
        self.isRHEL7OrLater = val

    def getUsers(self):
        """
        Get a list of users. Each element in the list is a tuple whose 
        format is (name, group, home dir, shell, exist).
        """
        return list(self.userInfoList)

    def setUsers(self, val):
        """
        Set users information.
        
        val: a list of user information, each element is a tuple whose format
             is (name, group, home dir, shell, exist).
        """
        self.userInfoList = list(val)
    
    def getGroups(self):
        """
        Get a list of groups. Each element in the list is a tuple whose 
        format is (name, exist).
        """
        return list(self.groupInfoList)
    
    def setGroups(self, val):
        """
        Set users information.
        
        val: a list of user information, each element is a tuple whose format
             is (name, group, home, shell, exist).
        """
        self.groupInfoList = list(val)

    def getScannerId(self):
        """
        Get scanner ID.
        """
        return self.scannerId
        
    def setScannerId(self, val):
        """
        Set scanner ID.
        """
        self.scannerId = val
    
    def getParentAddress(self):
        return self.parentAddr
    
    def setParentAddress(self, val):
        self.parentAddr = val
        
    def getAdminUser(self):
        return self.adminUser
    
    def setAdminUser(self, val):
        self.adminUser = val

    def getAdminPassword(self):
        """
        Get the admin password.
        """
        return self.adminPasswd
    
    def setAdminPassword(self, val):
        """
        Set the admin password.
        """
        self.adminPasswd = val
        
    def getRemoveData(self):
        return self.removeData
    
    def setRemoveData(self, val):
        self.removeData = val
    
    def getOrigRootDisk(self):
        return self.origRootDisk
    
    def setOrigRootDisk(self, val):
        self.origRootDisk = val
    
    def getOrigRootFs(self):
        return self.origRootFs
    
    def setOrigRootFs(self, val):
        self.origRootFs = val
    
    def getOrigRootMountPoint(self):
        return self.origRootMountPoint
    
    def setOrigRootMountPoint(self, val):
        self.origRootMountPoint = val
    
    def getBaseVersion(self):
        return self.baseVersion
    
    def setBaseVersion(self, val):
        self.baseVersion = val
        
    def getBaseMinBuildNo(self):
        return self.baseMinBuildNo
    
    def setBaseMinBuildNo(self, val):
        self.baseMinBuildNo = val
        
    def getAdminDBBackupFolder(self):
        return self.adminDBBackupFolder
    
    def setAdminDBBackupFolder(self, val):
        self.adminDBBackupFolder = val
    
    def getDryrunEnabledFlag(self):
        return self.dryrunEnabledFlag
    
    def setDryrunEnabledFlag(self, val):
        self.dryrunEnabledFlag = val
    
    def getDryrunStartedFlag(self):
        return self.dryrunStartedFlag
    
    def setDryrunStartedFlag(self, val):
        self.dryrunStartedFlag = val
        
    def getDryrunStoppedFlag(self):
        return self.dryrunStoppedFlag
    
    def setDryrunStoppedFlag(self, val):
        self.dryrunStoppedFlag = val
    
    def getDryrunTime(self):
        return self.dryrunTime
    
    def setDryrunTime(self, val):
        self.dryrunTime = val
    
    def getUpgradePrevSqlPath(self):
        return self.upgradePrevSqlPath
    
    def getUpgradePostSqlPath(self):
        return self.upgradePostSqlPath
    
    def getLogBackupFolder(self):
        return self.logBackupFolder
    
    # NOTE: following settings won't be specified in configuration file, 
    # because they are internal use settings and must not be modified by
    # the customer.
    def getServiceSetting(self):
        """
        Get the system service settings of IMSx.
        
        return: the service setting list.
        """
        initdPath = "/etc/init.d"
        self.serviceSetting = [("%s/imss/script/imssctl.sh" %(self.installPath), "%s/rcImss" %(initdPath), "rcImss")]
            
        return self.serviceSetting
    
    def getCronSetting(self):
        """
        Get the cron job setting.
        
        return: the cron setting list.
        """
        scriptPath = "%s/imss/script" %(self.installPath)
        logPath = "%s/imss/log" %(self.installPath)
        # The cron job settings need to be written to imss_home/config/cronjob.ini. The cronjob
        # monitor in imssmgr will use it.
        if self.product == 1:
            self.cronSetting = [(r"logrotate\.crond", "0 */2 * * * /etc/logrotate.crond > /dev/null 2>&1"),
                                (r"mailrotate\.sh", "0 */2 * * * /etc/mailrotate.sh > /dev/null 2>&1"),
                                (r"S99SCHEDULED", "0 * * * * %s/S99SCHEDULED start >/dev/null 2>&1" %(scriptPath)), 
                                (r"foxpurgelog", "0 */1 * * * %s/foxpurgelog > /dev/null 2>&1" %(scriptPath)),  
                                (r"monitor_cmagent\.pl", "*/5 * * * * perl %s/monitor_cmagent.pl >> %s/Agent.log 2>&1" %(scriptPath, logPath)), 
                                (r"ibe_server\.sh", "0 2 * * * %s/ibe_server.sh restart > /dev/null 2>&1" %(scriptPath)),
                                (r"db_maintain\.sh", "0 3 * * * %s/db_maintain.sh vacuumanalyzewholedb  > /dev/null 2>&1" %(scriptPath)),
                                (r"auto_import_export\.sh", "0 4 * * * %s/auto_import_export.sh > %s/auto_import_export.log 2>&1" %(scriptPath, logPath))
                                ]
        else:
            self.cronSetting = [(r"S99SCHEDULED", "0 * * * * %s/S99SCHEDULED start >/dev/null 2>&1" %(scriptPath)), 
                                (r"foxpurgelog", "0 */1 * * * %s/foxpurgelog > /dev/null 2>&1" %(scriptPath)),  
                                (r"monitor_cmagent\.pl", "*/5 * * * * perl %s/monitor_cmagent.pl >> %s/Agent.log 2>&1" %(scriptPath, logPath)), 
                                (r"ibe_server\.sh", "0 2 * * * %s/ibe_server.sh restart > /dev/null 2>&1" %(scriptPath)),
                                (r"db_maintain\.sh", "0 3 * * * %s/db_maintain.sh vacuumanalyzewholedb  > /dev/null 2>&1" %(scriptPath)),
                                (r"auto_import_export\.sh", "0 4 * * * %s/auto_import_export.sh > %s/auto_import_export.log 2>&1" %(scriptPath, logPath))
                                ]
        
        return self.cronSetting

    def getRole(self):
        return self.role
    
    def setRole(self, val):
        self.role = val

    def getSmtpPort(self):
        return self.smtpd_port
    
    def setSmtpPort(self, val):
        self.smtpd_port = val
        
    def getBootUUID(self):
        return self.boot_uuid
    
    def setBootUUID(self, val):
        self.boot_uuid = val

    def getAdminDbExportPath(self):
        if self.getProduct() == 0:
            return os.path.join(self.getDBBackupFolder(), Const.MIE_FILE_NAME)
        elif self.getProduct() ==1:
            return "/var/app_data/imss/imsx_config.dat"

    
    def getCronBackupPath(self):
        return "/var/spool/cron/root.bakForUpgrade"
    
    def getRemovePathsBeforeInstallation(self):
        """
        Return a list that contains paths needs to be removed before installing.
        """
        # IMSS
        if self.getProduct() == 0:
            ret = [# IMSS_HOME folder
                   os.path.join(self.getInstallPath(), "imss"),
                   # DB data folder
                   "/var/imss",
                   ]
        # IMSVA currently has no such paths.
        else:
            ret = []
        return list(ret)
    
    def getOptionalRemovePaths(self):
        """
        Return the paths that can be removed or not depends on user choice.
        """
        imssHome = "%s/imss" %(self.getInstallPath())
        # Optional remove paths, these paths depends on user's choice to be removed
        # automatically or manually. User-defined log and queue paths are ignored 
        # here.
        self.optionalPaths = [# Configs
                              "%s/config" %(imssHome),
                              "%s/UI/adminUI/conf" %(imssHome),
                              "%s/UI/euqUI/conf" %(imssHome),
                              "%s/UI/php/conf" %(imssHome),
                              # Logs
                              "%s/log" %(imssHome),
                              # Queues
                              "%s/queue" %(imssHome),
                              # Database program and data.
                              "%s/PostgreSQL" %(imssHome),
                              "/var/imss",
                              ]
        if self.getProduct() == 1:
            self.optionalPaths.extend(["/var/app_data/imss/db",
                                       "/var/app_data/imss/log",
                                       "/var/app_data/imss/queue",
                                       ])
        return list(self.optionalPaths)
    
    def getAutoRemovePaths(self):
        """
        Return the paths that will be removed automatically by the installer.
        """
        imssHome = "%s/imss" %(self.getInstallPath())
        # Auto remove paths.
        self.autoRemove = ["%s/bin" %(imssHome),
                           "%s/bind" %(imssHome),
                           "%s/cdt" %(imssHome),
                           "%s/ldap" %(imssHome),
                           "%s/ldaprep" %(imssHome),
                           "%s/lib" %(imssHome),
                           "%s/lib64" %(imssHome),
                           "%s/man" %(imssHome),
                           "%s/MsgTracing" %(imssHome),
                           "%s/OpenLDAP" %(imssHome),
                           "%s/patch" %(imssHome),
                           "%s/script" %(imssHome),
                           "%s/sql" %(imssHome),
                           "%s/temp" %(imssHome),
                           "%s/UI/adminUI/bin" %(imssHome),
                           "%s/UI/adminUI/ROOT" %(imssHome),
                           "%s/UI/apache" %(imssHome),
                           "%s/UI/euqUI/bin" %(imssHome),
                           "%s/UI/euqUI/ROOT" %(imssHome),
                           "%s/UI/javaJRE" %(imssHome),
                           "%s/UI/php/bin" %(imssHome),
                           "%s/UI/php/etc" %(imssHome),
                           "%s/UI/php/include" %(imssHome),
                           "%s/UI/php/lib" %(imssHome),
                           "%s/UI/php/php" %(imssHome),
                           "%s/UI/php/session" %(imssHome),
                           "%s/UI/tomcat" %(imssHome),
                           "%s/var" %(imssHome),
                           "%s/app_temp" %(imssHome),
                           "/var/app_data/imss/dtas_upload",
                           "/var/app_data/imss/openldap-data",
                           "/var/app_data/imss/snap_offline_report",
                           ]
        if self.getRemoveData() == 1:
            # If user chooses to remove data, the whole UI folder must be removed.
            self.autoRemove.append("%s/UI" %(imssHome))
        if self.getProduct() == 1:
            self.autoRemove.append("%s/postfix" %(imssHome))
        return list(self.autoRemove)
    
    def getManualRemovePaths(self):
        """
        Return paths that shall be removed by the user himself.
        """
        imssHome = "%s/imss" %(self.getInstallPath())
        self.manualRemove = ["%s/backup" %(imssHome)]
        return list(self.manualRemove)
    
    def getFailRemovePaths(self):
        """
        Get paths that fail to be removed.
        """
        return list(self.failRemovePaths)
    
    def setFailRemovePaths(self, val):
        self.failRemovePaths = list(val)

    def getBaseHfNo(self):
        return self.imssUpdBaseHf

    def getBasePatchNo(self):
        return self.imssUpdBasePatch

    def getBaseSpNo(self):
        return self.imssUpdBaseSp

    def getBackupFolder(self):
        return self.imssUpdOldBckPath

    def getDBBackupFolder(self):
        return self.imssUpdDbBckPath

    def getPgdataFolder(self):
        return self.pgdataFolder

    def getImssSystemdSettings(self):
        self.imssSystemdSettings = [(r"[Unit]"),
                                 (r"Description=InterScan Messaging Security Suite"),
                                 (r"After=network.target remote-fs.target nss-lookup.target "),
                                 (r"[Service]"),
                                 (r"Type=simple"),
                                 (r"RemainAfterExit=yes"),
                                 (r"ExecStart=%s/imss/script/imssctl.sh start" % self.getInstallPath()),
                                 (r"ExecStop=%s/imss/script/imssctl.sh stop" % self.getInstallPath()),
                                 (r"PrivateTmp=true"),
                                 (r"[Install]"),
                                 (r"WantedBy=multi-user.target")]
        return self.imssSystemdSettings
    
    def setDependencies(self, deps):
        self.dependencies = list(deps)
    
    def getDependencies(self):
        return list(self.dependencies)
