#!/usr/bin/env python
# -*- coding: utf-8 -*-

import resource
import utility
import step
import os.path
import urwid
import time
import threading
from model import g_upd_models, PostUpgradeAction, Const, Debugger
from step import ImssUpgradeInit,ImssUpgradeInstallNewVersion,ImssUpgradeAdminDB

class Installer(object):
    """
    Base class of installers.
    """
    
    def __init__(self, conf, res, logger, util):
        """
        Define necessary member variables.
        """
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
    
    def composeInstallSteps(self, mode=0):
        """
        Returns a list contains all install steps.
        
        mode: 0 the steps are composed for installation, 1 the steps are composed
              for uninstallation.
        """
        return []
    
    def composeUpgradeSteps(self):
        """
        Returns a list contains all uninstall steps.
        """
        return []
    
    def install(self):
        """
        Used to install IMSx.
        
        return: 0 if successful, -1 if failed.
        """
        return -1
    
    def uninstall(self):
        """
        Used to uninstall IMSx.
        
        return: 0 if successful, -1 if failed.
        """
        return -1
    
    def upgrade(self):
        """
        Used to upgrade previous IMSx version.
        
        return: 0 if successful, -1 if failed.
        """
        return -1
    
    def confirmUpgrade(self):
        """
        Used to confirm the upgrade.
        
        return: 0 if successful, -1 if failed.
        """
        return -1

class TerminalInstaller(Installer):
    """
    This is the installer which prints information on terminal directly.
    """
    def __init__(self, conf, res, logger, util):
        super(TerminalInstaller, self).__init__(conf, res, logger, util)
    
    def composeInstallSteps(self):
        updater = step.TerminalStatusUpdater()
        
        # Compose the steps of silent installation.
        steps = []
        steps.append(step.SilentInstallWelcome(self.conf, self.res, self.logger, self.util, False, updater))
        steps.append(step.SilentInstallPreCheck(self.conf, self.res, self.logger, self.util, True, updater))
        if self.conf.getCheckReq() == 1:
            steps.append(step.SilentInstallSysCheck(self.conf, self.res, self.logger, self.util, False, updater))
        # Skip system requirement check.
        else:
            self.logger.debug("Skip system requirement check.")
        steps.append(step.SilentInstallConfirmCompStatus(self.conf, self.res, self.logger, self.util, False, updater))
        # For IMSS only
        if self.conf.getProduct() == 0:
            steps.append(step.SilentInstallUsersAndGroups(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallRpms(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallReplaceConstants(self.conf, self.res, self.logger, self.util, True, updater))
        # If fresh install, need to install the db, including creating the db 
        # cluster, init db schemas and factory settings.
        if self.conf.getFreshInstall() == 1:
            steps.append(step.SilentInstallDatabase(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallRegisterDevice(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallService(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallCron(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallCleanup(self.conf, self.res, self.logger, self.util, True, updater))
        # For IMSVA, need to update the /etc/issue file.
        if self.conf.getProduct() == 1:
            steps.append(step.SilentInstallUpdateIssue(self.conf, self.res, self.logger, self.util, False, updater))
        steps.append(step.SilentInstallStartApp(self.conf, self.res, self.logger, self.util, False, updater))
        steps.append(step.SilentInstallImportDefAuComp(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallFoxhunterConfig(self.conf, self.res, self.logger, self.util, False, updater))
        
        return steps
    
    def install(self):
        self.logger.debug("Entering silent install mode.")
        
        updater = step.TerminalStatusUpdater()
        updater.append(self.res.getWording("msg.start_install"))
        steps = self.composeInstallSteps()
        # Perform the steps
        stepNo = -1
        interrupted = False
        for oneStep in steps:
            stepNo += 1
            self.logger.debug("Running step %d: %s..." %(stepNo, oneStep.__class__.__name__))
            if oneStep.apply() != 0:
                self.logger.debug("Fail to running step %d." %(stepNo))
                if not oneStep.isCritical:
                    self.logger.debug("This step is not critical, continue installation.")
                    continue
                else:
                    self.logger.debug("This step is critical, installation cannot continue.")
                    interrupted = True
                    break
        
        # If the installation doesn't finish successfully, need to rollback the changes.
        if interrupted:
            self.logger.debug("Installation failed in the middle, begin to rollback changes.")
            while stepNo >= 0:
                oneStep = steps[stepNo]
                self.logger.debug("Rolling back step %d: %s..." %(stepNo, oneStep.__class__.__name__))
                # Even if one step fails to rollback, still continue the whole rollback process.
                if oneStep.rollback() != 0:
                    self.logger.debug("Fail to rollback step %d." %(stepNo))
                stepNo -= 1
                    
            updater.append(self.res.getWording("msg.installFail"))
            # If install fail, pause to prompt the user.
            updater.append(self.res.getWording("msg.enterShellPrompt"))
            self.logger.debug("Silent installation finished.")
            return -1
        else:
            updater.append(self.res.getWording("msg.installSuccess"))
            self.logger.debug("Silent installation finished.")
            return 0
        
    def uninstall(self):
        """
        Uninstall the application silently.
        """
        self.logger.debug("Entering silent uninstall mode.")
        
        updater = step.TerminalStatusUpdater()
        # Currently not consider rollback for uninstallation. 
        oneStep = step.Step(self.conf, self.res, self.logger, self.util, False, updater)

        # Check installed rpms, if any one of rpms is not installed, the 
        # uninstall will stop because the install may not be complete.
        ret = oneStep.checkRpmForUninstallation(updater)
        if ret != 0:
            updater.append(self.res.getWording("msg.uninstallFail"))
            return -1
        
        # Uninstall the crontab.
        ret = oneStep.uninstallCron(updater)
        if ret != 0:
            updater.append(self.res.getWording("msg.uninstallFail"))
            return -1
        
        # Stop all services except DB.
        oneStep.stopAppExceptDB()
        
        # Try to unregister the device from admin DB.
        ret = oneStep.unregisterDeviceFromAdminDb(updater)
        if ret != 0:
            updater.append(self.res.getWording("msg.uninstallFail"))
            return -1
        
        # Stop all services.
        oneStep.stopApp()
        
        # For IMSS, revert the postfix settings.
        if self.conf.getProduct() == 0:
            ret = oneStep.removePostfixFoxlibSetting()
            if ret != 0:
                updater.append(self.res.getWording("msg.uninstallFail"))
                return -1
        
        # Uninstall the rpm.
        ret = oneStep.uninstallImsxRpms(updater)
        if ret != 0:
            updater.append(self.res.getWording("msg.uninstallFail"))
            return -1
        
        # Remove the files.
        ret = oneStep.removeImsxFiles(updater)
        if ret != 0:
            updater.append(self.res.getWording("msg.uninstallFail"))
            return -1
        
        # For IMSS, need to remove imsx related users and groups.
        if self.conf.getProduct() == 0:
            ret = oneStep.removeUsersAndGroups(updater)
            if ret != 0:
                updater.append(self.res.getWording("msg.uninstallFail"))
                return -1
        
        self.logger.debug("Silent uninstallation finished.")
        updater.append(self.res.getWording("msg.uninstallSuccess"))
        return 0
    
    def composeUpgradeSteps(self):
        # Compose the steps for upgrade.
        updater = step.TerminalStatusUpdater()
        steps = []
        steps.append(step.UpgradeWelcom(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradeInit(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradePreCheck(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradeInstallNewVersion(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradeLocalConfigurations(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradeAdminDB(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.UpgradeCleanup(self.conf, self.res, self.logger, self.util, True, updater))
        
        return steps
    
    def upgrade(self):
        # Currently only support the silent upgrade mode.
        updater = step.TerminalStatusUpdater()
        
        steps = self.composeUpgradeSteps()
        
        # Perform the steps
        stepNo = -1
        interrupted = False
        for oneStep in steps:
            stepNo += 1
            self.logger.debug("Running step %d: %s..." %(stepNo, oneStep.__class__.__name__))
            if oneStep.apply() != 0:
                self.logger.debug("Fail to running step %d." %(stepNo))
                if not oneStep.isCritical:
                    self.logger.debug("This step is not critical, continue installation.")
                    continue
                else:
                    self.logger.debug("This step is critical, installation cannot continue.")
                    interrupted = True
                    updater.append(self.res.getWording("msg.upgradeFailPrompt"))
                    break
        
        # If the upgrade doesn't finish successfully, need to rollback the changes.
        if interrupted:
            self.logger.debug("Installation failed in the middle, begin to rollback changes.")
            updater.append(self.res.getWording("msg.autoRollbackStartPrompt"))
            rollbackRes = True
            while stepNo >= 0:
                oneStep = steps[stepNo]
                self.logger.debug("Rolling back step %d: %s..." %(stepNo, oneStep.__class__.__name__))
                # If one step fails to rollback, stop the rollback process.
                if oneStep.rollback() != 0:
                    self.logger.debug("Fail to rollback step %d." %(stepNo))
                    rollbackRes = False
                    updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                    break
                stepNo -= 1
            
            if rollbackRes:
                updater.append(self.res.getWording("msg.rollbackSuccessPrompt"))
                    
            self.logger.debug("Upgrade failed.")
            return -1
        else:
            self.logger.debug("Upgrade succeeded.")
        
        # When upgrade finishes, no matter succeeded or failed, need to pause
        # to prompt the user.
        updater.append(self.res.getWording("msg.enterShellPrompt"))
        
        return 0
    
    def confirmUpgrade(self):
        updater = step.TerminalStatusUpdater()
        oneStep = step.Step(self.conf, self.res, self.logger, self.util, False, updater)
        # Check if the device is in dry run or finished dry run.
        if not os.path.exists(self.conf.getDryrunStartedFlag()) and not os.path.exists(self.conf.getDryrunStoppedFlag()):
            updater.append(self.res.getWording("msg.startDryrunThenConfirmPrompt"))
            return -1
        
        # Propmpt the user to accept or not.
        validYesResps = ["y", "yes"]
        validNoResps = ["n", "no"]
        userInput = raw_input(self.res.getWording("msg.confirmUpgradeStartPrompt"))
        self.logger.debug("The user input is %s" %(userInput))
        userInput = userInput.strip().lower()
        # Invalid user response.
        if userInput not in validYesResps and userInput not in validNoResps:
            self.logger.debug("The user input is invalid.")
            updater.append(self.res.getWording("msg.confirmInputInvalid"))
            return -1                
        # Valid user response.
        else:
            userConfirmYes = True    
            if userInput in validYesResps:
                userConfirmYes = True
            else:
                userConfirmYes = False
            
            # Confirm Yes.
            if userConfirmYes:
                # Stop dry run if necessary and restore imss_home.
                if os.path.exists(self.conf.getDryrunStartedFlag()):
                    ret = oneStep.stopDryrun()
                    if ret != 0:
                        self.logger.debug("Fail to stop dry run.")
                        return -1

                ret = oneStep.removeDryRunCronJob();
                if ret != 0:
                    self.logger.debug("Fail to remove cronjob.")
                    return -1

                try:
                    os.rename(os.path.join(self.conf.getInstallPath(), ".imss"), os.path.join(self.conf.getInstallPath(), "imss"))
                except Exception, e:
                    self.logger.debug("Fail to restore the imss home folder, %s" %(utility.excToString(e)))
                    return -1

                oneStep.startApp(updater)
                updater.append(self.res.getWording("msg.upgradeFinishPrompt") %(self.conf.getVersion()))
            # Confirm no, should do rollback
            else:
                steps = self.composeUpgradeSteps()
                # If in dry run period, the imss_home is normal, otherwise it is hidden.
                if os.path.exists(self.conf.getDryrunStartedFlag()):
                    imssiniPath = "%s/imss/config/imss.ini" %(self.conf.getInstallPath())
                else:
                    imssiniPath = "%s/.imss/config/imss.ini" %(self.conf.getInstallPath())
                if oneStep.confirmImsxRole(imssiniPath) != 0:
                    self.logger.debug("Fail to identify the current role of the device.")
                    updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                    return -1
                
                # Get the database setting.
                if os.path.exists(self.conf.getDryrunStartedFlag()):
                    odbciniPath = os.path.join(self.conf.getInstallPath(), "imss/config/odbc.ini")
                else:
                    odbciniPath = os.path.join(self.conf.getInstallPath(), ".imss/config/odbc.ini")
                if oneStep.confirmDatabaseConf(odbciniPath) != 0:
                    self.logger.debug("Fail to get the database setting.")
                    updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                    return -1
                
                # Parent needs to confirm all children has rolled back.
                if self.conf.getRole() == 0:
                    sql = "select app_ver from tb_component_list where scanner_id!=1;"
                    (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                               self.conf.getAdminDBPort(),
                                                               self.conf.getAdminDBUsername(),
                                                               self.util.decryptString(self.conf.getAdminDBPassword()),
                                                               self.conf.getAdminDBName(),
                                                               sql)
                    if e:
                        self.logger.debug("Fail to query children app versions from db, %s" %(utility.excToString(e)))
                        updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                        return -1
                    else:
                        for item in ret:
                            appVer = item[0]
                            self.logger.debug("The app version of one child is %s." %(appVer))
                            if appVer.count(".") != 3:
                                self.logger.debug("The format of app version %s is invalid." %(appVer))
                                updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                                return -1
                            else:
                                appVerSplit = appVer.split(".")
                                ver = float(".".join(appVerSplit[0:2]))
                                if ver != self.conf.getBaseVersion():
                                    updater.append(self.res.getWording("msg.rollbackChildFirstPrompt"))
                                    return -1
                
                # Stop dry run if necessary and restore imss_home.
                if os.path.exists(self.conf.getDryrunStartedFlag()):
                    ret = oneStep.stopDryrun()
                    if ret != 0:
                        self.logger.debug("Fail to stop dry run.")
                        updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                        return -1

                ret = oneStep.removeDryRunCronJob();
                if ret != 0:
                    self.logger.debug("Fail to remove cronjob.")
                    updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                    return -1

                try:
                    os.rename(os.path.join(self.conf.getInstallPath(), ".imss"), os.path.join(self.conf.getInstallPath(), "imss"))
                except Exception, e:
                    self.logger.debug("Fail to restore the imss home folder, %s" %(utility.excToString(e)))
                    updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                    return -1

                updater.append(self.res.getWording("msg.logsRemoveAfterRollback"))
                updater.append(self.res.getWording("msg.autoRollbackStartPrompt"))
                for item in reversed(steps):
                    if item.rollback() != 0:
                        self.logger.debug("Rollback failed at step %s." %(item.__class__.__name__))
                        updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                        return -1
                
                # IMSVA 9.1-000264, in our original design, after child rollback and restart, 
                # imssmgr will update its version to tb_component_list, and parent knows the
                # child has rolled back successfully. But in IMSVA 9.1, the database authentication
                # method is changed to md5, imssmgr in IMSVA 9.0 cannot update the version to
                # tb_component_list, as a result, parent don't know child has rolled back.
                # To fix this, here we update the app version to 9.0.0000, so that parent 
                # knows child has rolled back. And set an admin cmd to restart child's imssmgr,
                # then once child's imssmgr can connect to database again, it will restart
                # and update the correct version to tb_component_list.
                # 
                # NOTE: the code can be removed in next release if database authentication
                # method is unchanged.
                if self.conf.getRole() == 1:
                    sql = "update tb_component_list set app_ver=%s, admin_cmd=%s where scanner_id=%s;"
                    para = ["%s.0.0000" %(self.conf.getBaseVersion()), 192, oneStep.getLocalScannerId()]
                    (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                                self.conf.getAdminDBPort(),
                                                                self.conf.getAdminDBUsername(),
                                                                self.util.decryptString(self.conf.getAdminDBPassword()),
                                                                self.conf.getAdminDBName(),
                                                                sql,
                                                                para)
                    if ret != 0:
                        self.logger.debug("Failed to execute sql command: %s" %(utility.excToString(e)))
                        updater.append(self.res.getWording("msg.manualRollbackPrompt"))
                        return -1
                
                updater.append(self.res.getWording("msg.rollbackSuccessPrompt"))
            
            # Clean the backup files.
            fileList = [self.conf.getDryrunStoppedFlag(),
                        self.conf.getCronBackupPath(),
                        self.conf.getAdminDBBackupFolder(),
                        self.conf.getAdminDbExportPath(),
                        self.conf.getLogBackupFolder(),
                        "/var/app_data/imss/pg92"]
            for path in fileList:
                (ret, e) = self.util.removePath(path)
                if ret != 0:
                    self.logger.debug("Fail to remove %s, %s" %(path, utility.excToString(e)))
            
            self.logger.debug("Upgrade confirm finished.")
            return 0

class TuiInstaller(Installer):
    """
    This class is the TUI version of installer, this version is mainly for
    IMSS Linux.
    """
    def __init__(self, conf, res, logger, util):
        super(TuiInstaller, self).__init__(conf, res, logger, util)
        
        self.installFrame = InstallerFrame(conf, res, logger, util)
        self.uninstallFrame = InstallerFrame(conf, res, logger, util, mode=1)
        self.loop = None
        # The loop status:
        #   0: not start
        #   1: started
        #   2: exited
        self.loopStatus = 0
    
    def composeInstallSteps(self, mode):
        steps= []
        updater = None
        # For installation, the updater should be a widget in installation UI.
        if mode == 0:
            updater = self.installFrame.copyPage.updater
        # For uninstalltion, the updater should be a widget in uninstallation UI. 
        else:
            pass
        
        if self.conf.getProduct() == 0:
            steps.append(step.SilentInstallUsersAndGroups(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallRpms(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallReplaceConstants(self.conf, self.res, self.logger, self.util, True, updater))
        # If fresh install, need to install the db, including creating the db 
        # cluster, init db schemas and factory settings.
        if self.conf.getFreshInstall() == 1:
            steps.append(step.SilentInstallDatabase(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallRegisterDevice(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallService(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallCron(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallCleanup(self.conf, self.res, self.logger, self.util, True, updater))
        # For IMSVA, need to update the /etc/issue file.
        if self.conf.getProduct() == 1:
            steps.append(step.SilentInstallUpdateIssue(self.conf, self.res, self.logger, self.util, False, updater))
        steps.append(step.SilentInstallStartApp(self.conf, self.res, self.logger, self.util, False, updater))
        steps.append(step.SilentInstallImportDefAuComp(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(step.SilentInstallFoxhunterConfig(self.conf, self.res, self.logger, self.util, False, updater))
        
        return steps
    
    def refreshScreen(self):
        """
        Refresh the screen in schedule.
        """
        while self.loopStatus != 2:
            try:
                if self.loopStatus == 1:
                    self.loop.draw_screen()
            # Ignore exceptions.
            except:
                pass
            time.sleep(1)
    
    def install(self):
        self.loop = urwid.MainLoop(self.installFrame, palette=self.res.palette, handle_mouse=False)
        steps = self.composeInstallSteps(0)
        self.installFrame.copyPage.setCopySteps(steps)
        # Use a thread to refresh the screen in schedule. Otherwise, the screen
        # gets stuck if we stay in keypress method.
        refreshThd = threading.Thread(target=self.refreshScreen)
        refreshThd.start()
        self.loopStatus = 1
        self.loop.run()
        self.loopStatus = 2
        refreshThd.join()
        if self.installFrame.complete:
            self.logger.debug("Installation completed successfully.")
            return 0
        else:
            self.logger.debug("Installation aborted in the middle.")
            return -1
    
    def uninstall(self):
        self.loop = urwid.MainLoop(self.uninstallFrame, palette=self.res.palette, handle_mouse=False)
        refreshThd = threading.Thread(target=self.refreshScreen)
        refreshThd.start()
        self.loopStatus = 1
        self.loop.run()
        self.loopStatus = 2
        refreshThd.join()
        if self.uninstallFrame.complete:
            self.logger.debug("Uninstallation completed successfully.")
            return 0
        else:
            self.logger.debug("Uninstallation aborted in the middle.")
            return -1

class InstallerFrame(urwid.WidgetWrap):
    """
    The frame of installer, this is meant to organize pages.
    """
    def __init__(self, conf, res, logger, util, mode=0):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        # Status indicators
        # 
        # Whether the installation is completed successfully, or exited in the
        # middle.
        self.complete = False
        # The mode is installation (0) or uninstallation (1)
        self.mode = mode
        
        # Initialize UI widgets.
        self.header = InstallerHeader(self.conf, self.res, self.logger, self.util, urwid.Text(u""))
        self.footer = InstallerFooter(self.conf, self.res, self.logger, self.util, urwid.Text(u""))
        self.wrappedFrame = urwid.Frame(None, header=self.header, footer=self.footer)
        
        # Create pages for installation.
        if self.mode == 0:
            self.welcomePage = InstallWelcomePage(self.conf, self.res, self.logger, self.util, self)
            self.licensePage = InstallLicensePage(self.conf, self.res, self.logger, self.util, self)
            self.typePage = InstallTypePage(self.conf, self.res, self.logger, self.util, self)
            self.databasePage = InstallDatabasePage(self.conf, self.res, self.logger, self.util, self)
            self.regPage = InstallRegPage(self.conf, self.res, self.logger, self.util, self)
            self.destinationPage = InstallDestinationPage(self.conf, self.res, self.logger, self.util, self)
            self.senderFilteringPage = InstallSenderFilteringPage(self.conf, self.res, self.logger, self.util, self)
            self.preCheckPage = InstallPreCheckPage(self.conf, self.res, self.logger, self.util, self)
            self.copyPage = InstallCopyPage(self.conf, self.res, self.logger, self.util, self)
            self.wrappedFrame.body = self.welcomePage
            # Total step number.
            self.totalStepNum = 8
        # Create and organize pages for uninstallation.
        else:
            self.uninstallWelcomePage = UninstallWelcomePage(self.conf, self.res, self.logger, self.util, self)
            self.uninstallOptionPage = UninstallOptionPage(self.conf, self.res, self.logger, self.util, self)
            self.uninstallRemovePage = UninstallRemovePage(self.conf, self.res, self.logger, self.util, self)
            self.wrappedFrame.body = self.uninstallWelcomePage
            
            self.totalStepNum = 3
            
        # Step starts from 1.
        self.currentStepNo = 1
        
        super(InstallerFrame, self).__init__(self.wrappedFrame)
        self.wrappedFrame.body.drawPage()
    
    def setHeaderText(self, text):
        """
        Set text displayed on the header.
        """
        self.header.setText(text)
    
    def setFooterText(self, text):
        """
        Set text displayed on the footer.
        """
        self.footer.setText(text)
    
    def setPage(self, page):
        """
        Set the current page.
        
        page: the page to be the current page.
        """
        self.wrappedFrame.body = page
    
    def keypress(self, size, key):
        if key == "ctrl x":
            self.logger.debug("The user closes the UI.")
        return super(InstallerFrame, self).keypress(size, key)
    
    def switchToPrevPage(self, page):
        """
        Make the frame to show the previous page of the current page.
        
        return: 0 if it is OK to switch to previous page, -1 if failed.
        """
        prevPage = None
        # Installation
        if self.mode == 0:
            if page == self.preCheckPage:
                prevPage = self.senderFilteringPage
            elif page == self.senderFilteringPage:
                prevPage = self.destinationPage
            elif page == self.destinationPage and self.conf.getFreshInstall() == 1:
                prevPage = self.databasePage
            elif page == self.destinationPage and self.conf.getFreshInstall() == 0:
                prevPage = self.regPage
            elif page == self.databasePage or page == self.regPage:
                prevPage = self.typePage
            elif page == self.typePage:
                prevPage = self.licensePage
        # Uninstallation
        elif self.mode == 1:
            pass
        
        if prevPage:
            self.currentStepNo -= 1
            self.setPage(prevPage)
            prevPage.enterPage()
            return 0
        else:
            return -1
    
    def switchToNextPage(self, page):
        """
        Make the frame to show the next page of the current page.
        
        return: 0 if it is OK to switch to next page, -1 if failed.
        """
        nextPage = None
        # Installation
        if self.mode == 0:
            if page == self.welcomePage:
                nextPage = self.licensePage
            elif page == self.licensePage:
                nextPage = self.typePage
            elif page == self.typePage and self.conf.getFreshInstall() == 1:
                nextPage = self.databasePage
            elif page == self.typePage and self.conf.getFreshInstall() == 0:
                nextPage = self.regPage
            elif page == self.databasePage or page == self.regPage:
                nextPage = self.destinationPage
            elif page == self.destinationPage:
                nextPage = self.senderFilteringPage
            elif page == self.senderFilteringPage:
                nextPage = self.preCheckPage
            elif page == self.preCheckPage:
                nextPage = self.copyPage
        # Uninstallation
        elif self.mode == 1:
            if page == self.uninstallWelcomePage:
                nextPage = self.uninstallOptionPage
            elif page == self.uninstallOptionPage:
                nextPage = self.uninstallRemovePage
        
        if nextPage:
            self.currentStepNo += 1
            self.setPage(nextPage)
            nextPage.enterPage()
            return 0
        else:
            return -1
    
class InstallerHeader(urwid.WidgetWrap):
    """
    This is the header of the tui installer. Basically its content is unchanged.
    """
    def __init__(self, conf, res, logger, util, text):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        self.text = text
        padding = urwid.Padding(self.text, left=2)
        pile = urwid.Pile([urwid.Divider(), padding, urwid.Divider()])
        banner = urwid.AttrMap(pile, resource.PALETTE_HEADER)
        super(InstallerHeader, self).__init__(banner)
    
    def setText(self, text):
        self.text.set_text(text)

class InstallerFooter(urwid.WidgetWrap):
    """
    This is the footer of the installer, it is used to display operation instructions.
    """
    def __init__(self, conf, res, logger, util, text):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        self.text = text
        padding = urwid.Padding(self.text, left=2)
        attrMap = urwid.AttrMap(padding, resource.PALETTE_FOOTER)
        super(InstallerFooter, self).__init__(attrMap)
        
    def setText(self, text):
        """
        Set the display text on the footer.
        """
        self.text.set_text(text)

class InstallerPage(urwid.WidgetWrap):
    """
    The parent class of all tui installer pages. 
    """
    def __init__(self, widget, frame, mode=0):
        """
        Initialize a installer page.
        
        widget: the widget that will be displayed on the page.
        frame: the frame that contains this page.
        mode: the page mode means the page is used in installation, uninstallation
              or other scenarios. The value is the same as defined in frame.
        """
        self.content = widget
        self.frame = frame
        self.mode = mode
        
        displayWidget = urwid.AttrMap(urwid.Padding(self.content, left=6, right=6), resource.PALETTE_BODY)
        super(InstallerPage, self).__init__(displayWidget)
    
    def goPrevPage(self):
        """
        Go to the previous page.
        
        return: 0 if it is OK to go to previous page, -1 if cannot.
        """
        if self.frame.switchToPrevPage(self) == 0:
            return 0
        else:
            return -1
        
    def goNextPage(self):
        """
        Got to next page.
        
        return: 0 if it is OK to go to next page, -1 if cannot.
        """
        if self.frame.switchToNextPage(self) == 0:
            return 0
        else:
            return -1
    
    def enterPage(self):
        """
        Things need to do when entering this page from other pages.
        """
        self.drawPage()
    
    def drawPage(self):
        """
        Draw the page, fill in proper values.
        """
        if self.mode == 0:
            self.frame.setHeaderText([self.res.getWording("txt.bannerBegin"),
                                      (resource.PALETTE_HEADER_EM, self.res.getWording("txt.bannerMiddle")),
                                      self.res.getWording("txt.bannerEndInstall"),
                                      self.res.getWording("txt.bannerPage") %(self.frame.currentStepNo, self.frame.totalStepNum)])
        elif self.mode == 1:
            self.frame.setHeaderText([self.res.getWording("txt.bannerBegin"),
                                      (resource.PALETTE_HEADER_EM, self.res.getWording("txt.bannerMiddle")),
                                      self.res.getWording("txt.bannerEndUninstall"),
                                      self.res.getWording("txt.bannerPage") %(self.frame.currentStepNo, self.frame.totalStepNum)])
        elif self.mode == 2:
            self.frame.setHeaderText([self.res.getWording("txt.bannerBegin"),
                                      (resource.PALETTE_HEADER_EM, self.res.getWording("txt.bannerMiddle")),
                                      self.res.getWording("txt.bannerEndUpgrade"),
                                      self.res.getWording("txt.bannerPage") %(self.frame.currentStepNo, self.frame.totalStepNum)])


class InstallWelcomePage(InstallerPage):
    """
    The welcome page of the installer.
    """

    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util

        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.welcome.title")))
        desc = urwid.Text(self.res.getWording("txt.welcome.desc"))
        cont = urwid.Text([self.res.getWording("txt.welcome.continue.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.welcome.continue.mid")),
                           self.res.getWording("txt.welcome.continue.end")])
        canc = urwid.Text([self.res.getWording("txt.welcome.cancel.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.welcome.cancel.mid")),
                           self.res.getWording("txt.welcome.cancel.end")])
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      cont,
                      urwid.Divider(),
                      canc]
        content = urwid.ListBox(urwid.SimpleListWalker(widgetList))
        super(InstallWelcomePage, self).__init__(content, frame)

    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.welcome.footer"))
        super(InstallWelcomePage, self).drawPage()

    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallWelcomePage, self).keypress(size, key)


class UpgradeWelcomePage(InstallerPage):
    """
    The welcome page of the installer.
    """

    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util

        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.welcome.title")))
        desc = urwid.Text(self.res.getWording("txt.upgrade.welcome.desc"))
        cont = urwid.Text([self.res.getWording("txt.upgrade.welcome.continue.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.welcome.continue.mid")),
                           self.res.getWording("txt.welcome.continue.end")])
        canc = urwid.Text([self.res.getWording("txt.upgrade.welcome.cancel.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.welcome.cancel.mid")),
                           self.res.getWording("txt.welcome.cancel.end")])
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      cont,
                      urwid.Divider(),
                      canc]
        content = urwid.ListBox(urwid.SimpleListWalker(widgetList))
        super(UpgradeWelcomePage, self).__init__(content, frame, mode=2)

    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.welcome.footer"))
        super(UpgradeWelcomePage, self).drawPage()

    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UpgradeWelcomePage, self).keypress(size, key)

    
class InstallLicensePage(InstallerPage):
    def __init__(self, conf, res, logger, util, frame, mode=0):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.mode = mode
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.license.title")))
        title = urwid.Pile([urwid.Divider(), 
                            title,
                            urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]), 
                            ])
        desc = urwid.Text(self.res.getEULAContent())
        desc = urwid.Padding(desc, width=("relative", 80))
        self.licenseTextArea = urwid.Filler(urwid.ListBox(urwid.SimpleListWalker([desc])), height=('relative', 100), bottom=1)
        innerFrame = urwid.Frame(self.licenseTextArea, header=title, footer=urwid.Divider())
        super(InstallLicensePage, self).__init__(innerFrame, frame, self.mode)
    
    def enterPage(self):
        super(InstallLicensePage, self).enterPage()
        
    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.license.footer"))
        super(InstallLicensePage, self).drawPage()
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        # Let the license text area handle scroll actions
        elif key == "up" or key == "down" or key == "page up" or key == "page down":
            return self.licenseTextArea.keypress(size, key)
        else:
            return super(InstallLicensePage, self).keypress(size, key)

class InstallTypePage(InstallerPage):
    """
    This page let the user choose installation type: fresh or append.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.type.title")))
        desc = urwid.Text(self.res.getWording("txt.type.desc"))
        desc = urwid.Padding(desc, width=("relative", 80))
        typeBtns = []
        self.freshBtn = urwid.RadioButton(typeBtns, self.res.getWording("txt.type.fresh"), on_state_change=self.changeInstallType)
        self.appendBtn = urwid.RadioButton(typeBtns, self.res.getWording("txt.type.append"))
        freshBtnDesc = urwid.Text(self.res.getWording("txt.type.fresh.desc"))
        appendBtnDesc = urwid.Text(self.res.getWording("txt.type.append.desc"))
        btnWidth= 20
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      urwid.Padding(urwid.AttrMap(self.freshBtn, resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=btnWidth),
                      urwid.Padding(freshBtnDesc, left=4),
                      urwid.Padding(urwid.AttrMap(self.appendBtn, resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=btnWidth),
                      urwid.Padding(appendBtnDesc, left=4),
                      ]
        super(InstallTypePage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)),frame)
    
    def changeInstallType(self, btn, newState):
        if btn == self.freshBtn and newState:
            self.conf.setFreshInstall(1)
            self.logger.debug("Choose fresh install.")
        else:
            self.conf.setFreshInstall(0)
            self.logger.debug("Choose append install.")
    
    def drawPage(self):
        super(InstallTypePage, self).drawPage()
        self.frame.setFooterText(self.res.getWording("txt.type.footer"))
        if self.conf.getFreshInstall() == 1:
            self.freshBtn.toggle_state()
        else:
            self.appendBtn.toggle_state()
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallTypePage, self).keypress(size, key)

class ImssEdit(urwid.Edit):
    """
    This enhance the urwid built-in edit.
    
    maxLen: specify the maximum number of characters allowed to input, set to -1 if no limit
    """
    def __init__(self, caption=u'', edit_text=u'', multiline=False, align='left', wrap='space', 
                 allow_tab=False, edit_pos=None, layout=None, mask=None, maxLen=-1):
        super(ImssEdit, self).__init__(caption=caption, edit_text=edit_text, multiline=multiline,align=align,wrap=wrap,
                                       allow_tab=allow_tab, edit_pos=edit_pos, layout=layout, mask=mask)
        
        self.maxLen = maxLen
    
    def insert_text_result(self, text):
        # If the inserted text will cause exceeding max length, stop inserting.
        if self.maxLen >= 0 and len(text) + len(self.get_edit_text()) > self.maxLen:
            return (self.edit_text, self.edit_pos)
        else:
            return super(ImssEdit, self).insert_text_result(text)
        
class ImssIntEdit(urwid.IntEdit):
    """
    This enhance the urwid built-in int edit.
    
    maxLen: specify the maximum number of characters allowed to input, set to -1 if no limit
    """
    def __init__(self, caption='', default=None, maxLen=-1,maxNumber=-1):
        super(ImssIntEdit, self).__init__(caption=caption, default=default)
        
        self.maxLen = maxLen
        self.maxNumber = maxNumber
    def insert_text_result(self, text):
        # If the inserted text will cause exceeding max length, stop inserting.
        if self.maxLen >= 0 and len(text) + len(self.get_edit_text()) > self.maxLen:
            return (self.edit_text, self.edit_pos)
        elif self.maxNumber > 0 and long(self.edit_text + text)  > self.maxNumber:
            return (self.edit_text, self.edit_pos)
        else:
            return super(ImssIntEdit, self).insert_text_result(text)

class InstallDatabasePage(InstallerPage):
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.database.title")))
        desc = urwid.Text(self.res.getWording("txt.database.desc"))
        desc = urwid.Padding(desc, width=("relative", 80))
        dbTypeBtns = []
        self.bundleDbBtn = urwid.RadioButton(dbTypeBtns, self.res.getWording("txt.database.useBundle"), on_state_change=self.changeDbType)
        self.remoteDbBtn = urwid.RadioButton(dbTypeBtns, self.res.getWording("txt.database.useRemote"))
        self.infoDiv = urwid.Text((resource.PALETTE_BODY_EM, self.res.getWording("txt.database.dividerBundle")))
        addrLabel = urwid.Text(self.res.getWording("txt.database.address"))
        portLabel = urwid.Text(self.res.getWording("txt.database.port"))
        dbNameLabel = urwid.Text(self.res.getWording("txt.database.dbName"))
        userLabel = urwid.Text(self.res.getWording("txt.database.user"))
        passwdLabel = urwid.Text(self.res.getWording("txt.database.password"))
        self.addrEditable = ImssEdit(maxLen=Const.MAX_ADDRESS_LEN)
        self.portEditable = ImssIntEdit(maxLen=5, maxNumber=65535)
        # For bundled db, address and port cannot be modified.
        self.addrReadonly = urwid.Text("")
        self.portReadonly = urwid.Text("")
        self.dbNameEdit = ImssEdit(maxLen=Const.MAX_DATABASE_NAME_LEN)
        self.userEdit = ImssEdit(maxLen=Const.MAX_DATABASE_USER_LEN)
        self.passwdEdit = ImssEdit(mask="*", maxLen=Const.MAX_DATABASE_PASSWORD_LEN)
        labelLen = 20
        EditLen = 32
        tableWidth = labelLen + EditLen + 2
        # This the table for bundle db config.
        self.dbBundleTable = urwid.Pile([urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, addrLabel), (1, urwid.Text(u"")), (EditLen, self.addrReadonly)]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, portLabel), (1, urwid.Text(u"")), (EditLen, self.portReadonly)]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, dbNameLabel), (1, urwid.Text(u"[")), (EditLen, self.dbNameEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, userLabel), (1, urwid.Text(u"[")), (EditLen, self.userEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, passwdLabel), (1, urwid.Text(u"[")), (EditLen, self.passwdEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         ])
        self.dbBundleTable = urwid.Padding(self.dbBundleTable, left=4)
        # This is the table for remote db config.
        self.dbRemoteTable = urwid.Pile([urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, addrLabel), (1, urwid.Text(u"[")), (EditLen, self.addrEditable), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, portLabel), (1, urwid.Text(u"[")), (EditLen, self.portEditable), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, dbNameLabel), (1, urwid.Text(u"[")), (EditLen, self.dbNameEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, userLabel), (1, urwid.Text(u"[")), (EditLen, self.userEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, passwdLabel), (1, urwid.Text(u"[")), (EditLen, self.passwdEdit), (1, urwid.Text(u"]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
                                         ])
        self.dbRemoteTable = urwid.Padding(self.dbRemoteTable, left=4)
        self.errorMsg = urwid.Text("")
        btnWidth = 30
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      urwid.Padding(urwid.AttrMap(self.bundleDbBtn, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=btnWidth),
                      urwid.Padding(urwid.AttrMap(self.remoteDbBtn, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=btnWidth),
                      urwid.Divider(),
                      urwid.Padding(self.infoDiv, left=4),
                      self.dbBundleTable,
                      urwid.Divider(),
                      self.errorMsg]
        self.walker = urwid.SimpleFocusListWalker(widgetList)
        self.updater = self.DatabasePageUpdater(self.errorMsg)
        super(InstallDatabasePage, self).__init__(urwid.ListBox(self.walker), frame)
    
    def switchDbEditTable(self, dbType):
        """
        Switch the db edit table according to db type, for bundled db,
        address and port cannot be modified. Modify the list walker object
        so that the screen can be re-drawn. 
        """
        # The pop position may change if UI layout changes.
        self.walker.pop(-3)
        if dbType == 0:
            self.walker.insert(-2, self.dbBundleTable)
        elif dbType == 1:
            self.walker.insert(-2, self.dbRemoteTable)
    
    def changeDbType(self, btn, newState):
        """
        Handles db type change signal.
        """
        # Use bundled db.
        if newState:
            self.logger.debug("The user changes to use bundle db.")
            self.conf.setAdminDBType(0)
            # Bundled db always uses address 127.0.0.1 and port 5432.
            self.conf.setAdminDBAddress("127.0.0.1")
            self.conf.setAdminDBPort(5432)
            self.switchDbEditTable(0)
            self.infoDiv.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.database.dividerBundle")))
        # Remote
        else:
            self.logger.debug("The user changes to use remote db.")
            self.conf.setAdminDBType(1)
            self.switchDbEditTable(1)
            self.infoDiv.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.database.dividerRemote")))
        
        self.drawPage()
    
    def enterPage(self):
        # Clear the error message.
        self.errorMsg.set_text("")
        super(InstallDatabasePage, self).enterPage()
    
    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.database.footer"))
        self.addrReadonly.set_text(self.conf.getAdminDBAddress())
        self.portReadonly.set_text(str(self.conf.getAdminDBPort()))
        self.addrEditable.set_edit_text(self.conf.getAdminDBAddress())
        self.portEditable.set_edit_text(str(self.conf.getAdminDBPort()))
        self.dbNameEdit.set_edit_text(self.conf.getAdminDBName())
        self.userEdit.set_edit_text(self.conf.getAdminDBUsername())
        self.passwdEdit.set_edit_text(self.util.decryptString(self.conf.getAdminDBPassword()))
        super(InstallDatabasePage, self).drawPage()
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallDatabasePage, self).keypress(size, key)
    
    class DatabasePageUpdater(step.StatusUpdater):
        """
        The status updater for database page.
        """
        def __init__(self, text):
            self.text = text
        
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY_ERR_FOCUS, msg))
    
    def goNextPage(self):
        if self.conf.getAdminDBType() == 1:
            self.conf.setAdminDBAddress(self.addrEditable.edit_text)
            self.conf.setAdminDBPort(self.portEditable.value())
        self.conf.setAdminDBName(self.dbNameEdit.edit_text)
        self.conf.setAdminDBUsername(self.userEdit.edit_text)
        self.conf.setAdminDBPassword(self.util.encryptString(self.passwdEdit.edit_text))
        # Check if db config is correct.
        if self.step.checkDatabaseConfiguration(self.updater) == 0:
            return super(InstallDatabasePage, self).goNextPage()
        else:
            # Prepend the error label.
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS, self.errorMsg.text)])
            return -1
    
class InstallRegPage(InstallerPage):
    """
    The page is to let the user provide parent IMSx information and retrieve adminDB
    information.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.reg.title")))
        desc = urwid.Text(self.res.getWording("txt.reg.desc"))
        desc = urwid.Padding(desc, width=("relative", 80))
        
        addrLabel = urwid.Text(self.res.getWording("txt.reg.address"))
        userLabel = urwid.Text(self.res.getWording("txt.reg.user"))
        passwdLabel = urwid.Text(self.res.getWording("txt.reg.passwd"))
        self.addrEdit = ImssEdit(maxLen=Const.MAX_ADDRESS_LEN)
        self.userEdit = ImssEdit(maxLen=Const.MAX_ADMIN_USER_LEN)
        self.passwdEdit = ImssEdit(mask="*", maxLen=Const.MAX_ADMIN_PASSWORD_LEN)
        labelLen = 30
        editLen = 30
        rowLen = labelLen + editLen +2
        addrRow = urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, addrLabel), (1, urwid.Text("[")), (editLen, self.addrEdit), (1, urwid.Text("]"))]), resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=rowLen)
        userRow = urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, userLabel), (1, urwid.Text("[")), (editLen, self.userEdit), (1, urwid.Text("]"))]), resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=rowLen)
        passwdRow = urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, passwdLabel), (1, urwid.Text("[")), (editLen, self.passwdEdit), (1, urwid.Text("]"))]), resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=rowLen)
        
        self.errorMsg = urwid.Text("")
        self.updater = self.RegPageUpdater(self.errorMsg)
        
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      addrRow,
                      userRow,
                      passwdRow,
                      urwid.Divider(),
                      self.errorMsg
                      ]
        
        super(InstallRegPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame)
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallRegPage, self).keypress(size, key)
    
    def enterPage(self):
        # Clear the error message.
        self.errorMsg.set_text("")
        super(InstallRegPage, self).enterPage()
    
    def drawPage(self):
        super(InstallRegPage, self).drawPage()
        self.frame.setFooterText(self.res.getWording("txt.reg.footer"))
        self.addrEdit.set_edit_text(self.conf.getParentAddress())
        self.userEdit.set_edit_text(self.conf.getAdminUser())
        self.passwdEdit.set_edit_text(self.conf.getAdminPassword())
    
    def goNextPage(self):
        self.conf.setParentAddress(self.addrEdit.edit_text)
        self.conf.setAdminUser(self.userEdit.edit_text)
        self.conf.setAdminPassword(self.passwdEdit.edit_text)
        # Check if can get adminDB info from parent..
        if self.step.getAdminDBInfoFromParent(self.updater) == 0:
            return super(InstallRegPage, self).goNextPage()
        else:
            # Prepend the error label.
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS, self.errorMsg.text)])
            return -1
    
    class RegPageUpdater(step.StatusUpdater):
        """
        The status updater for register page.
        """
        def __init__(self, text):
            self.text = text
        
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY_ERR_FOCUS, msg))

class InstallDestinationPage(InstallerPage):
    """
    The page to specify installation folder.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
         
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.destination.title")))
        desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.destination.desc")))
        self.label = urwid.Text(self.res.getWording("txt.destination.input"))
        self.input = urwid.Edit()
        self.errorMsg = urwid.Text("")
        self.updater = self.DestinationPageUpdater(self.errorMsg)
        labelLen = 20
        inputLen = 42
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.label), (1, urwid.Text("[")), (inputLen, self.input), (1, urwid.Text("]"))]), resource.PALETTE_BODY_FOCUS), width=labelLen+inputLen+2),
                      urwid.Divider(),
                      self.errorMsg]
         
        super(InstallDestinationPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame)
    
    class DestinationPageUpdater(step.StatusUpdater):
        """
        Status updater for destination page.
        """
        def __init__(self, text):
            self.text = text
        
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY_ERR_FOCUS, msg))
    
    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.destination.footer"))
        self.input.set_edit_text(self.conf.getInstallPath())
        super(InstallDestinationPage, self).drawPage()
        
    def enterPage(self):
        # Clear the error message.
        self.errorMsg.set_text("")
        super(InstallDestinationPage, self).enterPage()
        
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallDestinationPage, self).keypress(size, key)
    
    def goNextPage(self):
        self.conf.setInstallPath(os.path.normpath(self.input.edit_text))
        # Check if the destination folder is valid.
        if self.step.checkInstallPath(self.updater) == 0:
            return super(InstallDestinationPage, self).goNextPage()
        else:
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS, self.errorMsg.text)])
            return -1

class InstallSenderFilteringPage(InstallerPage):
    """
    The page is to configure sender filtering service.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.senderfiltering.title")))
        desc = urwid.Text(self.res.getWording("txt.senderfiltering.desc"))
        desc = urwid.Padding(desc, width=("relative", 80))
        
        self.enableErs = urwid.CheckBox(self.res.getWording("txt.senderfiltering.enableErs"), on_state_change=self.changeEnableErs)
        self.enableIpProfiler = urwid.CheckBox(self.res.getWording("txt.senderfiltering.enableIpProfiler"), on_state_change=self.changeEnableIpProfiler)
        checkboxLen = 30
        
        portLabel = urwid.Text(self.res.getWording("txt.senderfiltering.port"))
        self.portInput = ImssIntEdit(maxLen=5,maxNumber=65535)
        labelLen = 18
        inputLen = 7
        
        self.errorMsg = urwid.Text("")
        if self.conf.getFreshInstall() == 0:
            senderfilteringPortDesc = self.res.getWording("txt.senderfiltering.port.desc.child")  
        else:
            senderfilteringPortDesc = self.res.getWording("txt.senderfiltering.port.desc.parent")
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      urwid.Padding(urwid.AttrMap(self.enableErs, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=checkboxLen),
                      urwid.Padding(urwid.Text(self.res.getWording("txt.senderfiltering.enableErs.desc")), left=4),
                      urwid.Padding(urwid.AttrMap(self.enableIpProfiler, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=checkboxLen),
                      urwid.Padding(urwid.Text(self.res.getWording("txt.senderfiltering.enableIpProfiler.desc")), left=4),
                      urwid.Divider(),
                      urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, portLabel), (1, urwid.Text("[")), (inputLen, self.portInput), (1, urwid.Text("]"))]), resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=labelLen+inputLen+2),
                      urwid.Padding(urwid.Text(senderfilteringPortDesc), left=4),
                      urwid.Divider(),
                      self.errorMsg]
        self.walker = urwid.SimpleFocusListWalker(widgetList)
        self.updater = self.SenderFilteringPageUpdater(self.errorMsg)
        super(InstallSenderFilteringPage, self).__init__(urwid.ListBox(self.walker), frame)
    
    def changeEnableErs(self, checkbox, newState):
        """
        Handles enable ERS checkbox change signal.
        """
        if newState:
            self.conf.setEnableErs(1)
        else:
            self.conf.setEnableErs(0)

        self.drawPage()
        
    def changeEnableIpProfiler(self, checkbox, newState):
        """
        Handles enable ip profiler checkbox change signal.
        """
        if newState:
            self.conf.setEnableIpProfiler(1)
        else:
            self.conf.setEnableIpProfiler(0)
        
        self.drawPage()
    
    def enterPage(self):
        # Clear the error message.
        self.errorMsg.set_text("")
        super(InstallSenderFilteringPage, self).enterPage()
    
    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.senderfiltering.footer"))
        #if it is append install, we just read senderfiltering configuration from parent.
        if self.conf.getFreshInstall() == 0:
            self.logger.debug("senderfiltering use parent's ERS and  IpProfile configuration")
            if self.util.isParentErsEnable():
                self.enableErs.set_state(True, False)
            else:
                self.enableErs.set_state(False, False)
            if self.util.isParentIpProfilerEnable():
                self.enableIpProfiler.set_state(True, False)
            else:
                self.enableIpProfiler.set_state(False, False)
        else:
            if self.conf.getEnableErs() == 1:
                self.enableErs.set_state(True, False)
            else:
                self.enableErs.set_state(False, False)
            
            if self.conf.getEnableIpProfiler() == 1:
                self.enableIpProfiler.set_state(True, False)
            else:
                self.enableIpProfiler.set_state(False, False)
            
        self.portInput.set_edit_text(str(self.conf.getFoxhunterListenPort()))
        
        super(InstallSenderFilteringPage, self).drawPage()
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        elif (key == " " or key == "enter") and (self.conf.getFreshInstall() == 0):
            self.logger.debug("InstallSenderFilteringPage:: received space keypress,ignore because it is child")
            self.updater.append((resource.PALETTE_BODY_WARN, self.res.getWording("msg.warnChangeSenderFilteringConfigOnChild")))
        else:
            return super(InstallSenderFilteringPage, self).keypress(size, key)
    
    class SenderFilteringPageUpdater(step.StatusUpdater):
        """
        The status updater for sender filtering page.
        """
        def __init__(self, text):
            self.text = text
        
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY_ERR_FOCUS, msg))
    
    def goNextPage(self):
        self.errorMsg.set_text("")
        if self.enableErs.get_state():
                self.conf.setEnableErs(1)
        else:
            self.conf.setEnableErs(0)
            
        if self.enableIpProfiler.get_state():
            self.conf.setEnableIpProfiler(1)
        else:
            self.conf.setEnableIpProfiler(0)
        if(int(self.portInput.value()) <=0 ):
            self.portInput.set_edit_text(str(self.conf.getFoxhunterListenPort()))
        self.conf.setFoxhunterListenPort(int(self.portInput.value()))
        
        if self.conf.getEnableIpProfiler() or self.conf.getEnableErs():
            ret = self.step.checkFoxHunterPort(self.updater)
            if ret != 0:
                # Prepend the error label.
                self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                        (resource.PALETTE_BODY_ERR_FOCUS, self.errorMsg.text)])
                return -1
        
        return super(InstallSenderFilteringPage, self).goNextPage()

class InstallPreCheckPage(InstallerPage):
    """
    The page shows pre-installation check result, including:
     - Installed imsx rpm.
     - Network interface status.
     - Ports available or not.
     - Available memory.
     - IMSx required users and groups.
     - IMSx dependencies
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
    
        self.rpmStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.rpmLabel = urwid.Text(self.res.getWording("txt.precheck.item.rpm"))
        self.rpmErr = urwid.Text("")
        self.rpmUpdater = self.PreCheckPageUpdater(self.rpmErr)
        self.netStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.netLabel = urwid.Text(self.res.getWording("txt.precheck.item.network"))
        self.netErr = urwid.Text("")
        self.netUpdater = self.PreCheckPageUpdater(self.netErr)
        self.portStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.portLabel = urwid.Text(self.res.getWording("txt.precheck.item.ports"))
        self.portErr = urwid.Text("")
        self.portUpdater = self.PreCheckPageUpdater(self.portErr)
        self.memStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.memLabel = urwid.Text(self.res.getWording("txt.precheck.item.mem"))
        self.memErr = urwid.Text("")
        self.memUpdater = self.PreCheckPageUpdater(self.memErr)
        self.usergroupStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.usergroupLabel = urwid.Text(self.res.getWording("txt.precheck.item.usergroup"))
        self.usergroupErr = urwid.Text("")
        self.usergroupUpdater = self.PreCheckPageUpdater(self.usergroupErr)
        self.folderStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.folderLabel = urwid.Text(self.res.getWording("txt.precheck.item.installFolder"))
        self.folderErr = urwid.Text("")
        self.folderUpdater = self.PreCheckPageUpdater(self.folderErr)
        self.depsStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.depsLabel = urwid.Text(self.res.getWording("txt.precheck.item.deps"))
        self.depsErr = urwid.Text("")
        self.depsUpdater = self.PreCheckPageUpdater(self.depsErr)
        statusColumnLen = 15
        itemsTable = urwid.Pile([urwid.Columns([(statusColumnLen, self.rpmStatus), self.rpmLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.rpmErr]),
                                 urwid.Columns([(statusColumnLen, self.netStatus), self.netLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.netErr]),
                                 urwid.Columns([(statusColumnLen, self.portStatus), self.portLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.portErr]),
                                 urwid.Columns([(statusColumnLen, self.memStatus), self.memLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.memErr]),
                                 urwid.Columns([(statusColumnLen, self.usergroupStatus), self.usergroupLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.usergroupErr]),
                                 urwid.Columns([(statusColumnLen, self.folderStatus), self.folderLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.folderErr]),
                                 urwid.Columns([(statusColumnLen, self.depsStatus), self.depsLabel]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), self.depsErr]),
                                 ])
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.precheck.title")))
        desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.desc")))
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      urwid.Padding(itemsTable, align="left", width=("relative", 80))
                      ]
        
        # PreCheck status:
        # 0 - Check not started
        # 1 - Checking
        # 2 - Check complete and can continue.
        # 3 - Check complete but cannot continue.
        self.checkStatus = 0
        
        super(InstallPreCheckPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame)
    
    def doCheck(self):
        """
        Perform the check.
        """
        if self.checkStatus == 1:
            # Check cannot run simultaneously.
            return
        else:
            self.checkStatus = 1
            # Reset UI to fresh status.
            self.rpmStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.rpmLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.rpm")))
            self.rpmErr.set_text((resource.PALETTE_BODY, ""))
            self.netStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.netLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.network")))
            self.netErr.set_text((resource.PALETTE_BODY, ""))
            self.portStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.portLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.ports")))
            self.portErr.set_text((resource.PALETTE_BODY, ""))
            self.memStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.memLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.mem")))
            self.memErr.set_text((resource.PALETTE_BODY, ""))
            self.usergroupStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.usergroupLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.usergroup")))
            self.usergroupErr.set_text((resource.PALETTE_BODY, ""))
            self.folderStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.folderLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.installFolder")))
            self.folderErr.set_text((resource.PALETTE_BODY, ""))
            self.depsStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.depsLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.deps")))
            self.depsErr.set_text((resource.PALETTE_BODY, ""))

            # Check installed rpm.
            self.rpmStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkRpmForInstallation(self.rpmUpdater)
            if ret == 0:
                self.rpmStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.rpmStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return
            
            # Check network.
            self.netStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkIcpStatus(self.netUpdater)
            if ret == 0:
                self.netStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.netStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return
            
            # Check ports required by IMSx.
            self.portStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkAvailablePorts(self.portUpdater)
            if ret == 0:
                self.portStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                # Show warning about conflict ports, installation can continue because the user can edit ports 
                # used by IMSx after installation.
                self.portStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))
            
            # Check memory.
            self.memStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            memRet = self.step.checkAvailableMemSize(self.memUpdater)
            diskRet = self.step.checkTotalDiskSize(self.memUpdater)
            if memRet == 0 and diskRet == 0:
                self.memStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.memStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))
            
            # Check user and group.
            self.usergroupStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkUsersAndGroups(self.usergroupUpdater)
            if ret == 0:
                self.usergroupStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.usergroupStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))
            
            # Check whether the installation folder exits.
            self.folderStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkInstallFolderContent(self.folderUpdater)
            if ret == 0:
                self.folderStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.folderStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))
            
            # Check dependencies.
            self.depsStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkDependencies(self.depsUpdater)
            if ret == 0:
                self.depsStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.depsStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return
            
            # Check complete and can continue
            self.checkStatus = 2
    
    def enterPage(self):
        # Always do check when entering this page from other pages.
        self.checkStatus = 0
        super(InstallPreCheckPage, self).enterPage()
        self.doCheck()
        # Draw the page to reflect check result.
        self.drawPage()
    
    def drawPage(self):
        if self.checkStatus == 2:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.pass"))
        elif self.checkStatus == 3:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.fail"))
        else:
            self.frame.setFooterText("")
        super(InstallPreCheckPage, self).drawPage()
    
    def keypress(self, size, key):
        if key == "f12":
            if self.checkStatus == 2:
                self.goNextPage()
            elif self.checkStatus == 3:
                self.enterPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(InstallPreCheckPage, self).keypress(size, key)

    class PreCheckPageUpdater(step.StatusUpdater):
        """
        The status updater for precheck page, this updater doesn't clear the 
        original text, but append to the text directly. 
        """
        def __init__(self, text):
            self.text = text
            
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY ,self.text.text + msg + "\n"))
    
class InstallCopyPage(InstallerPage):
    """
    This page is to copy files and initialize IMSx.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.steps = None
        # Use a thread to perform copy files action.
        self.copyThd = threading.Thread(target=self.copyThreadMain)
        
        self.title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.copy.title")))
        self.desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.copy.desc")))
        self.progressBar = urwid.ProgressBar(resource.PALETTE_PROGRESS_INCOMPLETE, resource.PALETTE_PROGRESS_COMPLETE)
        innerFrameHead = urwid.Pile([urwid.Divider(),
                                     self.title,
                                     urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                                     self.desc,
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Padding(self.progressBar, width=("relative", 80)),
                                     urwid.Divider()])
        self.progressTextBox = urwid.ListBox(urwid.SimpleFocusListWalker([]))
        # Use filler to limit the height of progress text. 
        innerFrame = urwid.Frame(urwid.Filler(self.progressTextBox, valign="top", height=10), header=innerFrameHead)
        self.updater = self.CopyPageUpdater(self.progressTextBox)
        super(InstallCopyPage, self).__init__(innerFrame, frame)
    
    def keypress(self, size, key):
        # Handles exit only after copy thread exits.
        if not self.copyThd.isAlive() and (key == "ctrl x" or key == "f12"):
            raise urwid.ExitMainLoop()
        else:
            return super(InstallCopyPage, self).keypress(size, key)
    
    class CopyPageUpdater(step.StatusUpdater):
        """
        The status updater for copy page. This updater appends content to original
        contents and must move the text to the bottom.
        """
        def __init__(self, listBoxWidget):
            self.listBoxWidget = listBoxWidget
        
        def append(self, msg):
            self.listBoxWidget.body.append(urwid.Text(msg))
            self.listBoxWidget.set_focus(len(self.listBoxWidget.body) - 1)
    
    def enterPage(self):
        super(InstallCopyPage, self).enterPage()
        self.frame.setFooterText("")
        self.copyThd.start()
    
    def setCopySteps(self, steps):
        """
        Set the steps needs to be done in this page.
        """
        self.steps = list(steps)
    
    def doCopy(self):
        """
        Do copy and configure IMSx actions.
        
        return: 0 if successful, -1 if failed.
        """
        stepNo = -1
        interrupted = False
        for oneStep in self.steps:
            stepNo += 1
            self.logger.debug("Running step %d: %s..." %(stepNo, oneStep.__class__.__name__))
            if oneStep.apply() != 0:
                self.logger.debug("Fail to running step %d." %(stepNo))
                if not oneStep.isCritical:
                    self.logger.debug("This step is not critical, continue installation.")
                    continue
                else:
                    self.logger.debug("This step is critical, installation cannot continue.")
                    interrupted = True
                    break
            else:
                self.setCurrentProgress(oneStep)
        
        # If the installation doesn't finish successfully, need to rollback the changes.
        if interrupted:
            self.logger.debug("Installation failed in the middle, begin to rollback changes.")
            step.ctlPostgreSQL("stop", self.conf)
            self.conf.setRemoveData(1)
            while stepNo >= 0:
                oneStep = self.steps[stepNo]
                self.logger.debug("Rolling back step %d: %s..." %(stepNo, oneStep.__class__.__name__))
                # Even if one step fails to rollback, still continue the whole rollback process.
                if oneStep.rollback() != 0:
                    self.logger.debug("Fail to rollback step %d." %(stepNo))
                stepNo -= 1
            return -1
        else:
            # The installation completed with no error.
            self.progressBar.set_completion(self.progressBar.done)
            self.frame.complete = True
            return 0
    
    def copyThreadMain(self):
        """
        The main entry of copy thread.
        """
        ret = self.doCopy()
        if ret == 0:
            self.title.set_text((resource.PALETTE_TITLE, self.res.getWording("txt.copy.title.complete")))
            self.desc.set_text(u"")
            msg = [self.res.getWording("txt.copy.ok.begin"), "\n\n"]
            if self.conf.getFreshInstall() == 1:
                # Get all ip address of this machine to generate adminUI address.
                for item in self.util.getNicInfo().values():
                    msg.append("https://%s:8445\n" %(item[0]))
            else:
                # Use the parent IP.
                msg.append("https://%s:8445\n" %(self.conf.getParentAddress()))
            msg.append("\n")
            msg.append(self.res.getWording("txt.copy.ok.end"))
            # Special handling, need to clear the installation progress messages
            # and show finish message.
            del self.progressTextBox.body[:]
            self.progressTextBox.body.append(urwid.Text((resource.PALETTE_BODY, "".join(msg))))
        else:
            self.updater.append((resource.PALETTE_BODY_ERR, self.res.getWording("msg.installFail")))
        self.frame.setFooterText(self.res.getWording("txt.copy.footer"))
    
    def setCurrentProgress(self, aStep):
        """
        Increase and set the progress bar according to the step just executed.
        """
        currentProgress = self.progressBar.current
        if isinstance(aStep, step.SilentInstallUsersAndGroups):
            currentProgress += 1
        elif isinstance(aStep, step.SilentInstallRpms):
            currentProgress += 37
        elif isinstance(aStep, step.SilentInstallReplaceConstants):
            currentProgress += 1
        elif isinstance(aStep, step.SilentInstallDatabase):
            currentProgress += 39
        elif isinstance(aStep, step.SilentInstallRegisterDevice):
            currentProgress += 5
        elif isinstance(aStep, step.SilentInstallService):
            currentProgress += 1
        elif isinstance(aStep, step.SilentInstallCron):
            currentProgress += 1
        elif isinstance(aStep, step.SilentInstallCleanup):
            currentProgress += 1
        elif isinstance(aStep, step.SilentInstallStartApp):
            currentProgress += 2
        elif isinstance(aStep, step.SilentInstallImportDefAuComp):
            currentProgress += 7

        if currentProgress > self.progressBar.done:
            currentProgress = self.progressBar.done
        self.progressBar.set_completion(currentProgress)

class UninstallWelcomePage(InstallerPage):
    """
    This is page is the first page of the uninstallation program.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.uninstallWelcome.title")))
        desc = urwid.Text(self.res.getWording("txt.uninstallWelcome.desc"))
        cont = urwid.Text([self.res.getWording("txt.uninstallWelcome.continue.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.uninstallWelcome.continue.mid")),
                           self.res.getWording("txt.uninstallWelcome.continue.end")])
        canc = urwid.Text([self.res.getWording("txt.uninstallWelcome.cancel.begin"),
                           (resource.PALETTE_BODY_EM, self.res.getWording("txt.uninstallWelcome.cancel.mid")),
                           self.res.getWording("txt.uninstallWelcome.cancel.end")])
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      cont,
                      urwid.Divider(),
                      canc]
        content = urwid.ListBox(urwid.SimpleListWalker(widgetList))
        super(UninstallWelcomePage, self).__init__(content, frame, mode=1)

    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.uninstallWelcome.footer"))
        super(UninstallWelcomePage, self).drawPage()
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UninstallWelcomePage, self).keypress(size, key)
        
class UninstallOptionPage(InstallerPage):
    """
    This page provides the user uninstallation options.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.uninstallOption.title")))
        removeDataDesc = urwid.Text(self.res.getWording("txt.uninstallOption.removeData.desc"))
        removeDataDesc = urwid.Padding(removeDataDesc, width=("relative", 80))
        removeDataBtns = []
        self.removeDataYesBtn = urwid.RadioButton(removeDataBtns, self.res.getWording("txt.uninstallOption.removeData.yes"), on_state_change=self.removeDataBtnChange)
        self.removeDataNoBtn = urwid.RadioButton(removeDataBtns, self.res.getWording("txt.uninstallOption.removeData.no"))
        removeDataBtnLen = 10
        removeDataYes = urwid.Padding(urwid.AttrMap(self.removeDataYesBtn, resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=removeDataBtnLen)
        removeDataNo = urwid.Padding(urwid.AttrMap(self.removeDataNoBtn, resource.PALETTE_BODY, focus_map=resource.PALETTE_BODY_FOCUS), width=removeDataBtnLen)
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      removeDataDesc,
                      removeDataYes,
                      removeDataNo
                      ]
        super(UninstallOptionPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame, mode=1)
    
    def removeDataBtnChange(self, btn, state):
        if btn == self.removeDataYesBtn and state:
            self.conf.removeData = 1
            self.logger.debug("Remove data on.")
        else:
            self.conf.removeData = 0
            self.logger.debug("Remove data off.")
    
    def drawPage(self):
        super(UninstallOptionPage, self).drawPage()
        if self.conf.removeData == 1:
            self.removeDataYesBtn.toggle_state()
        else:
            self.removeDataNoBtn.toggle_state()
        self.frame.setFooterText(self.res.getWording("txt.uninstallOption.footer"))
    
    def keypress(self, size, key):
        if key == "f12":
            self.goNextPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UninstallOptionPage, self).keypress(size, key)

class UninstallRemovePage(InstallerPage):
    """
    This page is to remove IMSx files from the machine.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        # Use a thread to perform removing files.
        self.removeThd = threading.Thread(target=self.removeThreadMain)
        
        self.title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.uninstallRemove.title")))
        self.desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.uninstallRemove.desc")))
        self.progressBar = urwid.ProgressBar(resource.PALETTE_PROGRESS_INCOMPLETE, resource.PALETTE_PROGRESS_COMPLETE)
        innerFrameHead = urwid.Pile([urwid.Divider(),
                                     self.title,
                                     urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                                     self.desc,
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Padding(self.progressBar, right=20),
                                     urwid.Divider()])
        self.progressText = urwid.Text("")
        self.progressTextBox = urwid.ListBox(urwid.SimpleFocusListWalker([self.progressText]))
        # Use filler to limit the height of progress text. 
        innerFrame = urwid.Frame(urwid.Filler(self.progressTextBox, valign="top", height=10), header=innerFrameHead)
        self.updater = self.UninstallRemovePageUpdater(self.progressText, self.progressTextBox)
        super(UninstallRemovePage, self).__init__(innerFrame, frame)
    
    def keypress(self, size, key):
        if not self.removeThd.isAlive() and (key == "ctrl x" or key == "f12"):
            raise urwid.ExitMainLoop()
        else:
            return super(UninstallRemovePage, self).keypress(size, key)
    
    class UninstallRemovePageUpdater(step.StatusUpdater):
        """
        The status updater for uninstallation remove page. This updater appends
        the message to the display panel and set the focus to bottom.
        """
        def __init__(self, textWidget, listBoxWidget):
            self.textWidget = textWidget
            self.listBoxWidget = listBoxWidget
        
        def append(self, msg):
            self.textWidget.set_text(self.textWidget.text + msg + "\n")
            self.listBoxWidget.set_focus_valign("bottom")
    
    def addCurrentProgress(self, incrementCompletion):
        """
        Add the completion.
        """
        currentCompletion = self.progressBar.current
        currentCompletion += incrementCompletion
        if currentCompletion > self.progressBar.done:
            currentCompletion = self.progressBar.done
        self.progressBar.set_completion(currentCompletion)
    
    def removeThreadMain(self):
        ret = self.doRemove()
        if ret == 0:
            self.title.set_text((resource.PALETTE_BODY, self.res.getWording("txt.uninstallRemove.title.complete")))
            self.desc.set_text(u"")
            msg = [self.res.getWording("txt.uninstallRemove.ok.begin"), "\n\n"]
            manualRemovePaths = self.conf.getManualRemovePaths() + self.conf.getFailRemovePaths()
            if self.conf.getRemoveData() == 0:
                manualRemovePaths.extend(self.conf.getOptionalRemovePaths())
            for aPath in manualRemovePaths:
                msg.append("%s\n" %(aPath))
            self.progressText.set_text((resource.PALETTE_BODY, "".join(msg)))
        else:
            self.progressText.set_text([self.progressText.text, (resource.PALETTE_BODY_ERR, self.res.getWording("msg.uninstallFail"))])
        self.frame.setFooterText(self.res.getWording("txt.uninstallRemove.footer"))        
    
    def doRemove(self):
        """
        This function performs the remove files job.
        
        return: 0 if successful, -1 if failed.
        """
        # Currently not consider rollback for uninstallation. 
        oneStep = step.Step(self.conf, self.res, self.logger, self.util, False, self.updater)

        # Check installed rpms, if any one of rpms is not installed, the 
        # uninstall will stop because the install may not be complete.
        ret = oneStep.checkRpmForUninstallation(self.updater)
        if ret != 0:
            return -1
        self.addCurrentProgress(1)
        
        # Uninstall the crontab.
        ret = oneStep.uninstallCron(self.updater)
        if ret != 0:
            return -1
        self.addCurrentProgress(3)
        
        # Stop all services except DB.
        oneStep.stopAppExceptDB()
        self.addCurrentProgress(2)
        
        # Try to unregister the device from admin DB.
        ret = oneStep.unregisterDeviceFromAdminDb(self.updater)
        if ret != 0:
            return -1
        self.addCurrentProgress(26)
        
        # Stop all services.
        oneStep.stopApp()
        self.addCurrentProgress(2)
        
        # For IMSS, revert the postfix settings.
        if self.conf.getProduct() == 0:
            ret = oneStep.removePostfixFoxlibSetting()
            if ret != 0:
                return -1
        
        # Uninstall the rpm.
        ret = oneStep.uninstallImsxRpms(self.updater)
        if ret != 0:
            return -1
        self.addCurrentProgress(28)
        
        # Remove the files.
        ret = oneStep.removeImsxFiles(self.updater)
        if ret != 0:
            return -1
        self.addCurrentProgress(45)
        
        # For IMSS, need to remove imsx related users and groups.
        if self.conf.getProduct() == 0:
            ret = oneStep.removeUsersAndGroups(self.updater)
            if ret != 0:
                return -1
        self.addCurrentProgress(1)
        
        self.logger.debug("TUI uninstallation finished.")
        self.progressBar.set_completion(self.progressBar.done)
        self.frame.complete = True
        return 0
    
    def enterPage(self):
        super(UninstallRemovePage, self).enterPage()
        self.frame.setFooterText("")
        self.removeThd.start()

##########################################################################
#### IMSS upgrade UI
##########################################################################


class UpgradeDatabasePage(InstallerPage):
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)
        self.rpm_status = g_upd_models.rpm_status

        if os.path.exists(Const.DEBUG_FLAG_FILE):
            self.logger.debug("UpgradeDatabasePage --- original db info:")
            Debugger.print_db_info(self.logger, self.conf)

        self.isInternalDB = self.util.isUsingInternalAdminDB(self.rpm_status, force_check=True)
        self.logger.debug("isUsingInternalAdminDB, %s"%str(self.isInternalDB))
        self.logger.debug("isUsingInternalDB,%s"%(str(self.util.isUsingInternalDB(force_check=True))))

        if self.isInternalDB:
            self.conf.setAdminDBType(0)
        else:
            self.conf.setAdminDBType(1)

        if os.path.exists(Const.DEBUG_FLAG_FILE):
            self.logger.debug("UpgradeDatabasePage --- db info after read odbc.ini")
            Debugger.print_db_info(self.logger, self.conf)

        self.title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.database.title")))
        self.descTxt = urwid.Text("")
        self.desc = urwid.Padding(self.descTxt, width=("relative", 80))
        dbTypeBtns = []
        self.dbTypeTxt = urwid.Text("")

        self.infoDiv = urwid.Text((resource.PALETTE_BODY_EM, self.res.getWording("txt.database.divider")))
        self.addrLabel = urwid.Text(self.res.getWording("txt.database.address"))
        self.portLabel = urwid.Text(self.res.getWording("txt.database.port"))
        self.dbNameLabel = urwid.Text(self.res.getWording("txt.database.dbName"))
        self.userLabel = urwid.Text(self.res.getWording("txt.database.user"))
        self.passwdLabel = urwid.Text(self.res.getWording("txt.database.password"))

        # For internal db all info is readonly
        self.addrReadonly = urwid.Text("")
        self.portReadonly = urwid.Text("")
        self.dbNameReadonly = urwid.Text("")
        self.userReadonly = urwid.Text("")
        self.passwdReadonly = urwid.Text("")

        self.addrEditable = urwid.Edit()
        self.portEditable = urwid.IntEdit()
        self.dbNameEdit = urwid.Edit()
        self.userEdit = urwid.Edit()
        self.passwdEdit = urwid.Edit(mask="*")
        labelLen = 20
        EditLen = 32
        tableWidth = labelLen + EditLen + 2
        # This the table for bundle db config.
        self.dbBundleTable = urwid.Pile([
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.addrLabel), (1, urwid.Text(u"")), (EditLen, self.addrReadonly)]),resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.portLabel), (1, urwid.Text(u"")),(EditLen, self.portReadonly)]), resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.dbNameLabel), (1, urwid.Text(u"")),(EditLen, self.dbNameReadonly), (1, urwid.Text(u""))]),resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.userLabel), (1, urwid.Text(u"")), (EditLen, self.userReadonly),(1, urwid.Text(u""))]), resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.passwdLabel), (1, urwid.Text(u"")),(EditLen, self.passwdReadonly), (1, urwid.Text(u""))]),resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),])
        self.dbBundleTable = urwid.Padding(self.dbBundleTable, left=4)
        # This is the table for remote db config.
        self.dbRemoteTable = urwid.Pile([
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.addrLabel), (1, urwid.Text(u"[")), (EditLen, self.addrEditable), (1, urwid.Text(u"]"))]),resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.portLabel), (1, urwid.Text(u"[")),(EditLen, self.portEditable), (1, urwid.Text(u"]"))]),resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.dbNameLabel), (1, urwid.Text(u"[")),(EditLen, self.dbNameEdit), (1, urwid.Text(u"]"))]),resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.userLabel), (1, urwid.Text(u"[")), (EditLen, self.userEdit),(1, urwid.Text(u"]"))]), resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        urwid.Padding(urwid.AttrMap(urwid.Columns([(labelLen, self.passwdLabel), (1, urwid.Text(u"[")),(EditLen, self.passwdEdit), (1, urwid.Text(u"]"))]),resource.PALETTE_BODY,resource.PALETTE_BODY_FOCUS), width=tableWidth),
        ])
        self.dbRemoteTable = urwid.Padding(self.dbRemoteTable, left=4)
        self.errorMsg = urwid.Text("")
        btnWidth = 30

        widgetList = []
        if self.util.isParent(self.rpm_status) == 0:
            self.logger.debug("Current server is parent component.")
            if self.isInternalDB:
                self.descTxt.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.db.desc.internal")))
                self.dbTypeTxt.set_text(self.res.getWording("txt.database.useBundle"))
                widgetList = [urwid.Divider(),
                              self.title,
                              urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                              self.desc,
                              urwid.Divider(),
                              urwid.Padding(
                                  urwid.AttrMap(self.dbTypeTxt, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS),
                                  width=btnWidth),
                              urwid.Divider(),
                              urwid.Padding(self.infoDiv, left=4),
                              self.dbBundleTable,
                              urwid.Divider(),
                              self.errorMsg]
            else:
                # fixme
                self.descTxt.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.db.desc.external")))
                self.dbTypeTxt.set_text(self.res.getWording("txt.database.useRemote"))
                widgetList = [urwid.Divider(),
                              self.title,
                              urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                              self.desc,
                              urwid.Divider(),
                              urwid.Padding(
                                  urwid.AttrMap(self.dbTypeTxt, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS),
                                  width=btnWidth),
                              urwid.Divider(),
                              urwid.Padding(self.infoDiv, left=4),
                              self.dbRemoteTable,
                              urwid.Divider(),
                              self.errorMsg]
        else:
            self.logger.debug("Current server is child component.")
            # self.descTxt.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.db.desc.child")))
            # widgetList = [urwid.Divider(),
            #               self.title,
            #               urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
            #               self.desc,
            #               urwid.Divider(),
            #               self.errorMsg]
            # user should think if he need change config
            self.descTxt.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.db.desc.childdb")))
            self.dbTypeTxt.set_text(self.res.getWording("txt.database.childdb.title"))
            widgetList = [urwid.Divider(),
                          self.title,
                          urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                          self.desc,
                          urwid.Divider(),
                          urwid.Padding(
                              urwid.AttrMap(self.dbTypeTxt, resource.PALETTE_BODY, resource.PALETTE_BODY_FOCUS),
                              width=btnWidth),
                          urwid.Divider(),
                          urwid.Padding(self.infoDiv, left=4),
                          self.dbRemoteTable,
                          urwid.Divider(),
                          self.errorMsg]


        self.walker = urwid.SimpleFocusListWalker(widgetList)
        self.updater = self.DatabasePageUpdater(self.errorMsg)
        super(UpgradeDatabasePage, self).__init__(urwid.ListBox(self.walker), frame, mode=2)


    def enterPage(self):
        # Clear the error message.
        self.errorMsg.set_text("")
        super(UpgradeDatabasePage, self).enterPage()

    def drawPage(self):
        self.frame.setFooterText(self.res.getWording("txt.database.footer"))
        if self.isInternalDB:
            self.addrReadonly.set_text(self.conf.getAdminDBAddress())
            self.portReadonly.set_text(str(self.conf.getAdminDBPort()))
            self.dbNameReadonly.set_text(self.conf.getAdminDBName())
            self.userReadonly.set_text(self.conf.getAdminDBUsername())
            self.passwdReadonly.set_text(len(self.util.decryptString(self.conf.getAdminDBPassword()))*"*")
        else:
            self.addrEditable.set_edit_text(self.conf.getAdminDBAddress())
            self.portEditable.set_edit_text(str(self.conf.getAdminDBPort()))
            self.dbNameEdit.set_edit_text(self.conf.getAdminDBName())
            self.userEdit.set_edit_text(self.conf.getAdminDBUsername())
            self.passwdEdit.set_edit_text(self.util.decryptString(self.conf.getAdminDBPassword()))
        super(UpgradeDatabasePage, self).drawPage()

    def doDBCheck(self):
        ret = 0

        (rc,re) = self.util.isDatabaseConnectable(self.addrEditable.edit_text, self.portEditable.value(),self.userEdit.edit_text, self.passwdEdit.edit_text)
        if rc != 0:
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS,
                                     self.res.getWording("txt.upgrade.db.check.noconn"))])
            self.logger.error("Exception when do DB check: %s"%(str(re)))
            ret = -1

        #if db is connectable, do not check the DB version
        if ret==0 and self.util.isDbNewVersion(self.addrEditable.edit_text, self.portEditable.value(),
                              self.userEdit.edit_text, self.passwdEdit.edit_text) != 0:
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS, self.res.getWording("txt.upgrade.db.check.errmsg"))])
            ret = -2

        if ret == 0 and self.util.checkDbUserRole(address=self.addrEditable.edit_text, port=self.portEditable.value(),
                                  username=self.userEdit.edit_text, password=self.passwdEdit.edit_text) != 0:
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                        (resource.PALETTE_BODY_ERR_FOCUS,self.res.getWording("txt.upgrade.db.check.rolerror"))])
            ret = -3

        if ret==0 and self.util.isDbHasDumpAndRestore(self.addrEditable.edit_text, self.portEditable.value(),
                              self.userEdit.edit_text, self.passwdEdit.edit_text,self.dbNameEdit.edit_text) != 0:
            self.errorMsg.set_text([(resource.PALETTE_BODY_ERR_EM_FOCUS, self.res.getWording("txt.errorLabel")),
                                    (resource.PALETTE_BODY_ERR_FOCUS, self.res.getWording("txt.upgrade.db.schema.errmsg"))])
            ret = -4

        if ret != 0:
            self.logger.debug("Remote DB error, address:%s:%s usr:%s dbname:%s check result:%s" % (
            self.addrEditable.edit_text, str(self.portEditable.value()),
            self.userEdit.edit_text, self.dbNameEdit.edit_text,str(ret)))

        #reset adminDB info
        self.conf.setAdminDBAddress(self.addrEditable.edit_text)
        self.conf.setAdminDBPort(self.portEditable.edit_text)
        self.conf.setAdminDBUsername(self.userEdit.edit_text)
        self.conf.setAdminDBPassword(self.util.encryptString(self.passwdEdit.edit_text))
        self.conf.setAdminDBName(self.dbNameEdit.edit_text)
        return ret

    def keypress(self, size, key):
        if key == "f12":
            #if remote DB then check DB version. if not 9.2, can't goto next step
            if self.isInternalDB:
                self.goNextPage()
            #should check db role permission
            elif not self.isInternalDB and self.doDBCheck() == 0 :
                self.goNextPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UpgradeDatabasePage, self).keypress(size, key)

    class DatabasePageUpdater(step.StatusUpdater):
        """
        The status updater for database page.
        """
        def __init__(self, text):
            self.text = text

        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY_ERR_FOCUS, msg))

class UpgradePreCheckPage(InstallerPage):
    """
    The page shows pre-installation check result, including:
     - Installed imsx rpm.
     - Network interface status.
     - Ports available or not.
     - Available memory.
     - IMSx required users and groups.
    """
    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)

        self.netStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.netLable = urwid.Text(self.res.getWording("txt.precheck.item.network"))
        self.netErr = urwid.Text("")
        self.netUpdater = self.PreCheckPageUpdater(self.netErr)

        self.memStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.memLable = urwid.Text(self.res.getWording("txt.precheck.item.mem"))
        self.memErr = urwid.Text("")
        self.memUpdater = self.PreCheckPageUpdater(self.memErr)

        # self.folderStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        # self.folderLable = urwid.Text(self.res.getWording("txt.upgrade.check.free.disk"))
        # self.folderErr = urwid.Text("")
        # self.folderUpdater = self.PreCheckPageUpdater(self.folderErr)

        self.backupFolderStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.backupFolderLable = urwid.Text(self.res.getWording("txt.upgrade.check.free.disk"))
        self.backupFolderErr = urwid.Text("")
        self.backupFolderUpdater = self.PreCheckPageUpdater(self.backupFolderErr)

        self.versionStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.versionLable = urwid.Text(self.res.getWording("txt.upgrade.check.check"))
        self.versionErr = urwid.Text("")
        self.versionUpdater = self.PreCheckPageUpdater(self.versionErr)

        self.usergroupStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.usergroupLabel = urwid.Text(self.res.getWording("txt.precheck.item.usergroup"))
        self.usergroupErr = urwid.Text("")
        self.usergroupUpdater = self.PreCheckPageUpdater(self.usergroupErr)

        self.osStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.osLable = urwid.Text(self.res.getWording("txt.upgrade.check.os"))
        self.osErr = urwid.Text("")
        self.osUpdater = self.PreCheckPageUpdater(self.osErr)

        self.depsStatus = urwid.Text(self.res.getWording("txt.precheck.status.notStart"))
        self.depsLabel = urwid.Text(self.res.getWording("txt.precheck.item.deps"))
        self.depsErr = urwid.Text("")
        self.depsUpdater = self.PreCheckPageUpdater(self.depsErr)

        self.rpm_status = g_upd_models.rpm_status
        self.version_checker = g_upd_models.version_checker
        self.disk_checker = g_upd_models.disk_checker


        statusColumnLen = 15
        lableCoulmnLen = 64
        itemsTable = urwid.Pile([
                                 urwid.Columns([(statusColumnLen, self.netStatus), (lableCoulmnLen, self.netLable)]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.netErr)]),
                                 urwid.Columns([(statusColumnLen, self.memStatus), (lableCoulmnLen, self.memLable)]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.memErr)]),
                                 urwid.Columns([(statusColumnLen, self.usergroupStatus), (lableCoulmnLen, self.usergroupLabel)]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.usergroupErr)]),
                                 # urwid.Columns([(statusColumnLen, self.folderStatus), (lableCoulmnLen, self.folderLable)]),
                                 # urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.folderErr)]),
                                 urwid.Columns([(statusColumnLen, self.backupFolderStatus), (lableCoulmnLen, self.backupFolderLable)]),
                                 urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.backupFolderErr)]),
                                 urwid.Columns([(statusColumnLen, self.versionStatus), (lableCoulmnLen, self.versionLable)]),
                                urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.versionErr)]),
                                urwid.Columns([(statusColumnLen, self.osStatus), (lableCoulmnLen, self.osLable)]),
                                urwid.Columns([(statusColumnLen, urwid.Divider()), (lableCoulmnLen, self.osErr)]),
                                urwid.Columns([(statusColumnLen, self.depsStatus), self.depsLabel]),
                                urwid.Columns([(statusColumnLen, urwid.Divider()), self.depsErr])
                                ])
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.precheck.title")))
        desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.desc")))
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      itemsTable
                      ]

        # PreCheck status:
        # 0 - Check not started
        # 1 - Checking
        # 2 - Check complete and can continue.
        # 3 - Check complete but cannot continue.
        self.checkStatus = 0
        super(UpgradePreCheckPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame, mode=2)

    def doCheck(self):
        """
        Perform the check.
        """
        if self.checkStatus == 1:
            # Check cannot run simultaneously.
            return
        else:
            self.checkStatus = 1
            # Reset UI to fresh status.
            self.netStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.netLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.network")))
            self.netErr.set_text((resource.PALETTE_BODY, ""))
            self.memStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.memLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.mem")))
            self.memErr.set_text((resource.PALETTE_BODY, ""))

            self.usergroupStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.usergroupLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.usergroup")))
            self.usergroupErr.set_text((resource.PALETTE_BODY, ""))

            # self.folderStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            # self.folderLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.check.free.disk")))
            # self.folderErr.set_text((resource.PALETTE_BODY, ""))

            self.backupFolderStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.backupFolderLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.check.free.backupfolder")))
            self.backupFolderErr.set_text((resource.PALETTE_BODY, ""))

            self.versionStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.versionLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.check.build")))
            self.versionErr.set_text((resource.PALETTE_BODY, ""))
            self.osStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.osLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.check.os")))
            self.osErr.set_text((resource.PALETTE_BODY, ""))

            self.depsStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.status.notStart")))
            self.depsLabel.set_text((resource.PALETTE_BODY, self.res.getWording("txt.precheck.item.deps")))
            self.depsErr.set_text((resource.PALETTE_BODY, ""))

            # Check network.
            self.netStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkIcpStatus(self.netUpdater)
            if ret == 0:
                self.netStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.netStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return

            # Check memory.
            self.memStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkAvailableMemSize(self.memUpdater)
            if ret == 0:
                self.memStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.memStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))

            # Check user and group.
            self.usergroupStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkUsersAndGroups(self.usergroupUpdater)
            if ret == 0:
                self.usergroupStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.usergroupStatus.set_text((resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))

            # Check whether the installation folder exits.
            # self.folderStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            # ret = self.step.checkAvailableDiskSize(self.folderUpdater)
            # if ret == 0:
            #     self.folderStatus.set_text(
            #         (resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            # else:
            #     self.folderStatus.set_text(
            #         (resource.PALETTE_BODY_WARN, self.res.getWording("txt.precheck.status.warn")))

            #backup folder free size check
            self.backupFolderStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.disk_checker.checkBackupFolderStatus(self.backupFolderUpdater)
            if ret == 0:
                self.backupFolderStatus.set_text(
                    (resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.backupFolderStatus.set_text(
                    (resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return

            #check whether the old version build num is OK
            self.versionStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.version_checker.checkImssAppVersion(self.versionUpdater)
            if ret == 0:
                self.versionStatus.set_text(
                    (resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.versionUpdater.append( self.res.getWording("txt.upgrade.check.build.errmsg"))
                self.versionStatus.set_text(
                    (resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return

            #only support Redhat5,6,7
            self.osStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.version_checker.checkOSVersion(self.osUpdater)
            if ret == 0:
                self.osStatus.set_text(
                    (resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.osStatus.set_text(
                    (resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return

            # Check dependencies.
            self.depsStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.precheck.status.checking")))
            ret = self.step.checkDependencies(self.depsUpdater)
            if ret == 0:
                self.depsStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.precheck.status.passed")))
            else:
                self.depsStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.precheck.status.failed")))
                self.checkStatus = 3
                return

            # Check complete and can continue
            self.checkStatus = 2


    def enterPage(self):
        # Always do check when entering this page from other pages.
        self.checkStatus = 0
        super(UpgradePreCheckPage, self).enterPage()
        self.doCheck()
        # Draw the page to reflect check result.
        self.drawPage()

    def drawPage(self):
        if self.checkStatus == 2:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.pass"))
        elif self.checkStatus == 3:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.fail"))
        else:
            self.frame.setFooterText("")
        super(UpgradePreCheckPage, self).drawPage()

    def keypress(self, size, key):
        if key == "f12":
            if self.checkStatus == 2:
                self.goNextPage()
            elif self.checkStatus == 3:
                self.enterPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UpgradePreCheckPage, self).keypress(size, key)

    class PreCheckPageUpdater(step.StatusUpdater):
        """
        The status updater for precheck page, this updater doesn't clear the
        original text, but append to the text directly.
        """
        def __init__(self, text):
            self.text = text
        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY, self.text.text + msg + "\n"))

class UpgradeOldVersionPage(InstallerPage):
    """show installed components when upgrading
    imsseuq
    imss
    imsscctrl
    nrs
    ipprofiler
    """
    class PreCheckPageUpdater(step.StatusUpdater):
        """
        The status updater for precheck page, this updater doesn't clear the
        original text, but append to the text directly.
        """

        def __init__(self, text):
            self.text = text

        def append(self, msg):
            self.text.set_text((resource.PALETTE_BODY, self.text.text + msg + "\n"))

    def __init__(self, conf, res, logger, util, frame):
        self.rpm_status = g_upd_models.rpm_status
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.step = step.Step(conf, res, logger, util, True, None)

        self.imssLable = urwid.Text(self.res.getWording("txt.upgrade.installed.imss"))
        self.imssStatus = urwid.Text(self.res.getWording("txt.upgrade.install.check.status"))

        self.scanLable = urwid.Text(self.res.getWording("txt.upgrade.installed.scan"))
        self.scanStatus = urwid.Text(self.res.getWording("txt.upgrade.install.check.status"))

        self.euqLable = urwid.Text(self.res.getWording("txt.upgrade.installed.euq"))
        self.euqStatus = urwid.Text(self.res.getWording("txt.upgrade.install.check.status"))

        self.nrsLable = urwid.Text(self.res.getWording("txt.upgrade.installed.nrs"))
        self.nrsStatus = urwid.Text(self.res.getWording("txt.upgrade.install.check.status"))

        self.ippLable = urwid.Text(self.res.getWording("txt.upgrade.installed.ipp"))
        self.ippStatus = urwid.Text(self.res.getWording("txt.upgrade.install.check.status"))

        lableCoulmnLen = 24
        statusColumnLen = 48
        itemsTable = urwid.Pile([
                                urwid.Columns([(lableCoulmnLen, self.imssLable), (statusColumnLen, self.imssStatus)]),
                                urwid.Columns([(statusColumnLen + lableCoulmnLen, urwid.Divider())]),
                                urwid.Columns([(lableCoulmnLen, self.scanLable), (statusColumnLen, self.scanStatus)]),
                                urwid.Columns([(statusColumnLen + lableCoulmnLen, urwid.Divider())]),
                                urwid.Columns([(lableCoulmnLen, self.euqLable), (statusColumnLen, self.euqStatus)]),
                                urwid.Columns([(statusColumnLen + lableCoulmnLen, urwid.Divider())]),
                                 urwid.Columns([(lableCoulmnLen, self.nrsLable),(statusColumnLen, self.nrsStatus)]),
                                 urwid.Columns([(statusColumnLen+lableCoulmnLen, urwid.Divider())]),
                                 urwid.Columns([ (lableCoulmnLen, self.ippLable),(statusColumnLen, self.ippStatus)]),
                                 urwid.Columns([(statusColumnLen+lableCoulmnLen, urwid.Divider())]),
                                 ])
        title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.upgrade.installed.title")))
        desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.desc")))
        widgetList = [urwid.Divider(),
                      title,
                      urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=title.pack()[0]),
                      desc,
                      urwid.Divider(),
                      itemsTable
                      ]
        # PreCheck status:
        # 0 - Check not started
        # 1 - Checking
        # 2 - Check complete and can continue.
        # 3 - Check complete but cannot continue.
        self.checkStatus = 0
        super(UpgradeOldVersionPage, self).__init__(urwid.ListBox(urwid.SimpleFocusListWalker(widgetList)), frame, mode=2)

    def getRpmStatus(self, key):
        rpms_status = self.rpm_status.getRpmStatus()
        if not rpms_status.has_key(key):
            return (-1,"")
        if rpms_status[key][0]:
            return (0,rpms_status[key][1])
        else:
            return (-1,"")

    def doCheck(self):
        """
        Perform the check.
        """
        #installed components check are not same with pre check
        #if one components installed. the final result is ok
        self.result = -1
        if self.checkStatus == 1:
            # Check cannot run simultaneously.
            return
        else:
            self.checkStatus = 1
            # Reset UI to fresh status.
            self.imssStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.status.check")))
            self.imssLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.imss")))

            self.scanStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.status.check")))
            self.scanLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.scan")))

            self.euqStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.status.check")))
            self.euqLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.euq")))

            #self.imssErr.set_text((resource.PALETTE_BODY, ""))
            self.nrsStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.status.check")))
            self.nrsLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.nrs")))
            #self.nrsErr.set_text((resource.PALETTE_BODY, ""))
            self.ippStatus.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.status.check")))
            self.ippLable.set_text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.installed.ipp")))

            # Check imss rpm.
            self.imssStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.upgrade.installed.status.checking")))
            (ret,path) = self.getRpmStatus(Const.RPM_CTRL)
            if ret == 0:
                self.imssStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.installed.status.passed")+" "+path))
                self.result = 2
            else:
                self.imssStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.installed.status.failed")))
                if self.result !=2:
                    self.result = 3

            self.scanStatus.set_text( (resource.PALETTE_BODY_EM, self.res.getWording("txt.upgrade.installed.status.checking")))
            (ret, path) = self.getRpmStatus(Const.RPM_IMSS)
            if ret == 0:
                self.scanStatus.set_text((resource.PALETTE_BODY_PASS,self.res.getWording("txt.upgrade.installed.status.passed") + " " + path))
                self.result = 2
            else:
                self.scanStatus.set_text(
                    (resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.installed.status.failed")))
                if self.result !=2:
                    self.result = 3

            self.euqStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.upgrade.installed.status.checking")))
            (ret, path) = self.getRpmStatus(Const.RPM_EUQ)
            if ret == 0:
                self.euqStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.installed.status.passed") + " " + path))
                self.result = 2
            else:
                self.euqStatus.set_text(
                    (resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.installed.status.failed")))
                if self.result !=2:
                    self.result = 3

                # Check nrs rpm.
            self.nrsStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.upgrade.installed.status.checking")))
            (ret, path) = self.getRpmStatus(Const.RPM_NRS)
            if ret == 0:
                self.nrsStatus.set_text((resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.installed.status.passed")+" "+path))
                self.result = 2
            else:
                self.nrsStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.installed.status.failed")))
                if self.result !=2:
                    self.result = 3

            # Check ipp rpm.
            self.ippStatus.set_text((resource.PALETTE_BODY_EM, self.res.getWording("txt.upgrade.installed.status.checking")))
            (ret, path) = self.getRpmStatus(Const.RPM_IPP)
            if ret == 0:
                self.ippStatus.set_text(
                    (resource.PALETTE_BODY_PASS, self.res.getWording("txt.upgrade.installed.status.passed")+" "+path))
                self.result = 2
            else:
                self.ippStatus.set_text((resource.PALETTE_BODY_ERR, self.res.getWording("txt.upgrade.installed.status.failed")))
                if self.result !=2:
                    self.result = 3

            if self.result == 2:
                self.checkStatus = 2
            elif self.result == 3:
                self.checkStatus = 3


    def enterPage(self):
        # Always do check when entering this page from other pages.
        self.checkStatus = 0
        super(UpgradeOldVersionPage, self).enterPage()
        self.doCheck()
        # Draw the page to reflect check result.
        self.drawPage()

    def drawPage(self):
        if self.checkStatus == 2:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.pass"))
        elif self.checkStatus == 3:
            self.frame.setFooterText(self.res.getWording("txt.precheck.footer.fail"))
        else:
            self.frame.setFooterText("")
        super(UpgradeOldVersionPage, self).drawPage()

    def keypress(self, size, key):
        if key == "f12":
            if self.checkStatus == 2:
                self.goNextPage()
            elif self.checkStatus == 3:
                self.enterPage()
        elif key == "esc":
            self.goPrevPage()
        elif key == "ctrl x":
            raise urwid.ExitMainLoop()
        else:
            return super(UpgradeOldVersionPage, self).keypress(size, key)

class UpgradeFrame(urwid.WidgetWrap):
    """The frame of  upgrade, this is meat to organize pages."""
    def __init__(self, conf, res, logger, util, mode=2):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.complete = False
        # The mode is upgrade (3)
        self.mode = mode
        #UI frame work
        self.header = InstallerHeader(self.conf, self.res, self.logger, self.util, urwid.Text(u""))
        self.footer = InstallerFooter(self.conf, self.res, self.logger, self.util, urwid.Text(u""))
        self.wrappedFrame = urwid.Frame(None, header=self.header, footer=self.footer)
        #create upgrade pages
        self.welcomePage = UpgradeWelcomePage(self.conf, self.res, self.logger, self.util, self)
        self.licensePage = InstallLicensePage(self.conf, self.res, self.logger, self.util, self, mode=2)
        self.oldVersionInfo = UpgradeOldVersionPage(self.conf, self.res, self.logger, self.util, self)
        self.preCheckPage = UpgradePreCheckPage(self.conf, self.res, self.logger, self.util, self)
        self.databasePage = UpgradeDatabasePage(self.conf, self.res, self.logger, self.util, self)
        self.copyPage = UpdatingPage(self.conf, self.res, self.logger, self.util, self)
        self.wrappedFrame.body = self.welcomePage
        self.totalStepNum = 6
        # Step starts from 1.
        self.currentStepNo = 1
        super(UpgradeFrame, self).__init__(self.wrappedFrame)
        self.wrappedFrame.body.drawPage()


    def updateProcess(self):
        self.copyPage.updateDotDotDot()

    def setHeaderText(self, text):
        """
        Set text displayed on the header.
        """
        self.header.setText(text)


    def setFooterText(self, text):
        """
        Set text displayed on the footer.
        """
        self.footer.setText(text)


    def setPage(self, page):
        """
        Set the current page.

        page: the page to be the current page.
        """
        self.wrappedFrame.body = page


    def keypress(self, size, key):
        if key == "ctrl x":
            self.logger.debug("The user closes the UI.")
        return super(UpgradeFrame, self).keypress(size, key)


    def switchToPrevPage(self, page):
        """
        Make the frame to show the previous page of the current page.

        return: 0 if it is OK to switch to previous page, -1 if failed.
        """
        prevPage = None
        #Upgrade
        if self.mode == 2:
            if page == self.licensePage:
                prevPage = self.welcomePage
            elif page == self.oldVersionInfo:
                prevPage = self.licensePage
            elif page == self.preCheckPage:
                prevPage = self.oldVersionInfo
            elif page == self.databasePage:
                prevPage = self.preCheckPage
            elif page == self.copyPage:
                prevPage = self.databasePage

        if prevPage:
            self.currentStepNo -= 1
            # need to refresh the whole page
            #because internal and external db have different UI controls
            if  isinstance(prevPage, UpgradeDatabasePage)  :
                self.databasePage = UpgradeDatabasePage(self.conf, self.res, self.logger, self.util, self)
                prevPage = self.databasePage
                self.setPage(prevPage)
                prevPage.enterPage()
            else:
                self.setPage(prevPage)
                prevPage.enterPage()
            return 0
        else:
            return -1


    def switchToNextPage(self, page):
        """
        Make the frame to show the next page of the current page.

        return: 0 if it is OK to switch to next page, -1 if failed.
        """
        nextPage = None
        # Upgrade
        if self.mode == 2:
            if page == self.welcomePage:
                nextPage = self.licensePage
            elif page == self.licensePage:
                nextPage = self.oldVersionInfo
            elif page == self.oldVersionInfo:
                nextPage = self.preCheckPage
            elif page == self.preCheckPage:
                nextPage = self.databasePage
            elif page == self.databasePage:
                nextPage = self.copyPage
        if nextPage:
            self.currentStepNo += 1
            # need to refresh the whole page
            #because internal and external db have different UI controls
            if isinstance(nextPage, UpgradeDatabasePage):
                self.databasePage = UpgradeDatabasePage(self.conf, self.res, self.logger, self.util, self)
                nextPage = self.databasePage
                self.setPage(nextPage)
                nextPage.enterPage()
            else:
                self.setPage(nextPage)
                nextPage.enterPage()
            return 0
        else:
            return -1


class UpdatingPage(InstallerPage):
    """
    This page is to copy files and initialize IMSx.
    """

    def __init__(self, conf, res, logger, util, frame):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        self.steps = None
        # Use a thread to perform copy files action.
        self.copyThd = threading.Thread(target=self.copyThreadMain)
        self.title = urwid.Text((resource.PALETTE_TITLE, self.res.getWording("txt.upgrade.title")))
        self.desc = urwid.Text((resource.PALETTE_BODY, self.res.getWording("txt.upgrade.desc")))
        self.progressBar = urwid.ProgressBar(resource.PALETTE_PROGRESS_INCOMPLETE, resource.PALETTE_PROGRESS_COMPLETE)
        innerFrameHead = urwid.Pile([urwid.Divider(),
                                     self.title,
                                     urwid.Padding(urwid.Divider(div_char=u"-", bottom=1), width=self.title.pack()[0]),
                                     self.desc,
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Divider(),
                                     urwid.Padding(self.progressBar, right=20),
                                     urwid.Divider()])

        self.dotdotdot = urwid.Text("")
        self.finished = False
        self.dotdotcount = 1
        self.progressText = urwid.Text("")
        self.notificationText = urwid.Text("")
        self.widgetList = [urwid.Divider(),
                           self.dotdotdot,
                           urwid.Divider(),
                      self.progressText,
                           urwid.Divider(),
                      self.notificationText,
                      urwid.Divider()
                      ]

        self.progressTextBox = urwid.ListBox(urwid.SimpleFocusListWalker(self.widgetList))
        # Use filler to limit the height of progress text.
        innerFrame = urwid.Frame(urwid.Filler(self.progressTextBox, valign="top", height=80), header=innerFrameHead)
        self.updater = self.CopyPageUpdater(self.progressText, self.progressTextBox)
        super(UpdatingPage, self).__init__(innerFrame, frame, mode=2)

    def updateDotDotDot(self):
        if not self.finished:
            self.dotdotcount = (self.dotdotcount+1)%20
            self.dotdotdot.set_text((resource.PALETTE_BODY_PASS,".."*self.dotdotcount))
        else:
            self.dotdotdot.set_text("")

    def keypress(self, size, key):
        # Handles exit only after copy thread exits.
        if not self.copyThd.isAlive() and (key == "ctrl x" or key == "f12"):
            raise urwid.ExitMainLoop()
        else:
            return super(UpdatingPage, self).keypress(size, key)

    class CopyPageUpdater(step.StatusUpdater):
        """
        The status updater for copy page. This updater appends content to original
        contents and must move the text to the bottom.
        """

        def __init__(self, textWidget, listBoxWidget):
            self.textWidget = textWidget
            self.listBoxWidget = listBoxWidget


        def append(self, msg):
            self.textWidget.set_text(self.textWidget.text + msg + "\n")
            self.listBoxWidget.set_focus_valign("bottom")

    def enterPage(self):
        super(UpdatingPage, self).enterPage()
        self.frame.setFooterText("")
        self.copyThd.start()

    def setCopySteps(self, steps):
        """
        Set the steps needs to be done in this page.
        """
        self.steps = list(steps)

    def doCopy(self):
        """
        Do copy and configure IMSx actions.

        return: 0 if successful, -1 if failed.
        """
        stepNo = -1
        interrupted = False
        for oneStep in self.steps:
            stepNo += 1
            self.logger.debug("Running step %d: %s..." % (stepNo, oneStep.__class__.__name__))
            if oneStep.apply() != 0:
                self.logger.debug("Fail to running step %d." % (stepNo))
                while True:
                    if os.path.exists(Const.DEBUG_FLAG_FILE):
                        time.sleep(5)
                    else:
                        break
                if not oneStep.isCritical:
                    self.logger.debug("This step is not critical, continue installation.")
                    continue
                else:
                    self.logger.debug("This step is critical, installation cannot continue.")
                    interrupted = True
                    break
            else:
                self.setCurrentProgress(oneStep)

        # If the installation doesn't finish successfully, need to rollback the changes.
        if interrupted:
            self.logger.debug("Installation failed in the middle, begin to rollback changes.")
            while stepNo >= 0:
                oneStep = self.steps[stepNo]
                self.logger.debug("Rolling back step %d: %s..." % (stepNo, oneStep.__class__.__name__))
                # Even if one step fails to rollback, still continue the whole rollback process.
                if oneStep.rollback() != 0:
                    self.logger.debug("Fail to rollback step %d." % (stepNo))
                stepNo -= 1
            return -1
        else:
            # The installation completed with no error.
            self.progressBar.set_completion(self.progressBar.done)
            self.frame.complete = True
        return 0

    def showNeedUpgradeComponents(self):
        description = self.res.getWording("txt.upgrade.result.notification")

        euqDbStr = ""
        euqDbList = self.util.getNoUpgradeEuqDbList();
        if len(euqDbList) > 0:
            euqDbStr += "Database\nServer          Port    User    DBName\n"
            for row in euqDbList:
                euqDbStr += "%s    %s    %s    %s" % (str(row[0]), str(row[1]), str(row[2]), str(row[3]))
                euqDbStr += "\n"

        compStr = ""
        compList = self.util.getNoUpgradeComponentList()

        if len(compList) > 0:
            compStr += "Servers\nAddress        version     Components \n"
            for row in compList:
                compStr += "%s  %s  %s" % (str(row[0]), str(row[1]), str(row[2]))
                compStr += "\n"

        if euqDbStr != "" or compStr != "":
            noti_info = "%s\n\n%s%s"%(description, compStr, euqDbStr)
            self.notificationText.set_text((resource.PALETTE_BODY_ERR, noti_info))


    def copyThreadMain(self):
        """
        The main entry of copy thread.
        """
        try:
            ret = self.doCopy()
            self.finished = True
            if ret == 0:
                self.title.set_text((resource.PALETTE_TITLE, self.res.getWording("txt.upgrade.title.complete")))
                self.desc.set_text(u"")
                msg = [self.res.getWording("txt.upgrade.result.ok"), "\n"]
                self.progressText.set_text((resource.PALETTE_BODY, "".join(msg)))
                self.showNeedUpgradeComponents()
            else:
                self.progressText.set_text(
                    [self.progressText.text, (resource.PALETTE_BODY_ERR, self.res.getWording("msg.installFail"))])
            self.frame.setFooterText(self.res.getWording("txt.copy.footer"))
        except Exception as e:
            self.logger.debug(utility.excToString(e))

    def setCurrentProgress(self, aStep):
        """
        Increase and set the progress bar according to the step just executed.
        """
        currentProgress = self.progressBar.current
        if isinstance(aStep, step.SilentInstallUsersAndGroups):
            currentProgress += 10
        elif isinstance(aStep, step.ImssUpgradeInit):
            currentProgress += 30
        elif isinstance(aStep, step.ImssUpgradeInstallNewVersion):
            currentProgress += 20
        elif isinstance(aStep, step.ImssUpgradeAdminDB):
            currentProgress += 40

        if currentProgress > self.progressBar.done:
            currentProgress = self.progressBar.done
        self.progressBar.set_completion(currentProgress)

class ImssTuiUpgrade(Installer):
    """TUI for IMSS inline upgrade"""
    def __init__(self, conf, res, logger, util):
        super(ImssTuiUpgrade, self).__init__(conf, res, logger, util)
        self.upgradeFrame = UpgradeFrame(conf=conf,res=res,logger=logger,util=util)
        self.loop = None
        self.loopStatus = 0

    def composeUpgradeSteps(self):
        steps = []
        ret = 0
        updater = self.upgradeFrame.copyPage.updater
        steps.append(step.SilentInstallUsersAndGroups(self.conf, self.res, self.logger, self.util, True, updater))
        steps.append(ImssUpgradeInit(res=self.res,conf = self.conf, logger=self.logger,util=self.util,isCritical=True,updater=updater))
        steps.append(ImssUpgradeInstallNewVersion(res=self.res,conf = self.conf, logger=self.logger,util=self.util,isCritical=True,updater=updater))
        steps.append(ImssUpgradeAdminDB(res=self.res,conf = self.conf, logger=self.logger,util=self.util,isCritical=True,updater=updater))
        return steps

    def refreshScreen(self):
        """
        Refresh the screen in schedule.
        """
        while self.loopStatus != 2:
            try:
                if self.loopStatus == 1:
                    self.loop.draw_screen()
            # Ignore exceptions.
            except:
                pass
            time.sleep(1)
            self.upgradeFrame.updateProcess()

    def pesudoSilentUpgrade(self):
        steps = self.composeUpgradeSteps()

        stepNo = -1
        for oneStep in steps:
            stepNo += 1
            self.logger.debug("Running step %d: %s..." % (stepNo, oneStep.__class__.__name__))
            if oneStep.apply() != 0:
                self.logger.debug("Fail to running step %d." % (stepNo))
                if not oneStep.isCritical:
                    self.logger.debug("This step is not critical, continue installation.")
                    continue
                else:
                    self.logger.debug("This step is critical, installation cannot continue.")
                    interrupted = True
                    break
                    # If the installation doesn't finish successfully, need to rollback the changes.
        if interrupted:
            self.logger.debug("Installation failed in the middle, begin to rollback changes.")
            while stepNo >= 0:
                oneStep = steps[stepNo]
                self.logger.debug("Rolling back step %d: %s..." % (stepNo, oneStep.__class__.__name__))
                # Even if one step fails to rollback, still continue the whole rollback process.
                if oneStep.rollback() != 0:
                    self.logger.debug("Fail to rollback step %d." % (stepNo))
                stepNo -= 1
            return -1
        else:
            self.logger.debug("Silent installation finished.")
            return 0


    def upgrade(self):
        self.loop = urwid.MainLoop(self.upgradeFrame, palette=self.res.palette, handle_mouse=False)

        steps = self.composeUpgradeSteps()
        self.upgradeFrame.copyPage.setCopySteps(steps)
        # Use a thread to refresh the screen in schedule. Otherwise, the screen
        # gets stuck if we stay in keypress method.
        refreshThd = threading.Thread(target=self.refreshScreen)
        refreshThd.start()
        self.loopStatus = 1
        self.loop.run()
        self.loopStatus = 2
        refreshThd.join()
        if self.upgradeFrame.complete:
            self.logger.debug("Upgrade completed successfully.")
            return 0
        else:
            self.logger.debug("Upgrade aborted in the middle.")
            return -1