#!/usr/bin/env python
# -*- coding: utf-8 -*-

import ConfigParser
import os.path

# The resources are stored in SECTION_RESOURCE section.
SECTION_WORDING = "wording"

# Palette names of tui
PALETTE_HEADER = "header"
PALETTE_HEADER_EM = "header_em"
PALETTE_FOOTER = "footer"
PALETTE_TITLE = "title"
PALETTE_BODY = "body"
PALETTE_BODY_EM = "body_em"
PALETTE_BODY_PASS = "body_pass"
PALETTE_BODY_ERR = "body_err"
PALETTE_BODY_WARN = "body_warn"
PALETTE_BODY_FOCUS = "body_focus"
PALETTE_BODY_EM_FOCUS = "body_em_focus"
PALETTE_BODY_ERR_FOCUS = "body_err_focus"
PALETTE_BODY_ERR_EM_FOCUS = "body_err_em_focus"
PALETTE_PROGRESS_INCOMPLETE = "progress_incomplete"
PALETTE_PROGRESS_COMPLETE = "progress_complete"

class Resource(object):
    """
    This class is used to access resources, usually wordings. The resources 
    file is an ini file.
    """
    
    def __init__(self):
        """
        Use a ConfigParser object to store resources.
        """
        self.confParser = ConfigParser.SafeConfigParser()
        self.palette = [(PALETTE_HEADER, "white", "dark red"),
                        (PALETTE_HEADER_EM, "yellow", "dark red"),
                        (PALETTE_FOOTER, "black", "light gray"),
                        (PALETTE_TITLE, "white,bold", "black"),
                        (PALETTE_BODY, "white", "black"),
                        (PALETTE_BODY_PASS, "light green", "black"),
                        (PALETTE_BODY_ERR, "light red", "black"),
                        (PALETTE_BODY_WARN, "yellow", "black"),
                        (PALETTE_BODY_EM, "light cyan", "black"),
                        (PALETTE_BODY_FOCUS, "black", "light gray"),
                        (PALETTE_BODY_EM_FOCUS, "white", "light gray"),
                        (PALETTE_BODY_ERR_FOCUS, "black", "light gray"),
                        (PALETTE_BODY_ERR_EM_FOCUS, "light red", "light gray"),
                        (PALETTE_PROGRESS_COMPLETE, "white", "dark blue"),
                        (PALETTE_PROGRESS_INCOMPLETE, "black", "light gray")]
        self.eulaPath = ""
        self.eula = ""
    
    def loadResFile(self, path):
        """
        Load the resource file.
        
        path: the full file path of the resource file.
        
        return: a tuple (retcode, exception). Retcode 0 means successful, in 
        this case, exception is None. Retcode -1 means failed, and exception
        is set.
        """
        ret = 0
        exception = None
        
        try:
            self.confParser.readfp(open(path, "r"))
            # Read EULA content from the file.
            self.eulaPath = os.path.join(os.path.dirname(path), self.getWording("const.eulaFileName"))
            if os.path.isfile(self.eulaPath):
                with open(self.eulaPath, "r") as eulaFile:
                    self.eula = eulaFile.read()
        except Exception, e:
            ret = -1
            exception = e
        
        return (ret, exception)

    def getWording(self, key):
        """
        Return the content of a specific wording by key.
        NOTE: in the current ConfigParser implementation, leading and tailing 
        spaces of a value are removed, if you want to reserve it, just quote the 
        value using ", e.g., key = " space ".
        
        key: the key of the wording.
        
        return: the content of the wording, if the key doesn't exist, an 
        empty string is returned.
        """
        try:
            return self.confParser.get(SECTION_WORDING, key).strip('"')
        except:
            return ""
    
    def getEULAContent(self):
        """
        Return the EULA file content in a string.
        """
        return self.eula
    