#!/usr/bin/env python
# -*- coding: utf-8 -*-

import abc
import os
import subprocess
import re
import socket
import time
import ConfigParser
import datetime

import sys

import utility
import pwd
import stat
import urllib
import urllib2
import cookielib
import base64
import xml.etree.ElementTree
import shutil
from utility import Utility
from model import Const, CronMgr, PostUpgradeAction, Debugger
from model import g_upd_models


class StatusUpdater(object):
    """
    This class and its subclasses are used to display a series of status 
    updates to a certain object.  
    """
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def append(self, msg):
        """
        Append the content to the current display object.
        """
        return

class TerminalStatusUpdater(StatusUpdater):
    """
    Display status to the terminal (standard output).
    """
    def append(self, msg):
        print msg

class Step(object):
    """
    This class defines a step of in the whole installation, uninstallation or 
    upgrade process, the whole process is made up of a list of steps.
    """

    def __init__(self, conf, res, logger, util, isCritical, updater):
        self.conf = conf
        self.res = res
        self.logger = logger
        self.util = util
        # A critical step means if this fails, won't perform the following steps.
        self.isCritical = isCritical
        self.updater = updater

    def apply(self):
        """
        Perform the step.
        
        return: 0 if successful, -1 if failed.
        """
        return 0

    def rollback(self):
        """
        Rollback the current step.
        
        return: 0 if successful, -1 if failed.
        """
        return 0

    def uninstallCron(self, updater):
        """
        Remove the cron jobs created during installation.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        self.logger.debug("Try to uninstall IMSx related crontab.")
        cronSetting = self.conf.getCronSetting()
        (ret, stdout, stderr) = self.util.getLocalCronJobs("root")
        if ret != 0:
            self.logger.debug("Fail to get root's crontab, stdout: %s, stderr: %s" %(stdout, stderr))
            updater.append(self.res.getWording("msg.uninstallCronFail"))
            return -1

        newCronContent = []
        for line in stdout.splitlines():
            match = False
            for (sig, _) in cronSetting:
                # If the line contains the signature, then this line will be removed.
                if re.search(sig, line):
                    match = True
                    self.logger.debug('Remove "%s" from crontab.' %(line))
                    break
            if not match:
                newCronContent.append(line)
        newCronContent = "\n".join(newCronContent)
        if not newCronContent.endswith("\n"):
            newCronContent += "\n"

        self.logger.debug("The crontab after uninstallation is:\n%s" %(newCronContent))
        (ret, stdout, stderr) = self.util.setLocalCronJobs("root", newCronContent)
        if ret != 0:
            self.logger.debug("Fail to update root's crontab, stdout: %s, stderr: %s" %(stdout, stderr))
            updater.append(self.res.getWording("msg.uninstallCronFail"))
            return -1

        # Restart crond service.
        subprocess.call(["service", "crond", "restart"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        return 0

    def unregisterDeviceFromAdminDb(self, updater):
        """
        Unregister the current package from the admin db.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        # Check if admin DB can be connected.
        (ret, e) = self.util.isDatabaseConnectable(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName())
        if ret != 0:
            self.logger.debug("The admin DB cannot be connected, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.dbCannotConnect"))
            updater.append(self.res.getWording("msg.unregisterFail"))
            return -1

        localScannerId = self.getLocalScannerId()
        if localScannerId == -1:
            self.logger.debug("Fail to get the local scanner ID.")
            updater.append(self.res.getWording("msg.unregisterFail"))
            return -1
        self.logger.debug("Try to unregister current device with scanner ID %d from admin DB %s:%d (name:%s)."
                          %(localScannerId, self.conf.getAdminDBAddress(), self.conf.getAdminDBPort(), self.conf.getAdminDBName()))

        sql = "delete from tb_component_list where scanner_id=%s"
        para = [localScannerId]
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    para)
        if ret != 0:
            self.logger.debug("Fail to delete current device from tb_component_list, %s", utility.excToString(e))
            updater.append(self.res.getWording("msg.unregisterFail"))
            return -1

        #delete child's service cert usage and change cert status in AdminDB
        #get all certificates used by the child, child only has mta, parent should not come in this function
        sql = "select cert_id from tb_tls_service_certificate_usage where scanner_id=%s and (service_type=0 or service_type=3)"
        para = [localScannerId]
        (usedCerts, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                         self.conf.getAdminDBPort(),
                                                         self.conf.getAdminDBUsername(),
                                                         self.util.decryptString(self.conf.getAdminDBPassword()),
                                                         self.conf.getAdminDBName(),
                                                         sql,
                                                         para)
        if usedCerts == None:
            self.logger.debug("Fail to get cert_id for current device from tb_tls_service_certificate_usage, %s", utility.excToString(e))
            updater.append(self.res.getWording("msg.unregisterFail"))
            return -1

        # Delete all ceritifcate usages of the child.
        sql = "delete from tb_tls_service_certificate_usage where scanner_id=%s and (service_type=0 or service_type=3)"
        para = [localScannerId]
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    para)
        if ret != 0:
            self.logger.debug("Fail to delete current service from tb_tls_service_certificate_usage, %s", utility.excToString(e))
            updater.append(self.res.getWording("msg.unregisterFail"))
            return -1

        # Check all certificates used by the child, if it is not used by others, set its status.
        for item in usedCerts:
            child_cert_id = item[0]
            sql = "select * from tb_tls_service_certificate_usage where cert_id=%s"
            para = [child_cert_id]
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            #cert is not using, change its status to 2
            if ret == []:
                sql = "update tb_tls_certificate set cert_status=2 where cert_id=%s"
                para = [child_cert_id]
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql,
                                                            para)
                if ret != 0:
                    self.logger.debug("Fail to update certificate status, %s", utility.excToString(e))
                    updater.append(self.res.getWording("msg.unregisterFail"))
                    return -1

        # Restart central controller's imssmgr.
        if localScannerId != 1:
            sql = "update tb_component_list set admin_cmd=%s where scanner_id=1"
            # 192 is CMD_IMSSMGR_RESTART.
            para = [192]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Fail to request restarting imssmgr on parent device, %s", utility.excToString(e))
                updater.append(self.res.getWording("msg.unregisterFail"))
                return -1

        updater.append(self.res.getWording("msg.unregisterSuccess"))
        return 0

    def removeImsxFiles(self, updater):
        """
        Remove files and services.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        updater.append(self.res.getWording("msg.uninstallRemoveFiles"))

        failedRemoveList = self.conf.getFailRemovePaths()
        manualRemoveList = self.conf.getManualRemovePaths()
        autoRemoveList = self.conf.getAutoRemovePaths()
        if self.conf.getRemoveData() == 1:
            autoRemoveList.extend(self.conf.getOptionalRemovePaths())
        else:
            manualRemoveList.extend(self.conf.getOptionalRemovePaths())
        for aPath in autoRemoveList:
            (ret, e) = self.util.removePath(aPath)
            if ret != 0:
                self.logger.debug("Fail to remove path %s, %s" %(aPath, utility.excToString(e)))
                failedRemoveList.append(aPath)
            else:
                self.logger.debug("Successfully removed path %s." %(aPath))

        # Remove services.
        services = self.conf.getServiceSetting()
        if self.conf.getIsRHEL7OrLater() is False:
            for (src, dst, name) in services:
                self.logger.debug("Try to remove service %s, source: %s, link: %s" %(name, src, dst))
                (ret, e) = self.util.removePath(dst)
                if ret != 0:
                    self.logger.debug("Fail to remove path %s, %s" %(dst, utility.excToString(e)))
                    failedRemoveList.append(dst)
                subprocess.call(["chkconfig", "--del", name],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))
        else:
            IMSS_SERVICE = "imss.service"
            IMSS_SERVICE_DST_FILE = "/usr/lib/systemd/system/%s" % IMSS_SERVICE
            subprocess.call(["systemctl", "disable", IMSS_SERVICE],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))
            subprocess.call(["rm", "-rf", IMSS_SERVICE_DST_FILE],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))






        self.logger.debug("Following path must be manually removed after installation:")
        for aPath in manualRemoveList+failedRemoveList:
            self.logger.debug("\t%s" %(aPath))
        self.conf.setFailRemovePaths(failedRemoveList)

        return 0

    def checkRpmForUninstallation(self, updater):
        """
        Check if all rpms were installed before uninstallation.
        """
        self.logger.debug("Check installed IMSx rpms for uninstallation.")
        updater.append(self.res.getWording("msg.checkInstallStatus"))
        installedRpms = self.checkInstalledImsxRpms()
        ret = 0
        for rpm in self.conf.getRpmNames():
            if rpm not in installedRpms:
                self.logger.debug("RPM %s is not installed." %(rpm))
                updater.append(self.res.getWording("msg.rpmNotInstalled") %(rpm))
                ret = -1

        return ret

    def checkRpmForInstallation(self, updater):
        """
        Check if there's IMSx installed already, if installed, this IMSx must 
        be uninstalled before proceeding.
        
        updater: the status updater.
        
        return: 0 if no rpm installed, -1 if any rpm installed.
        """
        ret = 0

        installedRpms = self.checkInstalledImsxRpms()
        if installedRpms:
            ret = -1
            for rpm in installedRpms:
                updater.append(self.res.getWording("msg.rpmExistBeforeInstall") %(rpm))

        return ret

    def updateIssueFile(self, updater):
        """
        On IMSVA, need to update the issue file.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        webIp = ""
        # Get network interfaces info.
        nicInfo = self.util.getNicInfo()
        if not nicInfo:
            self.logger.debug("Failed to get the network interfaces info.")
            return -1

        # Get adminUI access status.
        p = subprocess.Popen(["/opt/TrendMicro/GoldenGate/bin/glcfg.sh", "val", "firewall", "webaccess", "status"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (webConf, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to execute command to get adminUI access status, stdout: %s, stderr: %s" %(webConf, stderr))
            return -1
        webConf = webConf.strip()

        if webConf == "all":
            # Get parent or central controller's IP, the scanner id of parent or
            # central controller must be 1.
            sql = "SELECT ip_addr from tb_component_list where scanner_id=1;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to get the ip address of admin console, %s" %(utility.excToString(e)))
                return -1

            if not ret:
                self.logger.debug("Fail to get the ip address of device whose scanner_id is 1.")
                return -1

            webIp = ret[0][0]
        elif webConf == "none":
            self.logger.debug("The adminUI access is closed.")
        else:
            if webConf in nicInfo.keys():
                webIp = nicInfo[webConf][0]
            else:
                self.logger.debug("Cannot find device %s on local system." %(webConf))
                return -1

        # Update the issue file.
        try:
            with open("/etc/issue", "w") as f:
                content = []
                content.append("\n%s\n\n" %(self.res.getWording("msg.issueTitile")))
                if webConf == "none":
                    content.append("%s\n" %(self.res.getWording("msg.issueWebDisable")))
                else:
                    content.append("%s\n" %(self.res.getWording("msg.issueWebConsole")))
                    content.append("\n\thttps://%s:8445\n\n" %(webIp))
                content.append("%s\n\n" %(self.res.getWording("msg.issueAuthentication")))
                content.append("%s\n\n" %(self.res.getWording("msg.issueCLI")))
                f.write("".join(content))
        except Exception, e:
            self.logger.debug("Failed to update the /etc/issue file, %s" %(utility.excToString(e)))
            return -1

        return 0

    def getIcpName(self):
        """
        Get the interface name of current ICP.
        
        return: the name of ICP, or "" if failed.
        """
        return  self.util.getIcpName()


    def startApp(self, updater):
        """
        Start the IMSx application. Don't check the result.
        """
        self.logger.debug("Try to start the application.")
        startScript = "%s/imss/script/imssctl.sh" %(self.conf.getInstallPath())
        p = subprocess.Popen([startScript, "start"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        self.logger.debug("Execute script to start all services, stdout:%s, stderr: %s" %(stdout, stderr))

    def stopApp(self):
        """
        Stop the IMSx application.
        """
        p = subprocess.Popen(["%s/imss/script/imssctl.sh" %(self.conf.getInstallPath()), "stop"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        self.logger.debug("Execute script to stop all services, stdout:%s, stderr: %s" %(stdout, stderr))

    def stopAppExceptDB(self):
        """
        Stop all IMSX services except adminDB.
        """
        p = subprocess.Popen(["%s/imss/script/imssctl.sh" %(self.conf.getInstallPath()), "stop_others"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        self.logger.debug("Execute script to stop all services except admin DB and UI, stdout:%s, stderr: %s" %(stdout, stderr))
        p = subprocess.Popen(["%s/imss/script/S99ADMINUI" %(self.conf.getInstallPath()), "stop"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        self.logger.debug("Execute script to stop admin UI, stdout:%s, stderr: %s" %(stdout, stderr))

    def saveInstallData(self, updater):
        """
        Save some files/data generated in installation, so they can be referred by
        other utilities and for uninstallation.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        # Save the install config to IMSS_HOME/config folder.
        # Modify the log path to $IMSS_HOME/log/debugFile.txt.
        self.conf.setLogPath("%s/imss/log/debugFile.txt" %(self.conf.getInstallPath()))
        confDst = "%s/imss/config/installer.ini" %(self.conf.getInstallPath())
        (ret, e) = self.conf.saveConfig(confDst)
        if ret != 0:
            self.logger.debug("Fail to save install config data to %s, %s" %(confDst, utility.excToString(e)))
            updater.append(self.res.getWording("msg.saveFileFail") %(confDst))
            return -1
        
        uninstallScriptSrc = os.path.join(os.path.dirname(os.path.dirname(os.path.realpath(__file__))), ".uninstall.sh")
        uninstallScriptDst = os.path.join(self.conf.getInstallPath(), "imss", "backup", "uninstall.sh")
        try:
            shutil.copyfile(uninstallScriptSrc, uninstallScriptDst)
        except Exception, e:
            self.logger.debug("Fail to save uninstall script to %s, %s" %(uninstallScriptDst, utility.excToString(e)))
            updater.append(self.res.getWording("msg.saveFileFail") %(uninstallScriptDst))
            return -1
        os.chown(uninstallScriptDst, pwd.getpwnam("imss").pw_uid, pwd.getpwnam("imss").pw_gid)
        # Mode: 750
        os.chmod(uninstallScriptDst, stat.S_IRUSR|stat.S_IWUSR|stat.S_IXUSR|stat.S_IRGRP|stat.S_IXGRP)

        return 0

    def removeInstallTempFiles(self, updater):
        """
        Remove the temp files generated during installation.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        return 0

    def installCron(self, updater):
        """
        Update the cron job.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        self.logger.debug("Try to update the crontab.")
        # Get root user's crontab content.
        (ret, stdout, stderr) = self.util.getLocalCronJobs("root")
        if ret != 0:
            self.logger.debug("Failed to get current crontab content, stdout: %s, stderr: %s"
                              %(stdout, stderr))
            self.logger.debug("Cannot get crontab content, will create a new crontab file.")
            stdout = ""

        cronRules = self.conf.getCronSetting()
        newCronContent = stdout.splitlines()
        for sig, content in cronRules:
            if stdout and re.search(sig, stdout):
                continue
            else:
                newCronContent.append(content)
        newCronContent = "\n".join(newCronContent)
        if not newCronContent.endswith("\n"):
            newCronContent += "\n"

        self.logger.debug("The new crontab content is:\n%s" %(newCronContent))

        # Write back the crontab content.
        (ret, stdout, stderr) = self.util.setLocalCronJobs("root", newCronContent)
        if ret != 0:
            self.logger.debug("Failed to write the content to crontab, stdout: %s, stderr: %s"
                              %(stdout, stderr))
            return -1

        if self.exportCronSetting() != 0:
            return -1

        # Restart crond service.
        subprocess.call(["service", "crond", "restart"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        return 0

    def exportCronSetting(self):
        """
        Export the cron settings to a file so that it can be used by
        cronjob monitor in imssmgr.
        
        return: 0 if successful, -1 if failed. 
        """
        exportPath = "%s/imss/config/cronjob.ini" %(self.conf.getInstallPath())
        try:
            f = open(exportPath, "w")
            for (key, value) in self.conf.getCronSetting():
                f.write("%s=%s\n" %(key, value))
        except Exception, e:
            self.logger.debug("Fail to export the cron setting for imssmgr's cronjob monitor to use, %s" %(utility.excToString(e)))
            return -1

        return 0

    def uninstallService(self):
        ret = 0
        if self.conf.getIsRHEL7OrLater() is False:
            services = self.conf.getServiceSetting()
            for (src, dst, name) in services:
                if os.path.exists(dst):
                    try:
                        os.unlink(dst)
                    except Exception, e:
                        self.logger.debug("Failed to delete link %s " % (src, dst, utility.excToString(e)))
                        ret = -1
                subprocess.call(["chkconfig", "--del", name],
                                stdout=open("/dev/null"),
                                stderr=open("/dev/null"))
        else:
            IMSS_SERVICE = "imss.service"
            IMSS_SERVICE_DST_FILE = "/usr/lib/systemd/system/%s" % IMSS_SERVICE
            if os.path.exists(IMSS_SERVICE_DST_FILE):
                ret = os.system("rm -f %s"%IMSS_SERVICE_DST_FILE)
                if ret!=0:
                    self.logger.debug("Failed to delete imss.service file %s " % IMSS_SERVICE_DST_FILE)
        return ret


    def installService(self, updater):
        """
        Configure system service for IMSx (e.g., Sxx and Kxx scripts).
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.  
        """
        self.logger.debug("Setting IMSS as service, System version is latter than 7: %r", self.conf.getIsRHEL7OrLater())
        if self.conf.getIsRHEL7OrLater() is False:
            services = self.conf.getServiceSetting()
            for (src, dst, name) in services:
                if not os.path.exists(src):
                    self.logger.debug("The source file %s doesn't exist, cannot create RC link." %(src))
                    continue
                if os.path.exists(dst):
                    self.logger.debug("The dest file %s exists already, don't create RC link." %(dst))
                    continue
                try:
                    os.symlink(src, dst)
                except Exception, e:
                    self.logger.debug("Failed to create link from %s to %s, %s" %(src, dst, utility.excToString(e)))
                    return -1
                subprocess.call(["chkconfig", "--add", name],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))
        else:
            IMSS_SERVICE = "imss.service"
            IMSS_SERVICE_SRC_FILE = "%s/imss/script/%s" % (self.conf.getInstallPath(), IMSS_SERVICE)
            IMSS_SERVICE_DST_FILE = "/usr/lib/systemd/system/%s" % IMSS_SERVICE
            try:
                f = open(IMSS_SERVICE_DST_FILE, "w")
                for line in self.conf.getImssSystemdSettings():
                    self.logger.debug(line)
                    f.write(line)
                    f.write("\n")
                f.close()
                subprocess.call(["chown", "root:root", IMSS_SERVICE_DST_FILE],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))

                subprocess.call(["chmod", "644", IMSS_SERVICE_DST_FILE],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))

                subprocess.call(["systemctl", "enable", IMSS_SERVICE],
                                stdout = open("/dev/null"),
                                stderr = open("/dev/null"))
            except Exception, e:
                self.logger.debug("Fail to create systemd item %s %s %s" %
                                  IMSS_SERVICE_SRC_FILE, IMSS_SERVICE_DST_FILE,
                                  utility.excToString(e))
        return 0


    def installUsersAndGroups(self, updater):
        """
        Add imsx related users and groups to local machine.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        # First add groups.
        for name, exist in self.conf.getGroups():
            if not exist:
                self.logger.debug("Try to add group %s." %(name))
                (ret, desc) = self.util.addGroup(name)
                if ret != 0:
                    updater.append(self.res.getWording("msg.addGroupFail") %(name))
                    self.logger.debug("Fail to add group %s, %s" %(name, desc))
                    return -1

        # Then add users.
        for name, group, home, shell, exist in self.conf.getUsers():
            if not exist:
                self.logger.debug("Try to add user %s, group: %s, home: %s, shell: %s." %(name, group, home, shell))
                (ret, desc) = self.util.addUser(name, group, home, shell)
                if ret != 0:
                    updater.append(self.res.getWording("msg.addUserFail") %(name))
                    self.logger.debug("Fail to add user %s, %s" %(name, desc))
                    return -1

        return 0

    def removeUsersAndGroups(self, updater):
        """
        Delete imsx related users and groups from local machine.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        # First remove users.
        for name, _, _, _, exist in self.conf.getUsers():
            if not exist:
                self.logger.debug("Try to remove user %s." %(name))
                (ret, desc) = self.util.delUser(name)
                if ret != 0:
                    updater.append(self.res.getWording("msg.removeUserFail") %(name))
                    self.logger.debug("Fail to remove user %s, %s" %(name, desc))
                    return -1

        # Then remove groups.
        for name, exist in self.conf.getGroups():
            if not exist:
                self.logger.debug("Try to remove group %s." %(name))
                (ret, desc) = self.util.delGroup(name)
                if ret != 0:
                    updater.append(self.res.getWording("msg.removeGroupFail") %(name))
                    self.logger.debug("Fail to remove group %s, %s" %(name, desc))
                    return -1

        return 0

    def replaceConstants(self, updater):
        """
        Replace the constants in scripts and config files with real values.
        
        updater: the status updater.
        
        return: if successful, return 0, otherwise return -1.
        """
        self.logger.debug("Try to replace constants in scripts and config files.")

        nicInfo = self.util.getNicInfo()
        if not nicInfo:
            self.logger.debug("Failed to get the network interfaces information.")
            return -1
        # For IMSS, use the first interface as ICP.
        if self.conf.getProduct() == 0:
            interface = nicInfo.keys()[0]
            self.logger.debug("Use %s as the ICP." %(interface))
            ip = nicInfo[interface][0]
        else:
            icp = self.getIcpName()
            if not icp:
                self.logger.debug("Cannot get ICP device name.")
                return -1
            self.logger.debug("The ICP is %s." %(icp))
            if icp not in nicInfo.keys():
                self.logger.debug("The ICP %s doesn't exist on this machine." %(icp))
                return -1
            ip = nicInfo[icp][0]

        imssHome = "%s/imss" %(self.conf.getInstallPath())

        ruleSet = [(r"_PKG_FOX_INSTALL_ROOT_", imssHome),
                   (r"_PKG_NRS_INSTALL_ROOT_", imssHome),
                   (r"_PKG_IMSS_HOME_", imssHome),
                   (r"_PKG_PSCMD_", r'"ps -ef --width 1000"'),
                   (r"_PKG_FOX_PROXY_PORT_", r"%d" %(self.conf.getFoxhunterListenPort())),
                   (r"_PKG_FOX_POSTFIX_PORT_", r"2500"),
                   (r"_PKG_FOX_DNS_ADDR_", r"127.0.0.1"),
                   (r"_PKG_DIG_ROOT_", r"%s/bind/dig" %(imssHome)),
                   (r"_PKG_RNDC_ROOT_", r"%s/bind/rndc" %(imssHome)),
                   (r"_PKG_PID_ROOT_", r"%s/bind/var/run/named" %(imssHome)),
                   (r"_PKG_NAMED_ROOT_", r"%s/bind/var/named/ipprofiler" %(imssHome)),
                   (r"_PKG_FOX_MSG_DOMAIN__", self.conf.getFoxhunterDomain()),
                   (r"_PKG_UI_SSL_PORT_", r"8445"),
                   (r"_PKG_EUQ_SSL_PORT_", r"8446"),
                   (r"_PKG_UI_MOD_JK_PORT1_", r"8444"),
                   (r"_PKG_UI_MOD_JK_PORT2_", r"8442"),
                   (r"_PKG_APACHE_SSL_PORT_", r"8447"),
                   (r"_PKG_APACHE_ADDRESS_", ip),
                   ]
        # Some constants (e.g., _PKG_INSTALL_ROOT_) cannot be replaced in cfgtool.sh, 
        # so we handle cfgtool.sh first, then expand the rule set and replace constants in other files.
        cfgtoolPath = "%s/script/cfgtool.sh" %(imssHome)
        (ret, e) = self.util.replaceInFile(ruleSet, cfgtoolPath)
        if ret != 0:
            self.logger.debug("Failed to replace constants in file %s, %s" %(cfgtoolPath, utility.excToString(e)))
            return -1

        # Expand the rule set for files other than cfgtool.sh
        ruleSet.extend([(r"_PKG_INSTALL_ROOT_", imssHome),
                        (r"_PKG_SERVER_", self.conf.getAdminDBAddress()),
                        # _PKG_DB_PORT_ must be replace before _PKG_DB_, otherwise the token _PKG_DB_PORT_ itself will be replaced.
                        (r"_PKG_DB_PORT_", self.conf.getAdminDBPort()),
                        (r"_PKG_DB_", self.conf.getAdminDBName()),
                        (r"_PKG_USER_", self.conf.getAdminDBUsername()),
                        (r"_PKG_PASS_", self.conf.getAdminDBPassword()),
                        ])

        fileSet = ["%s/config/Agent.ini" %(imssHome),
                   "%s/config/database.ini" %(imssHome),
                   "%s/config/fox_dns_server.list" %(imssHome),
                   "%s/config/imss.ini" %(imssHome),
                   "%s/config/odbc.ini" %(imssHome),
                   "%s/config/odbcinst.ini" %(imssHome),
                   "%s/config/foxdns.ini" %(imssHome),
                   "%s/config/foxproxy.ini" %(imssHome),
                   "%s/config/dkim/dkim.conf"%(imssHome),
                   "%s/config/dkim/dkim.ini"%(imssHome),
                   "%s/UI/adminUI/bin/Tomcat.sh" %(imssHome),
                   "%s/UI/adminUI/conf/server.xml" %(imssHome),
                   "%s/UI/adminUI/ROOT/WEB-INF/imss-resources.xml" %(imssHome),
                   "%s/UI/adminUI/ROOT/WEB-INF/classes/log4j.properties" %(imssHome),
                   "%s/UI/euqUI/bin/Tomcat.sh" %(imssHome),
                   "%s/UI/euqUI/bin/Apache.sh" %(imssHome),
                   "%s/UI/euqUI/conf/server.xml" %(imssHome),
                   "%s/UI/euqUI/conf/EUQ.conf" %(imssHome),
                   "%s/UI/euqUI/ROOT/WEB-INF/imss-servlet.xml" %(imssHome),
                   "%s/UI/euqUI/ROOT/WEB-INF/web.xml" %(imssHome),
                   "%s/UI/euqUI/ROOT/WEB-INF/classes/log4j.properties" %(imssHome),
                   "%s/UI/php/bin/apache.sh" %(imssHome),
                   "%s/UI/php/conf/widget.conf" %(imssHome),
                   "%s/UI/php/conf/php.ini" %(imssHome),
                   "%s/UI/php/conf/workers.properties" %(imssHome),
                   "%s/script/Apache.sh" %(imssHome),
                   "%s/script/check_prefilter_PR.sh" %(imssHome),
                   "%s/script/db_idle_watch.sh" %(imssHome),
                   "%s/script/dbctl.sh" %(imssHome),
                   "%s/script/euqtrunc.sh" %(imssHome),
                   "%s/script/ibe_job_doer_script.sh" %(imssHome),
                   "%s/script/ibe_job_getter_script.sh" %(imssHome),
                   "%s/script/imssctl.sh" %(imssHome),
                   "%s/script/imssstart.sh" %(imssHome),
                   "%s/script/imssstop.sh" %(imssHome),
                   "%s/script/is_euq_enable.sh" %(imssHome),
                   "%s/script/is_master.sh" %(imssHome),
                   "%s/script/ldapsync.sh" %(imssHome),
                   "%s/script/mailtraffic.sh" %(imssHome),
                   "%s/script/monitor_cmagent.pl" %(imssHome),
                   "%s/script/openldap.sh" %(imssHome),
                   "%s/script/postctl.sh" %(imssHome),
                   "%s/script/postfix" %(imssHome),
                   "%s/script/purge_by_dbsize.sh" %(imssHome),
                   "%s/script/purge_log.sh" %(imssHome),
                   "%s/script/purge_pp.sh" %(imssHome),
                   "%s/script/purge_scanner_info.sh" %(imssHome),
                   "%s/script/release_java.sh" %(imssHome),
                   "%s/script/rtstat.sh" %(imssHome),
                   "%s/script/S99DTASAGENT" %(imssHome),
                   "%s/script/S99IMSS" %(imssHome),
                   "%s/script/S99MSGTRACING" %(imssHome),
                   "%s/script/db_maintain.sh" %(imssHome),
                   "%s/script/euqtrans" %(imssHome),
                   "%s/script/forceUpdate.sh" %(imssHome),
                   "%s/script/foxlibd" %(imssHome),
                   "%s/script/foxproxyd" %(imssHome),
                   "%s/script/foxpurgelog" %(imssHome),
                   "%s/script/ibe_server.sh" %(imssHome),
                   "%s/script/imp_exp.sh" %(imssHome),
                   "%s/script/isfunctions" %(imssHome),
                   "%s/script/postfixctl.sh" %(imssHome),
                   "%s/script/S99ADMINUI" %(imssHome),
                   "%s/script/S99CLEANEUQ" %(imssHome),
                   "%s/script/S99CLEANEXPIRE" %(imssHome),
                   "%s/script/S99CMAGENT" %(imssHome),
                   "%s/script/S99DIGEST" %(imssHome),
                   "%s/script/S99EUQ" %(imssHome),
                   "%s/script/S99FOXDNS" %(imssHome),
                   "%s/script/S99MANAGER" %(imssHome),
                   "%s/script/S99MONITOR" %(imssHome),
                   "%s/script/S99MSGTRACING" %(imssHome),
                   "%s/script/S99POLICY" %(imssHome),
                   "%s/script/S99REPORT" %(imssHome),
                   "%s/script/S99SCHEDULED" %(imssHome),
                   "%s/script/S99UPDATE" %(imssHome),
                   "%s/script/S99WRSAGENT" %(imssHome),
                   "%s/script/S99DELIVERY" %(imssHome),
                   "%s/script/S99DKIM" %(imssHome),
                   "%s/script/S99TLSAGENT" %(imssHome),
                   "%s/script/S99SMTPCONNAGENT" %(imssHome),
                   "%s/script/startall.sh" %(imssHome),
                   "%s/script/bif_connect_feedback.sh" %(imssHome),
                   "%s/script/sync_outbound_ip.sh" %(imssHome),
                   "%s/script/sync_rcpt.sh" %(imssHome),
                   "%s/script/auto_import_export.sh" %(imssHome),
                   "%s/script/mgr_monitor_restart_process.sh" %(imssHome),
                   "%s/script/install_util.sh" %(imssHome),
                   "%s/script/dbupdate.sh" %(imssHome),
                   #"%s/script/euqdbreg.sh" %(imssHome),
                   "%s/bind/bindctl.sh" %(imssHome),
                   "%s/bind/dig.sh" %(imssHome),
                   "%s/bind/rndc.sh" %(imssHome),
                   "%s/bind/named.conf" %(imssHome),
                   "%s/cdt/imsacdt.sh" %(imssHome),
                   "%s/OpenLDAP/etc/openldap/slapd.conf" %(imssHome),
                   "%s/script/S99LOCALSERVERMGMT" %(imssHome),
                   "%s/script/S99LOGTRANSFER" %(imssHome),
                   "%s/script/UpdateDlpTemplate.sh" %(imssHome),
                   "%s/script/ctp_update_account.sh" %(imssHome),
                   "%s/script/ctp_get_stats.sh" %(imssHome),
                   "%s/cdt/ExInterface/ExInterface_IMSS.ini" %(imssHome),
                   "%s/cdt/cdt.ini" %(imssHome),
                   "%s/script/gen_crl_file.sh" %(imssHome),
                   "%s/bin/log.ini" %(imssHome),
                   "%s/script/pg.rc" %(imssHome),
                   ]

        # IMSVA needs some more files.
        if self.conf.getProduct() == 1:
            fileSet.extend(["%s/config/MsgTracing.conf" %(imssHome),
                            "%s/config/UserSync.conf" %(imssHome),
                            ])

        for item in fileSet:
            if not os.path.exists(item):
                continue
            (ret, e) = self.util.replaceInFile(ruleSet, item)
            if ret != 0:
                self.logger.debug("Failed to replace constants in file %s, %s" %(item, utility.excToString(e)))
                return -1

        return 0

    def installDatabase(self, updater):
        """
        Install the database:
        1. If fresh install, create the database for IMSx to use.
        2. Initialize the database schemas and factory settings.
        3. if child mode installation in IMSS, only create EUQ db
        updater: the object used to display progress.
        
        return: 0 if successful, -1 if failed.
        """
        # Init the local db.
        updater.append(self.res.getWording("msg.installDatabaseStart"))
        ret = self.installBundledDatabase()
        if ret != 0:
            updater.append(self.res.getWording("msg.installDatabaseFail"))
            return -1
        else:
            updater.append(self.res.getWording("msg.installDatabaseSuccess"))
            
        # For fresh install and bundled database, admin DB and EUQ db shares the same account name and password.
        if self.conf.getFreshInstall() == 1 and self.conf.getAdminDBType() == 0:
            self.conf.setEuqDBUsername(self.conf.getAdminDBUsername())
            self.conf.setEuqDBPassword(self.util.encryptString(self.conf.getAdminDBPassword()))
            
        # Set the EUQ db address, this must be a public address can be accessed by other machines.
        nicInfo = self.util.getNicInfo()
        icpName = self.getIcpName()
        if icpName in nicInfo:
            self.conf.setEuqDBAddress(nicInfo[icpName][0])

        self.logger.debug("AdminDB info: %s  %s  %s  %s ", self.conf.getAdminDBAddress(),
                     self.conf.getAdminDBPort(),
                     self.conf.getAdminDBUsername(),
                     self.conf.getAdminDBName())
        self.logger.debug("EUQ db info: %s  %s  %s  %s " % (self.conf.getEuqDBAddress(),
                     self.conf.getEuqDBPort(),
                     self.conf.getEuqDBUsername(),
                     self.conf.getEuqDBName()))

        #if child mode installation in IMSS, only create EUQ db
        #check connection to parent adminDB => check local euqdb exist => create local euqdb=>register to adminDB
        if self.conf.getFreshInstall() == 0 and self.conf.getProduct() == 0:
            self.logger.debug("Try to install IMSS child database")
            (ret, exception) = self.util.isDatabaseConnectable(
                                    self.conf.getAdminDBAddress(),
                                    self.conf.getAdminDBPort(),
                                    self.conf.getAdminDBUsername(),
                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                    self.conf.getAdminDBName())
            # The db exists.
            if ret == 0:
                self.logger.debug("The admin db can be connected.")
            else:
                self.logger.debug("Cannot connect to the admin db, %s"
                                  %(utility.excToString(exception)))
                updater.append(self.res.getWording("msg.dbCannotConnect"))
                return -1
            # Init the EUQ database schemas for IMSS
            if self.checkDbExist(self.conf.getEuqDBAddress(),
                                 self.conf.getEuqDBPort(),
                                 self.conf.getEuqDBUsername(),
                                 self.util.decryptString(self.conf.getEuqDBPassword()),
                                 self.conf.getEuqDBName()) != 0:
                updater.append(self.res.getWording("msg.dbExistForFreshInstall") %(self.conf.getEuqDBName()))
                return -1
            ret = self.createEuqDb(self.conf.getEuqDBAddress(),
                                   self.conf.getEuqDBPort(),
                                   self.conf.getEuqDBUsername(),
                                   self.util.decryptString(self.conf.getEuqDBPassword()),
                                   self.conf.getEuqDBName())
            if ret != 0:
                self.logger.debug("Failed to init EUQ db.")
                return -1

            # when register to adminDB, should use local ip address instead of 127.0.0.1
            # if IMSS child have multi nics, chose the first IP that in the same network with parent
            # after register child euqdb to adminDB reset the euqdb address to old one, no special meaning
            # different with fresh install, need not to initial tb_euq_table_info table
            localip = self.util.getIPFromAllNicsByParentIP(self.conf.getAdminDBAddress())
            port = self.conf.getEuqDBPort()
            username = self.conf.getEuqDBUsername()
            password = self.util.encryptString(self.conf.getEuqDBPassword())
            dbname = self.conf.getEuqDBName()
            sql = "insert into tb_euq_db_info (server,port,username,password,db_name,db_type,status) values" \
                  " (%s,%s,%s,%s,%s,'1',1)"
            para = [localip, port, username, password, dbname]

            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to insert the EUQ db information into the admin db, %s" %(utility.excToString(e)))
                return -1

        else:
            # Init the admin database (schemas and factory-settings)
            # Before creating the database, we need to make sure no connection to template1,
            # template1 is the source database used to create new database, refer to
            # http://www.postgresql.org/docs/9.1/static/manage-ag-templatedbs.html for details.
            templateDbName = "template1"
            if self.checkDbActivity(self.conf.getAdminDBAddress(),
                                    self.conf.getAdminDBPort(),
                                    self.conf.getAdminDBUsername(),
                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                    templateDbName) != 0:
                updater.append(self.res.getWording("msg.databaseUsedByOthers") %(templateDbName))
                return -1
            # Before init the admin database, check if anyone else is operating the database.
            if self.checkDbExist(self.conf.getAdminDBAddress(),
                                 self.conf.getAdminDBPort(),
                                 self.conf.getAdminDBUsername(),
                                 self.util.decryptString(self.conf.getAdminDBPassword()),
                                 self.conf.getAdminDBName()) != 0:
                updater.append(self.res.getWording("msg.dbExistForFreshInstall") %(self.conf.getAdminDBName()))
                return -1
            updater.append(self.res.getWording("msg.initAdminDbStart"))
            ret = self.initAdminDb()
            if ret != 0:
                updater.append(self.res.getWording("msg.initAdminDbFail"))
                return -1
            else:
                updater.append(self.res.getWording("msg.initAdminDbSuccess"))

            # Init the EUQ database schemas for IMSVA.
            if self.checkDbExist(self.conf.getEuqDBAddress(),
                                 self.conf.getEuqDBPort(),
                                 self.conf.getEuqDBUsername(),
                                 self.util.decryptString(self.conf.getEuqDBPassword()),
                                 self.conf.getEuqDBName()) != 0:
                updater.append(self.res.getWording("msg.dbExistForFreshInstall") %(self.conf.getEuqDBName()))
                return -1
            ret = self.createEuqDb(self.conf.getEuqDBAddress(),
                                   self.conf.getEuqDBPort(),
                                   self.conf.getEuqDBUsername(),
                                   self.util.decryptString(self.conf.getEuqDBPassword()),
                                   self.conf.getEuqDBName())
            if ret != 0:
                self.logger.debug("Failed to init EUQ db.")
                return -1

            ret = self.registerEuqDbToAdminDb()
            if ret != 0:
                self.logger.debug("Failed to register EUQ db to admin db.")
                return -1

        return 0

    def removeUseleseEuqDb(self):

        self.logger.debug("Begin remove useless euq db entry")

        sql = "select count(*) as count from tb_euq_db_info where server not in (select ip_addr from tb_component_list)\
         and db_type='1'"

        (count, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                     self.conf.getAdminDBPort(),
                                                     self.conf.getAdminDBUsername(),
                                                     self.util.decryptString(self.conf.getAdminDBPassword()),
                                                     self.conf.getAdminDBName(),
                                                     sql)

        if count is not None:
            sql = "delete from tb_euq_db_info where server not in (select ip_addr from tb_component_list) and db_type='1'"
            (res, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql)
            if res == -1:
                self.logger.debug("Faild to remove useless euq db entry")
                return -1
            else:
                self.logger.debug("Remove useless euq db entry successfully")
        self.logger.debug("End remove useless euq db entry successfully")
        return  0

    def reallocEuqTable(self):
        self.logger.debug("reallocEuqTable()......start")
        sql = "select db_id from tb_euq_db_info where status='1'"
        (euq_db_id, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                         self.conf.getAdminDBPort(),
                                                         self.conf.getAdminDBUsername(),
                                                         self.util.decryptString(self.conf.getAdminDBPassword()),
                                                         self.conf.getAdminDBName(),
                                                         sql)
        if euq_db_id is None:
            self.logger.debug("get euq db info error")
            return 0
        else:
            self.logger.debug("get euq db info success")

        sql = "select table_id from tb_euq_table_info"
        (euq_table_id, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
        if euq_table_id is None:
            self.logger.debug("get euq table info error")
            return 0
        else:
            self.logger.debug("get euq table info success")


        if len(euq_db_id) == 0:
            self.logger.debug("None entries in tb_euq_db_info")
            return -1

        average = len(euq_table_id) / len(euq_db_id)
        left = len(euq_table_id) - average * len(euq_db_id)

        index = 1
        index_euq_table_id = 0

        for db_id in euq_db_id:
            if index_euq_table_id == len(euq_table_id):
                break
            n_avg = average
            if left > 0 and index > len(euq_db_id) - left:
                n_avg += 1

            while n_avg > 0:
                sql = "update tb_euq_table_info set db_id=%d where table_id=%d" \
                      % (db_id[0], euq_table_id[index_euq_table_id][0])
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
                if ret == -1:
                    self.logger.debug("Error occured: update tb_euq_table_info error %s " % (sql))
                    return 0

                index_euq_table_id += 1
                if index_euq_table_id == len(euq_table_id):
                    break
                n_avg -= 1
            index += 1
            
        # After updating EUQ db table mappings, need to restart related services on scanners.
        # Restart running EUQ tomcats except itself.
        sql = "update tb_component_list set admin_cmd=admin_cmd|%s where euq=2 and scanner_id!=%s"
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    (48, self.conf.getScannerId()))
        if ret == -1:
            self.logger.debug("Fail to set command to restart EUQ tomcat: %s" % e)
        
        # Restart EUQ apache for master.
        sql = "update tb_component_list set admin_cmd=admin_cmd|%s where euq=2 and scanner_id=1"
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    (0x3000,))
        if ret == -1:
            self.logger.debug("Fail to set command to restart EUQ apache: %s" % e)
            
        # Restart imssmgr for all scanners except itself because imssmgr need the correct mapping to sync message.
        sql = "update tb_component_list set admin_cmd=admin_cmd|%s where scanner_id!=%s"
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    (192, self.conf.getScannerId()))
        if ret == -1:
            self.logger.debug("Fail to set command to restart imssmgr: %s" % e)

        self.logger.debug("reallocEuqTable()......end")
        return 0

    def createEuqDb(self, address, port, user, password, dbname):
        """
        Initialize the EUQ db (schemas). This is used to initialize the bundled EUQ database.
        
        return: 0 if successul, -1 if failed.
        """
        """
        Initialize admin db data (schemas and factory-settings)
        
        return: 0 if successful, -1 if failed.
        """

        self.logger.debug("Try to create the db %s." %(dbname))
        sql = "CREATE DATABASE %s ENCODING 'unicode';" %(dbname)
        (ret, e) = self.util.executeUpdateStatement(address, port, user, password, "postgres", sql)
        if ret != 0:
            self.logger.debug("Failed to create database %s, %s" %(dbname, utility.excToString(e)))
            return -1

        # Create the procedure language if the language doesn't exist.
        # If the PostgreSQL has been upgraded to 9.1, we can use "CREATE OR REPLACE.." statement.
        sql = "SELECT * FROM pg_language WHERE lanname = 'plpgsql';"
        (ret, e) = self.util.executeQueryStatement(address, port, user, password, dbname, sql)
        if ret == []:
            sql = "CREATE LANGUAGE plpgsql;"
            (ret, e) = self.util.executeUpdateStatement(address, port, user, password, dbname, sql)
            if ret != 0:
                self.logger.debug("Failed to create the procedure language, %s" %(utility.excToString(e)))
                return -1
        elif e:
            self.logger.debug("Failed to check if language plpgsql exists, %s" %(utility.excToString(e)))
            return -1

        # Execute sql files.
        sqlPath = "%s/imss/sql/postgresql/install" %(self.conf.getInstallPath())
        createSqls = ["%s/createEuqDB.sql" %(sqlPath)]
        for item in createSqls:
            self.logger.debug("Try to execute sql file %s" %(item))
            (ret, e) = self.util.executeSqlFile(address, port, user, password, dbname, item)
            if ret != 0:
                self.logger.debug("Failed to execute sql file %s, %s" %(item, utility.excToString(e)))
                return -1

        return 0

    def registerEuqDbToAdminDb(self):
        """
        Register the EUQ db to admin db. This is only to implement the IMSVA 
        installation, this function shall be deprecated after supporting external
        EUQ db.
        
        return: 0 if successful, -1 if failed.
        """
        address = self.conf.getEuqDBAddress()
        port = self.conf.getEuqDBPort()
        username = self.conf.getEuqDBUsername()
        password = self.util.encryptString(self.conf.getEuqDBPassword())
        dbname = self.conf.getEuqDBName()
        sql = "insert into tb_euq_db_info (server,port,username,password,db_name,db_type,status) values (%s,%s,%s,%s,%s,'1',1)"
        para = [address, port, username, password, dbname]

        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    para)
        if ret != 0:
            self.logger.debug("Failed to insert the EUQ db information into the admin db, %s" %(utility.excToString(e)))
            return -1

        # Init tb_euq_table_info
        sql = "insert into tb_euq_table_info (table_id,db_id) values (%s,%s);"
        for tableId in xrange(128):
            para = [tableId, 1]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if e:
                self.logger.debug("Failed to insert table_id %d into tb_euq_table_info, %s" %(tableId, utility.excToString(e)))
                return -1

        return 0

    def registerDeviceToAdminDb(self, updater):
        """
        Register the device to admin db, it has to set the correct status of 
        each component, and it also has to set some configurations according 
        to component status.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        self.logger.debug("Try to register the current device to admin DB.")
        # Prepare data needed for registration.
        scannerId = self.getNextScannerId()
        if scannerId == -1:
            self.logger.debug("Failed to get the scanner id, cannot register the device to admin db.")
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1
        self.conf.setScannerId(scannerId)
        hostname = socket.gethostname()
        nicInfo = self.util.getNicInfo()
        if not nicInfo:
            self.logger.debug("Failed to get the IPv4 address or mac of the network interface.")
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1
        # For IMSS, use the first interface as ICP.
        if self.conf.getProduct() == 0:
            interface = nicInfo.keys()[0]
            self.logger.debug("Use %s as the ICP." %(interface))
            ip = nicInfo[interface][0]
            mac = nicInfo[interface][2]
        else:
            icp = self.getIcpName()
            if not icp:
                self.logger.debug("Cannot get ICP device name.")
                return -1
            self.logger.debug("The ICP is %s." %(icp))
            if icp not in nicInfo.keys():
                self.logger.debug("The ICP %s doesn't exist on this machine." %(icp))
                return -1
            ip = nicInfo[icp][0]
            mac = nicInfo[icp][2]

        if self.conf.getEnableScanner() == 1:
            daemonStatus = 2
            policyStatus = 2
        else:
            daemonStatus = 1
            policyStatus = 1
        if self.conf.getEnableEuqConsole() == 1:
            euqStatus = 2
        else:
            euqStatus = 1
        if self.conf.getEnableFoxhunter() == 1:
            nrsStatus = 2
            foxhunterStatus = 2
        else:
            nrsStatus = 1
            foxhunterStatus = 1
        if self.conf.getFreshInstall() == 1:
            isMaster = 1
        else:
            isMaster = 0
        osVer = self.getOsVersion()
        appVer = self.getAppVersion()

        sql = "insert into tb_component_list (scanner_id,scanner_name,ip_addr,daemon,policy,euq,nrs,ipprofiler,is_master,os_ver,app_ver,mac_addr) values (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)"
        para = [scannerId, hostname, ip, daemonStatus, policyStatus, euqStatus, nrsStatus, foxhunterStatus, isMaster, osVer, appVer, mac]
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    para)
        if ret != 0:
            self.logger.debug("Failed to execute sql command to register current device to admin db, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1

        # Update scanner id in imss.ini.
        iniPath = "%s/imss/config/imss.ini" %(self.conf.getInstallPath())
        (ret, e) = self.util.replaceInFile([(r"^scanner_id\s*=\s*.+", r"scanner_id=%d" %(scannerId))],
                                           iniPath)
        if ret != 0:
            self.logger.debug("Failed to update scanner_id in imss.ini, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1

        # Update scanner id in foxproxy.ini
        iniPath = "%s/imss/config/foxproxy.ini" %(self.conf.getInstallPath())
        (ret, e) = self.util.replaceInFile([(r"^scanner_id\s*=\s*.+", r"scanner_id=%d" %(scannerId))],
                                           iniPath)
        if ret != 0:
            self.logger.debug("Failed to update scanner_id in foxproxy.ini, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1

        # Update scanner id in EUQ's server.xml.
        xmlPath = "%s/imss/UI/euqUI/conf/server.xml" %(self.conf.getInstallPath())
        (ret, e) = self.util.replaceInFile([(r'jvmRoute=".+"', r'jvmRoute="euq%d"' %(scannerId))],
                                           xmlPath)
        if ret != 0:
            self.logger.debug("Failed to update scanner_id in EUQ's server.xml, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.registerDbFail"))
            return -1

        # If it is IMSVA, need to set has_foxlib_installed=1 in foxproxy.ini.
        if self.conf.getProduct() == 1:
            iniPath = "%s/imss/config/foxproxy.ini" %(self.conf.getInstallPath())
            (ret, e) = self.util.replaceInFile([(r"has_foxlib_installed=0", r"has_foxlib_installed=1")],
                                               iniPath)
            if ret != 0:
                self.logger.debug("Failed to set has_foxlib_installed in foxproxy.ini, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

        # If it is IMSVA, and it is a child, need to set cert for child mta
        if self.conf.getProduct() == 1 and isMaster == 0:
            #get cert id for master incoming mta
            sql = "select cert_id from tb_tls_service_certificate_usage where scanner_id in (select scanner_id from tb_component_list where is_master=1) and service_type = 0"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if ret == []:
                self.logger.debug("Failed to get cert_id for master incoming mta, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1
            master_cert_id = ret[0][0]
            #insert default cert id for child incoming mta
            sql = "insert into tb_tls_service_certificate_usage values(%s, '0', %s, '0')"
            para = [scannerId, master_cert_id]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

            #get cert id for master outgoing mta
            sql = "select cert_id from tb_tls_service_certificate_usage where scanner_id in (select scanner_id from tb_component_list where is_master=1) and service_type = 3"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if ret == []:
                self.logger.debug("Failed to get cert_id for master outgoing mta, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1
            master_cert_id = ret[0][0]
            #insert default cert id for child outgoing mta
            sql = "insert into tb_tls_service_certificate_usage values(%s, '3', %s, '0')"
            para = [scannerId, master_cert_id]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

        # Need to initial some data in tb_global_setting for central controller..
        if self.conf.getFreshInstall() == 1:
            sql = "insert into tb_global_setting values (%s,%s,%s,%s)"
            para = ["adminui", "port", "8445", "imss.ini"]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

            para = ["adminui", "scannerid", scannerId, "imss.ini"]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

            para = ["euqmaster", "port", "8447", "imss.ini"]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

            para = ["euqmaster", "scannerid", scannerId, "imss.ini"]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

            para = ["cmagent", "ConfigUrl", "https://%s:8445/" %(ip), "imss.ini"]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to execute sql command, %s" %(utility.excToString(e)))
                updater.append(self.res.getWording("msg.registerDbFail"))
                return -1

        updater.append(self.res.getWording("msg.registerDbSuccess"))
        return 0

    def getParentAddress(self):
        """
        Get parent address from admin DB.
        
        return: parent address, or -1 if failed to get parent address.
        """
        sql = "select ip_addr from tb_component_list where scanner_id=1;"
        (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(),
                                                   sql)
        if ret == None:
            self.logger.debug("Failed to get parent address, %s", utility.excToString(e))
            return -1

        return ret[0][0]
        
    def getSoapPort(self):
        """
        Get manager soap port from admin DB.
        
        return: manager soap port, or -1 if failed to get soap port.
        """
        sql = "select value from tb_global_setting where section='imss_manager' and name='listen_port';"
        (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(),
                                                   sql)
        if ret == None:
            self.logger.debug("Failed to get soap port, %s", utility.excToString(e))
            return -1

        return ret[0][0]
        
    def getOsVersion(self):
        """
        Get the current OS version via "uname -r" command, which is actually 
        the Linux kernel verison.
        
        return: the OS version, if failed, a default value 0.0.0 is returned. 
        """
        p = subprocess.Popen(["uname", "-r"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to get the OS version, stdout: %s, stderr: %s"
                              %(stdout, stderr))
            return "0.0.0"

        ret = stdout[0:stdout.find("-")]
        return ret

    def getAppVersion(self):
        """
        Get the application version via the "S99IMSS version" command.
        
        return: the version, e.g., 8.5.0.1266, if failed, return a default
        0.0.0.0000.
        """
        path = "%s/imss/script/S99IMSS" %(self.conf.getInstallPath())
        p = subprocess.Popen([path, "version"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug('Failed to execute "S99IMSS version", stdout: %s, stderr: %s'
                              %(stdout, stderr))
            return "0.0.0.0000"

        result = re.search(r"Version (\d+\.\d+)-Build_Linux_(\d{4})", stdout)
        if result == None:
            self.logger.debug("Failed to find the app version in command output, stdout: %s, stderr: %s"
                              %(stdout, stderr))
            return "0.0.0.0000"
        ret = result.group(1) + ".0." + result.group(2)
        return ret

    def getLocalScannerId(self):
        """
        Get the scanner id of current device from imss.ini.
        
        return: the scanner id, -1 if any error happens.
        """
        imssIniPath = "%s/imss/config/imss.ini" %(self.conf.getInstallPath())
        try:
            parser = ConfigParser.SafeConfigParser()
            parser.readfp(open(imssIniPath, "r"))
            return parser.getint("imss_manager", "scanner_id")
        except Exception, e:
            self.logger.debug("Fail to get local scanner ID from imss.ini, %s" %(utility.excToString(e)))
            return -1

    def getNextScannerId(self):
        """
        Get the next scanner id from admin DB.
        
        return: the next scanner id, or -1 if failed to get the scanner id.
        """
        sql = "select nextval('tb_scanner_id_seq') as nextid;"
        (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(),
                                                   sql)
        if ret == None:
            self.logger.debug("Failed to get the next scanner id, %s", utility.excToString(e))
            return -1

        return ret[0][0]

    def importDefAuComp(self, updater):
        """
        Import the default engines and patterns in to tb_active_update.
        
        updater: the status updater.
        
        return: 0 if successful, -1 if failed.
        """
        # Check if imssmgr has started, it shall be started. If it is not, wait 5 seconds 
        # and retry
        i = 0
        mgrOk = False
        mgrPidPath = "%s/imss/bin/imssmgr.pid" %(self.conf.getInstallPath())
        # Now wait at most 120 seconds
        while i < 24:
            time.sleep(5)
            if os.access(mgrPidPath, os.F_OK):
                mgrOk = True
                break
            i += 1

        if not mgrOk:
            self.logger.debug("Imssmgr cannot start within the time, cannot import engines/patterns to admin db.")
            updater.append(self.res.getWording("msg.importDefAuCompFail"))
            return -1

        # Insert engines and patterns into database.
        self.logger.debug("Try to insert default engines/patterns into admin db.")
        binPath = "%s/imss/script/S99UPDATE" %(self.conf.getInstallPath())
        p = subprocess.Popen([binPath, "import"],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to import default engines/patterns into database, stdout: %s, stderr: %s" %(stdout, stderr))
            updater.append(self.res.getWording("msg.importDefAuCompFail"))
            return -1

        return 0

    def initAdminDb(self):
        """
        Initialize admin db data (schemas and factory-settings)
        
        return: 0 if successful, -1 if failed.
        """
        address = self.conf.getAdminDBAddress()
        port = self.conf.getAdminDBPort()
        user = self.conf.getAdminDBUsername()
        password = self.util.decryptString(self.conf.getAdminDBPassword())
        dbname = self.conf.getAdminDBName()

        self.logger.debug("Try to create the db %s." %(dbname))
        # Create the admin db.
        sql = "CREATE DATABASE \"%s\" ENCODING 'unicode';" %(dbname)
        (ret, e) = self.util.executeUpdateStatement(address, port, user, password, "postgres", sql)
        if ret != 0:
            self.logger.debug("Failed to create database %s, %s" %(dbname, utility.excToString(e)))
            return -1

        # Create the procedure language if the language doesn't exist.
        # If the PostgreSQL has been upgraded to 9.1, we can use "CREATE OR REPLACE.." statement.
        sql = "SELECT * FROM pg_language WHERE lanname = 'plpgsql';"
        (ret, e) = self.util.executeQueryStatement(address, port, user, password, dbname, sql)
        if ret == []:
            sql = "CREATE LANGUAGE plpgsql;"
            (ret, e) = self.util.executeUpdateStatement(address, port, user, password, dbname, sql)
            if ret != 0:
                self.logger.debug("Failed to create the procedure language, %s" %(utility.excToString(e)))
                return -1
        elif e:
            self.logger.debug("Failed to check if language plpgsql exists, %s" %(utility.excToString(e)))
            return -1

        # Execute sql files.
        sqlPath = "%s/imss/sql/postgresql/install" %(self.conf.getInstallPath())
        createSqls = ["%s/create.sql" %(sqlPath)]
        insertSqls = ["%s/insert_named_obj.sql" %(sqlPath),
                      "%s/dlp_CTL.sql" %(sqlPath),
                      # The insert_imsa.sql in source code is appended to insert.sql during building IMSVA
                      # so we just need to execute insert.sql only here.
                      "%s/insert.sql" %(sqlPath),
                      "%s/insertKeywords.sql" %(sqlPath)]
        spSqls = []
        for item in os.listdir(sqlPath):
            path = os.path.join(sqlPath, item)
            if path.endswith("SP.sql") and os.path.isfile(path):
                spSqls.append(path)

        for item in createSqls:
            self.logger.debug("Try to execute sql file %s" %(item))
            (ret, e) = self.util.executeSqlFile(address, port, user, password, dbname, item)
            if ret != 0:
                self.logger.debug("Failed to execute sql file %s, %s" %(item, utility.excToString(e)))
                return -1
        for item in spSqls:
            self.logger.debug("Try to execute sql file %s" %(item))
            (ret, e) = self.util.executeSqlFile(address, port, user, password, dbname, item)
            if ret != 0:
                self.logger.debug("Failed to execute sql file %s, %s" %(item, utility.excToString(e)))
                return -1
        for item in insertSqls:
            self.logger.debug("Try to execute sql file %s" %(item))
            (ret, e) = self.util.executeSqlFile(address, port, user, password, dbname, item)
            if ret != 0:
                self.logger.debug("Failed to execute sql file %s, %s" %(item, utility.excToString(e)))
                return -1

        # For IMSVA, import local users to adminDB.
        if self.conf.getProduct() == 1:
            if self.importLocalUsers() != 0:
                self.logger.debug("Fail to import local users to adminDB.")
                return -1
        
        # Generate the DLPComplianceTemplateFullXML record in tb_named_obj.
        ret = subprocess.call(["/bin/sh", "%s/imss/script/UpdateDlpTemplate.sh" %(self.conf.getInstallPath())],
                              stdout = open("/dev/null"),
                              stderr = open("/dev/null"))
        if ret != 0:
            self.logger.debug("Fail to update DLPComplianceTemplateFullXML in tb_named_obj.")
            return -1
            
        return 0


    def checkDbUserRole(self, address, port, username, password):
        """
        Check if user has superuser role
        
        address: address of the db.
        port: port of the db.
        username: the username can be used to access db.
        password: the password can be used to access db.
        
        return: 0 if user is superuser, otherwise return -1.
        """
        sql = "select * from pg_roles where rolname=%s and rolsuper='t';"
        (ret, e) = self.util.executeQueryStatement(address,
                                                   port,
                                                   username,
                                                   password,
                                                   "postgres",
                                                   sql,
                                                   [username])
        if e == None:
            # Non-empty result means exist
            if ret:
                return 0
            else:
                return -1
        else:
            self.logger.debug("Failed to check whether db exist, %s" %(utility.excToString(e)))
            return -1

    def checkDbActivity(self, address, port, username, password, dbname):
        """
        Check if there's any client connected to a specific db.
        
        address: address of the db.
        port: port of the db.
        username: the username can be used to access db.
        password: the password can be used to access db.
        dbname: the dbname that will be checked.
        
        return: 0 if no one has connected to the db, otherwise return -1.
        """
        sql = "select datname from pg_stat_activity where datname=%s;"
        (ret, e) = self.util.executeQueryStatement(address,
                                                   port,
                                                   username,
                                                   password,
                                                   "postgres",
                                                   sql,
                                                   [dbname])
        if e == None:
            # Non-empty result means someone is connected to the db. 
            if ret:
                return -1
            else:
                return 0
        else:
            self.logger.debug("Failed to check whether there's someone connecting the db, %s" %(utility.excToString(e)))
            return -1

    def checkDbExist(self, address, port, username, password, dbname):
        """
        Check if database exists
        
        address: address of the db.
        port: port of the db.
        username: the username can be used to access db.
        password: the password can be used to access db.
        dbname: the dbname that will be checked.
        
        return: 0 if no one has connected to the db, otherwise return -1.
        """
        sql = "select datname from pg_database where datname=%s;"
        (ret, e) = self.util.executeQueryStatement(address,
                                                   port,
                                                   username,
                                                   password,
                                                   "postgres",
                                                   sql,
                                                   [dbname])
        if e == None:
            # Non-empty result means exist
            if ret:
                return -1
            else:
                return 0
        else:
            self.logger.debug("Failed to check whether db exist, %s" %(utility.excToString(e)))
            return -1

    def installBundledDatabase(self):
        """
        Init the bundled database. These are configurations initializations, 
        not related to data.
        
        return: 0 if successful, -1 if failed.
        """
        pgRootPath = "%s/imss/PostgreSQL" %(self.conf.getInstallPath())
        pgLibPath = "%s/lib" %(pgRootPath)
        pgSharePath = "%s/share" %(pgRootPath)
        # For IMSVA, the real db data path is /var/app_data/imss/db, /var/imss 
        # is a link to it.
        pgDataRootPath = os.path.normpath("/var/imss")
        if self.conf.getProduct() == 1:
            pgDataRootPathLink = os.path.normpath("/var/imss")
            pgDataRootPath = os.path.normpath("/var/app_data/imss/db")
        pgDataPath = "%s/pgdata" %(pgDataRootPath)
        pgBinPath = "%s/bin" %(pgRootPath)
        pgSharePgPath = "%s" %(pgSharePath)
        pgConfPath = "%s/postgresql.conf" %(pgDataPath)
        pgAuthConfPath = "%s/pg_hba.conf" %(pgDataPath)

        # Create the database data folder.
        self.logger.debug("Try to create db data folder %s" %(pgDataPath))

        #for IMSS shouldn't remove the root dir. bacause it's backup folder.
        #otherwise all backup files will be delete
        if self.conf.getProduct() == 1:
            if os.path.lexists(pgDataRootPath):
                (ret, e) = self.util.removePath(pgDataRootPath)
                if ret != 0:
                    self.logger.debug("Cannot delete %s, %s" %(pgDataRootPath, utility.excToString(e)))
                    return -1

        (ret, e) = self.util.createDir(pgDataPath, 0700)
        if ret != 0:
            self.logger.debug("Cannot create PostgreSQl data folder %s, %s"
                              %(pgDataPath, utility.excToString(e)))
            return -1

        # For IMSVA, create the link /var/imss to /var/app_data/imss/db
        if self.conf.getProduct() == 1:
            self.logger.debug("Try to create a link from %s to %s." %(pgDataRootPath, pgDataRootPathLink))
            if os.path.lexists(pgDataRootPathLink):
                (ret, e) = self.util.removePath(pgDataRootPathLink)
                if ret != 0:
                    self.logger.debug("Cannot delete %s, %s" %(pgDataRootPathLink, utility.excToString(e)))
                    return -1
            try:
                os.symlink(pgDataRootPath, pgDataRootPathLink)
            except Exception, e:
                self.logger.debug("Fail to create a link from %s to %s." %(pgDataRootPath, pgDataRootPathLink))
                return -1

        # Change its owner to imss.
        # Call shell command directly because the python chown doesn't have -R ability.
        self.logger.debug("Try to modify the owner of %s to imss." %(pgDataRootPath))
        p = subprocess.Popen(["chown", "-R", "imss:imss", pgDataRootPath],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to change owner of folder %s, stdout: %s, stderr: %s"
                              %(pgDataRootPath, stdout, stderr))
            return -1

        # Initialize the database cluster.
        self.logger.debug("Try to init the db cluster.")
        binPath = "%s/initdb" %(pgBinPath)
        p = subprocess.Popen(["su", "-m", "imss", "-c", "%s -L %s -D %s --locale=en_US.UTF-8" %(binPath, pgSharePgPath, pgDataPath)],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE,env=utility.getEnvForRunCmd(self.conf))
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to initialize the database cluster, stdout: %s, stderr: %s"
                              %(stdout, stderr))
            return -1

        # Allow tcp/ip connection
        self.logger.debug("Try to config db tcp connection settings.")
        # IMSVA
		# enable idle_in_transaction_session_timeout, this is because we use lock in transaction. 
		# If the process is crash/kill before commit. there is always one "idle transaction" left.
		# this prevents other process to get the lock.
        if self.conf.getProduct() == 1:
            rules = [(r"max_connections =.*", r"max_connections = 600"),
                     (r"#max_locks_per_transaction =.*#", r"max_locks_per_transaction = 256\t#"),
                     (r"#idle_in_transaction_session_timeout =.*#", r"idle_in_transaction_session_timeout = 3600000\t#") ]
        # IMSS
        else:
            rules = [(r"#tcpip_socket =.*", r"tcpip_socket = true"),
                     (r"max_connections =.*", r"max_connections = 600"),
                     (r"#max_locks_per_transaction =.*#", r"max_locks_per_transaction = 256\t#"),
                     (r"#idle_in_transaction_session_timeout =.*#", r"idle_in_transaction_session_timeout = 3600000\t#") ]
        (ret, e) = self.util.replaceInFile(rules, pgConfPath)
        if ret != 0:
            self.logger.debug("Failed to modify connection settings in %s, %s" %(pgConfPath, utility.excToString(e)))
            return -1

        # Modify PostgreSQL log setting for IMSVA
        self.logger.debug("Try to modify db log settings.")
        if self.conf.getProduct() == 1:
            rules = [(r"^#logging_collector =.*#", r"logging_collector = on\t\t\t#"),
                     (r"^#log_filename =.*#", r"log_filename = 'postgresql.log.%a'\t#"),
                     (r"^#log_truncate_on_rotation =.*#", r"log_truncate_on_rotation = on\t\t#"),
                     (r"^#log_line_prefix =.*#", r"log_line_prefix = '<%t> '\t\t#")]
            (ret, e) = self.util.replaceInFile(rules, pgConfPath)
            if ret != 0:
                self.logger.debug("Failed to modify log settings in %s, %s" %(pgConfPath, utility.excToString(e)))
                return -1

        # Modify PostgreSQL client authentication configurations.
        self.logger.debug("Try to modify db client authentication settings.")
        # For IMSVA, users from tcp connections use password authentication.
        if self.conf.getProduct() == 1:
            line = "host    all         all         0/0         md5"
        # For IMSS, only users from the subnet use password authentication.
        # Connections from other subnets shall be denied by default.
        else:
            nicInfo = self.util.getNicInfo()
            if not nicInfo:
                self.logger.debug("Failed to get IP address and mask.")
                return -1
            # For IMSS, use the first interface as ICP.
            interface = nicInfo.keys()[0]
            self.logger.debug("Use %s as the ICP." %(interface))
            ip = nicInfo[interface][0]
            mask = nicInfo[interface][1]
            line = "host    all         all         %s      %s   md5" %(ip, mask)

        try:
            with open(pgAuthConfPath, "a") as f:
                f.write(line)
        except Exception, e:
            self.logger.debug("Failed to modify %s, %s" %(pgAuthConfPath), utility.excToString(e))
            return -1

        # For IMSS, make sure system link to correct library.
        if self.conf.getProduct() == 0:
            p = subprocess.Popen(["/sbin/ldconfig", pgLibPath],
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Failed to configure dynamic link, stdout: %s, stderr: %s"
                                  %(stdout, stderr))
                return -1

        # If the user chooses to use external admin db or append install to an existing parent, 
        # local bundled db uses default user name and password.
        if self.conf.getAdminDBType() == 1 or self.conf.getFreshInstall() == 0:
            userName = "sa"
            passwd = "!CRYPT!20D31871A50579F0E15C2A17D69103AF10B09775C1F"
        else:
            userName = self.conf.getAdminDBUsername()
            passwd = self.conf.getAdminDBPassword()


        #if is imss upgrade
        #if parent  use adminDB user and password
        #if child     use scanner_id to get local databse user and password

        #self.logger.debug("-------------->[debug] conf user:%s psw:%s decrypt:%s"%(self.conf.getAdminDBUsername(),
        #                                                                           self.conf.getAdminDBPassword(),self.util.decryptString(self.conf.getAdminDBPassword())))

        if g_upd_models.mode == Const.MODE_IMSS_UPGRADE:
            scanner_id = self.getLocalScannerId()
            self.logger.debug("Imss upgrade, prepare for create db user. scanner_id:%s"%scanner_id)
            #parent
            if scanner_id ==1:
                # has internal db. user and password follow it
                if self.util.isUsingInternalAdminDB(g_upd_models.rpm_status):
                    userName = self.conf.getAdminDBUsername()
                    passwd = self.conf.getAdminDBPassword()
                #only has internal euqdb. user and password follow it.
                elif self.util.isUsingInternalDB() and not self.util.isUsingInternalAdminDB(g_upd_models.rpm_status):
                    userName = self.conf.getEuqDBUsername()
                    passwd = self.conf.getEuqDBPassword()
            else:
                #if child and has euqdb
                # if there has old euq db use previous username and password
                db_mgr = g_upd_models.db_mgr
                if db_mgr.is_need_euqdb_dump:
                    userName = db_mgr.local_euq_db_info.username
                    passwd = db_mgr.local_euq_db_info.password
                # if there is no euqdb use default username and password
                else:
                    userName = "sa"
                    passwd = "!CRYPT!20D31871A50579F0E15C2A17D69103AF10B09775C1F"
        if os.path.exists(Const.DEBUG_FLAG_FILE):
            self.logger.debug("Create db user:%s password:%s %s"%(userName,passwd,self.util.decryptString(passwd)))

        self.logger.debug("Try to create db user %s." %(userName))
        # Start up PostgreSQL and create the user.
        dbctlPath = "%s/imss/script/dbctl.sh" %(self.conf.getInstallPath())
        # Not check the return code of this script, if PostgreSQL is already running,
        # The script will return a non-zero value.
        ctlPostgreSQL("start", self.conf)
        # Execute command to create user.
        binPath = "%s/createuser" %(pgBinPath)
        p = subprocess.Popen(["su", "-m", "imss", "-c", "%s -s -d %s" %(binPath, userName)],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE, env=utility.getEnvForRunCmd(self.conf))
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Failed to create db user, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1
        self.logger.debug("Succeed to create db user.")
        # Modify the password of db user.
        sql = "ALTER ROLE \"%s\" WITH PASSWORD %%s" %(userName)
        para = [self.util.decryptString(passwd)]
        #self.logger.debug("-------------------->[debug]. user:%s psw:%s decrypt:%s"%(userName,passwd, str(para)))
        (ret, e) = self.util.executeUpdateStatement(None, 5432, userName, None, "postgres", sql, para)
        if ret != 0:
            self.logger.debug("Failed to change %s's password, %s" %(userName, utility.excToString(e)))
            return -1
        self.logger.debug("Succeed to change user password.")
        
        # For IMSS, enforce all access to DB using password,
        # not apply this enforcement for IMSVA, since DB is protected by firewall
        # Don't put this before createuser command, otherwise createuser command will wait password input
        if self.conf.getProduct() == 0:
            rules = [(r"trust$", r"password")]
            (ret, e) = self.util.replaceInFile(rules, pgAuthConfPath)
            if ret != 0:
                self.logger.debug("Failed to modify %s, %s" %(pgAuthConfPath), utility.excToString(e))
                return -1
            else:
                ctlPostgreSQL("restart", self.conf)

        return 0

    def checkUsersAndGroups(self, updater):
        """
        Check if IMSx needed users or groups already exists on local machine.
        
        updater: the status updater.
        
        return: 0 if no user or group exists on local machine, -1 if some user 
                and group exists on local machine.
        """
        ret = 0
        userInfoList = self.conf.getUsers()
        groupInfoList = self.conf.getGroups()
        existingUserNames = []
        existingGroupNames = []

        for index, (name, group, home, shell, _) in enumerate(userInfoList):
            if self.util.isUserExist(name):
                self.logger.debug("User %s already exist on this machine." %(name))
                existingUserNames.append(name)
                userInfoList[index] = (name, group, home, shell, 1)

        for index, (name, _) in enumerate(groupInfoList):
            if self.util.isGroupExist(name):
                self.logger.debug("Group %s already exist on this machine." %(name))
                existingGroupNames.append(name)
                groupInfoList[index] = (name, 1)

        if existingUserNames:
            ret = -1
            updater.append(self.res.getWording("msg.userExists"))
            for user in existingUserNames:
                updater.append("%s %s" %(self.res.getWording("msg.listBullet"), user))
        if existingGroupNames:
            ret = -1
            updater.append(self.res.getWording("msg.groupExists"))
            for group in existingGroupNames:
                updater.append("%s %s" %(self.res.getWording("msg.listBullet"), group))
        self.conf.setUsers(userInfoList)
        self.conf.setGroups(groupInfoList)
        self.logger.debug("User info: %s" %(self.conf.getUsers()))
        self.logger.debug("Group info: %s" %(self.conf.getGroups()))

        return ret

    def checkFoxHunterPort(self, updater):
        """
        Check the validity of foxhunter listening port.
        
        updater: the status updater.
        
        return: if all ports are free, return 0, otherwise return -1.
        """
        port = self.conf.getFoxhunterListenPort()
        # First check if foxhunter listen port conflicts with reserved ports.
        if ((port in self.conf.getPortsFoxhunter()) or
            (port in self.conf.getPortsCentralController()) or
            (port in self.conf.getPortsScanner()) or
            (port in self.conf.getPortsEuq()) or
            (port in self.conf.getPortsEuqMaster())
            ):
            self.logger.debug("The foxhunter listen port %d conflicts with IMSx reserved ports." %(port))
            updater.append(self.res.getWording("msg.foxhunterPortConflict") %(port))
            return -1

        # Then check if the foxhunter listen port is being used.
        if self.util.isConnectable("127.0.0.1", port)[0]:
            self.logger.debug("Foxhunter cannot listen on port %d, it is used by other process" %(port))
            updater.append(self.res.getWording("msg.foxhunterPortUsed") %(port))
            return -1
        
        return 0

    def checkAvailablePorts(self, updater):
        """
        Check if the IMSx needed ports are occupied or not.
        Now only check the IPv4 connection. Foxhunter related ports are 
        not checked in this function, use checkFoxHunterPort instead.
        
        updater: the status updater.
        
        return: if all ports are free, return 0, otherwise return -1.
        """
        ret = 0
        usedPorts = []
        
        # Check ports needed by all IMSx components.
        for port in self.conf.getPortsAll():
            if self.util.isConnectable("127.0.0.1", port)[0]:
                self.logger.debug("Port %d is needed by IMSx, now it is occupied by another program." %(port))
                usedPorts.append(port)

        # Check ports needed by central controller.
        for port in self.conf.getPortsCentralController():
            if self.util.isConnectable("127.0.0.1", port)[0]:
                self.logger.debug("Port %d is needed by IMSx, now it is occupied by another program." %(port))
                usedPorts.append(port)

        # Check ports needed by scanner.
        for port in self.conf.getPortsScanner():
            if self.util.isConnectable("127.0.0.1", port)[0]:
                self.logger.debug("Port %d is needed by IMSx, now it is occupied by another program." %(port))
                usedPorts.append(port)

        # Check ports needed by EUQ console.
        for port in self.conf.getPortsEuq():
            if self.util.isConnectable("127.0.0.1", port)[0]:
                self.logger.debug("Port %d is needed by IMSx, now it is occupied by another program." %(port))
                usedPorts.append(port)
        # Check ports needed by EUQ master.
        for port in self.conf.getPortsEuqMaster():
            if self.util.isConnectable("127.0.0.1", port)[0]:
                self.logger.debug("Port %d is needed by IMSx, now it is occupied by another program." %(port))
                usedPorts.append(port)
                    
        if usedPorts:
            ret = -1
            updater.append(self.res.getWording("msg.portUsed") %(self.res.getWording("msg.portSep").join(str(x) for x in usedPorts)))

        return ret

    def confirmCompEnableStatus(self):
        """
        Check the enable status of components, correct the status if some
        dependencies are not ready.
        
        return: no return.
        """
        if self.conf.getFreshInstall() == 0:
            self.conf.setEnableCentralController(0)
        self.logger.debug("Central controller enable status: %d"
                          %(self.conf.getEnableCentralController()))
        self.logger.debug("Scanner enable status: %d"
                          %(self.conf.getEnableScanner()))
        self.logger.debug("EUQ web console enable status: %d"
                          %(self.conf.getEnableEuqConsole()))
        self.logger.debug("Foxhunter enable status: %d"
                          %(self.conf.getEnableFoxhunter()))

    def checkDatabaseConfiguration(self, updater):
        """
        Check the database configurations.
        
        First the database configurations must be complete.
        
        1. Fresh install:
            a) If using bundled db, need to check whether there's another 
               process listens on the database port.
            b) If using external db, it must be remote, it cannot be on the
               same machine where IMSS is installed. Besides, need to check 
               whether the db can be connected, and the db with the configured 
               name must NOT exist, otherwise the installation cannot continue. 
        2. Append install:
           Need to check the db can be connected, and the db with the 
           configured name must exist.
        
        updater: the status updater.
        
        return: if no error return 0, otherwise return -1.
        """
        self.logger.debug("Database configurations: host=%s, port=%d, "
                          "user=%s, dbname=%s" %(self.conf.getAdminDBAddress(),
                                                 self.conf.getAdminDBPort(),
                                                 self.conf.getAdminDBUsername(),
                                                 self.conf.getAdminDBName()))

        # First check if all necessary configurations are not empty.
        if (self.conf.getAdminDBAddress() and self.conf.getAdminDBPort() and
            self.conf.getAdminDBName() and self.conf.getAdminDBUsername() and
            self.util.decryptString(self.conf.getAdminDBPassword())):
            # Check db name format, besides, adminDB name cannot colide with EUQ db.
            if not re.match(r'^[0-9a-zA-Z-_.@]+$', self.conf.getAdminDBName()) or self.conf.getAdminDBName() == self.conf.getEuqDBName():
                updater.append(self.res.getWording('msg.invalidDbName'))
                return -1
            
            # Check db user name format
            if not re.match(r'^[0-9a-zA-Z-_.@]+$', self.conf.getAdminDBUsername()):
                updater.append(self.res.getWording('msg.invalidDbUser'))
                return -1
            
            # Check db password format
            if not re.match(r"^[0-9a-zA-Z`~!@#$%^&*()[\]\{\}\+\-\|:'<>\?/,\.=_]+$", self.util.decryptString(self.conf.getAdminDBPassword())):
                updater.append(self.res.getWording('msg.invalidDbPassword'))
                return -1
            
            # Fresh install
            if self.conf.getFreshInstall() == 1:
                # Bundled DB.
                if self.conf.getAdminDBType() == 0:
                    self.logger.debug("The user chooses to use bundled db.")
                    # Normally, the address shall be 127.0.0.1.
                    if self.util.isConnectable(self.conf.getAdminDBAddress(),
                                               self.conf.getAdminDBPort())[0]:
                        self.logger.debug("Another process is on %s:%d, cannot install "
                                          "a new database on this machine."
                                          %(self.conf.getAdminDBAddress(), self.conf.getAdminDBPort()))
                        updater.append(self.res.getWording("msg.localDbPortOccupied") %(self.conf.getAdminDBPort()))
                        return -1
                # External DB.
                elif self.conf.getAdminDBType() == 1:
                    self.logger.debug("The user chooses to use external db.")
                    # Check if the db address is the current machine.
                    if self.util.isLocalAddress(self.conf.getAdminDBAddress()):
                        self.logger.debug("The external db cannot be on the same machine as IMSx.")
                        updater.append(self.res.getWording("msg.externalDbMustRemote"))
                        return -1
                    # Check if db can be connected.
                    (ret, exception) = self.util.isDatabaseConnectable(
                                                    self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()))
                    # DB can be connected.
                    if ret == 0:
                        self.logger.debug("Database can be connected.")
                        # check user role (need superuser role)
                        ret = self.checkDbUserRole(
                                      self.conf.getAdminDBAddress(),
                                      self.conf.getAdminDBPort(),
                                      self.conf.getAdminDBUsername(),
                                      self.util.decryptString(self.conf.getAdminDBPassword()))
                        if ret == 0:
                            self.logger.debug("The user %s has superuser role."
                                              % (self.conf.getAdminDBUsername()))
                        else:
                            self.logger.debug("The user %s does not have superuser role!"
                                              % (self.conf.getAdminDBUsername()))
                            updater.append(self.res.getWording("msg.externalDbUserrole"))
                            return -1

                        (ret, exception) = self.util.isDatabaseConnectable(
                                                    self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName())
                        # The db exists.
                        if ret == 0:
                            self.logger.debug("The database %s exists on host %s, "
                                              "it must be removed before proceeding."
                                              %(self.conf.getAdminDBName(), self.conf.getAdminDBAddress()))
                            updater.append(self.res.getWording("msg.dbExistForFreshInstall") %(self.conf.getAdminDBName()))
                            return -1
                        # The db doesn't exists.
                        else:
                            self.logger.debug("The database %s doesn't exit, "
                                              "will use this db name."
                                              %(self.conf.getAdminDBName()))
                            return 0
                    # Cannot connect admin db.
                    else:
                        self.logger.debug("Cannot connect to the admin db, %s"
                                          %(utility.excToString(exception)))
                        updater.append(self.res.getWording("msg.dbCannotConnect"))
                        return -1
            # Append install
            elif self.conf.getFreshInstall() == 0:
                (ret, exception) = self.util.isDatabaseConnectable(
                                                    self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName())
                # The db exists.
                if ret == 0:
                    self.logger.debug("The admin db can be connected.")
                    return 0
                # The db cannot be accessed.
                else:
                    self.logger.debug("Cannot connect to the admin db, %s"
                                      %(utility.excToString(exception)))
                    updater.append(self.res.getWording("msg.dbCannotConnect"))
                    return -1

        # Database configurations are incomplete.
        else:
            self.logger.debug("Database configurations are incomplete.")
            updater.append(self.res.getWording("msg.dbConfigIncomplete"))
            return -1

        return 0

    def checkAvailableMemSize(self, updater):
        """
        Check whether the memory size is sufficient to install imsx.
        
        updater: the status updater.
        
        return: if size is sufficient, return 0, otherwise return -1.
        """
        memSize = self.util.getMemSize()
        # This is tricky. We won't compare the actual available memory size with threshold. Instead, add 500MB and then 
        # compare. Because the available memory is always less than the real physical memory, we do this to eliminate the 
        # warning on the machine that has the physical memory just the same size with the threshold.
        if memSize + 500*1024*1024 < self.conf.getMinMemSize():
            self.logger.debug("The memory size %d bytes is less than "
                              "required memory size %d bytes."
                              %(memSize, self.conf.getMinMemSize()))
            # Change size to human-readable MB format.
            updater.append(self.res.getWording("msg.insufficientMemory")
                           %(memSize/1024/1024, self.conf.getMinMemSize()/1024/1024))
            return -1
        else:
            self.logger.debug("The memory size %d bytes is not less than "
                              "required memory size %d bytes."
                              %(memSize, self.conf.getMinMemSize()))
            return 0
        
    def checkTotalDiskSize(self, updater):
        """
        Check whether the total disk size of this machine can holds IMSx running data.
        
        updater: the status updater
        
        return: 0 if meets minimal requirement, -1 otherwise
        """
        totalDiskSize = self.util.getTotalDiskSize()
        if totalDiskSize < 0:
            self.logger.debug("Fail to get the total disk size.")
            # If fail to get the disk size, treat as meet requirement.
            return 0
        
        self.logger.debug("The total disk size is %d bytes." %(totalDiskSize))
        if totalDiskSize < self.conf.getMinDiskSize():
            updater.append(self.res.getWording("msg.smallDiskSize") %(self.conf.getMinDiskSize()/1024/1024/1024))
            return -1
        
        return 0

    def checkInstallPath(self, updater):
        """
        Check whether the install path is valid.
        
        updater: the status updater.
        
        return: if valid return 0, otherwise return -1.
        """
        # Check format
        if not re.search(r"^[0-9a-zA-Z/_\-.+]+$", self.conf.getInstallPath()):
            updater.append(self.res.getWording("msg.invalidInstallPathFormat"))
            return -1
        
        # Check if it is an absolute path
        if not os.path.isabs(self.conf.getInstallPath()):
            self.logger.debug("The install path %s is not a valid absolute path." %(self.conf.getInstallPath()))
            updater.append(self.res.getWording("msg.invalidInstallPathNotAbs"))
            return -1
        
        # Check free disk space
        if self.checkAvailableDiskSize(updater) != 0:
            return -1
        
        return 0

    def checkAvailableDiskSize(self, updater):
        """
        Check whether the free disk size is sufficient to install imsx.
        
        updater: the status updater.
        
        return: if size is sufficient return 0, otherwise return -1.
        """
        avaiDiskSize = self.util.getDiskFreeSize(self.conf.getInstallPath())
        if avaiDiskSize < self.conf.getMinFreeDiskSize():
            self.logger.debug("The available disk size %d bytes is less than "
                              "required minimum disk size %d bytes."
                              %(avaiDiskSize, self.conf.getMinFreeDiskSize()))
            # Change size from bytes to human-readable MB format.
            updater.append(self.res.getWording("msg.insufficientDisk")
                           %(self.conf.getMinFreeDiskSize()/1024/1024))
            return -1
        else:
            self.logger.debug("The available disk size %d bytes is not less than "
                              "required minimum disk size %d bytes."
                              %(avaiDiskSize, self.conf.getMinFreeDiskSize()))
            return 0

    def checkInstallFolderContent(self, updater):
        """
        Check whether the install folder exits, if it exists, it will be 
        removed.
        
        return: return 0 if the folder doesn't exist, -1 if it exits.
        """
        ret = 0
        for aPath in self.conf.getRemovePathsBeforeInstallation():
            if os.path.exists(aPath):
                updater.append(self.res.getWording("msg.installPathExist") %(aPath))
                ret = -1

        return ret

    def checkInstalledImsxRpms(self):
        """
        Check the installed imsx rpms on this machine.
        
        return: return a list of installed rpms' names, if none rpm is installed,
        an empty list is returned.
        """
        result = []

        for rpm in self.conf.getRpmNames():
            if self.util.isRpmInstalled(rpm):
                self.logger.debug("Find installed rpm %s." %(rpm))
                result.append(rpm)

        return result


    def installImsxRpms(self, updater):
        """
        Install the necessary rpms, rpms of IMSS and IMSVA are different.
        The file paths of these rpms must be specified in the configuration
        file. See config module for details.
        
        updater: the status updater object used to display progress.
        
        return: 0 if all rpms are installed successfully, -1 if any of rpms 
        fail to install.
        """
        # Remove paths that must be removed before installation.
        for aPath in self.conf.getRemovePathsBeforeInstallation():
            (ret, e) = self.util.removePath(aPath)
            if ret != 0:
                self.logger.debug("Fail to remove %s, %s" %(aPath, utility.excToString(e)))
                updater.append(self.res.getWording("msg.installPathRemoveFail") %(aPath))
                return -1
        self.logger.debug("Install path:%s RpmPaths:%s RemovePaths:%s" %
                          (self.conf.getInstallPath(), self.conf.getRpmPaths(),
                           self.conf.getRemovePathsBeforeInstallation()))
        for path in self.conf.getRpmPaths():
            if not os.path.exists(path):
                self.logger.debug("File %s doesn't exits." %(path))
                updater.append(self.res.getWording("msg.rpmFileNotExist") %(path))
                return -1

            self.logger.debug("Start to install %s." %(path))
            updater.append(self.res.getWording("msg.rpmInstallBegin") %(os.path.basename(path)))
            if self.conf.getProduct() == 0:
                (ret, stdout, stderr) = self.util.installRpm(path, ["--force", "--prefix", self.conf.getInstallPath()])
            # IMSVA doesn't need to specify prefix.
            else:
                (ret, stdout, stderr) = self.util.installRpm(path, ["--force"])
            if ret != 0:
                self.logger.debug("Failed to install rpm %s, stdout: %s, stderr: %s"
                                  %(path, stdout, stderr))
                updater.append(self.res.getWording("msg.rpmInstallFail") %(path, stderr))
                return -1
            if self.util.isOSRedHat9():
                self.util.addRedHat9Dependency()
            self.logger.debug("%s is installed successfully." %(path))
            updater.append(self.res.getWording("msg.rpmInstallSuccess") %(path))

        return 0

    def uninstallImsxRpms(self, updater):
        """
        Uninstall IMSx rpm packages.
        
        updater: the status updater.
        
        return: 0 if all IMSx rpms are removed, -1 if any of them failed to uninstall.
        """
        # Uninstall in a reversed order.
        for rpm in reversed(self.conf.getRpmNames()):
            self.logger.debug("Try to uninstall rpm %s." %(rpm))
            if self.util.isRpmInstalled(rpm):
                updater.append(self.res.getWording("msg.rpmUninstallBegin") %(rpm))
                (ret, stdout, stderr) = self.util.uninstallRpm(rpm, ["--nodeps"])
                if ret != 0:
                    self.logger.debug("Fail to uninstall rpm %s, stdout: %s, stderr: %s" %(rpm, stdout, stderr))
                    updater.append(self.res.getWording("msg.uninstallRpmFail") %(rpm, stderr))
                    return -1
                else:
                    updater.append(self.res.getWording("msg.uninstallRpmSuccess") %(rpm))
            else:
                self.logger.debug("The rpm %s is not installed, skip it." %(rpm))

        return 0

    def testDb(self, addr, port, username, password, dbname):
        """
        Test if the specified database can be connected or not.
        
        addr: the address of the database.
        port: the port of the database.
        username: the user name of the database.
        password: the encrypted password of the database.
        dbname: the name of the database.
        
        return: 0 if it can be connected, -1 if failed.
        """
        (ret, e) = self.util.isDatabaseConnectable(addr, int(port), username, self.util.decryptString(password), dbname)
        if ret == 0:
            self.logger.debug("The database %s can be connected." %(dbname))
            return 0
        else:
            self.logger.debug("The database %s cannot be connected, %s" %(dbname, utility.excToString(e)))
            return -1

    def addTrustIp(self, ip):
        """
        Add an ip into trusted IP list if it is not in it.
        
        ip: the ip to be added.
        
        return: 0 if successful, -1 if failed.
        """
        addr = self.conf.getAdminDBAddress()
        port = self.conf.getAdminDBPort()
        user = self.conf.getAdminDBUsername()
        passwd = self.util.decryptString(self.conf.getAdminDBPassword())
        dbname = self.conf.getAdminDBName()

        sqlCheck = "select count(*) from tb_trusted_ip_list where ip_addr=%s"
        (ret, e) = self.util.executeQueryStatement(addr, int(port), user, passwd, dbname, sqlCheck, [ip])
        if e:
            self.logger.debug("Failed to query trust ip count, %s" %(utility.excToString(e)))
            return -1
        else:
            if ret and ret[0][0] > 0:
                self.logger.debug("IP %s is already in trust ip list, no need to add." %(ip))
                return 0

        # Add the ip.
        sqlAdd = "insert into tb_trusted_ip_list (ip_addr) values (%s)"
        (ret, e) = self.util.executeUpdateStatement(addr, int(port), user, passwd, dbname, sqlAdd, [ip])
        if ret != 0:
            self.logger.debug("Fail to add %s into trust list, %s." %(ip, utility.excToString(e)))
            return -1
        else:
            self.logger.debug("IP %s is added into trust list." %(ip))

        return 0

    def printParentVersions(self, addr, port, user, passwd, dbname):
        """
        Print the OS and app versions in below format:
        os_ver app_ver app_build
        
        addr: the address of the database.
        port: the port of the database.
        user: the username of the database.
        passwd: the password of the database.
        dbname: the name of the database.
        
        return: always return 0.
        """
        passwd = self.util.decryptString(passwd)

        sql = "select os_ver, app_ver from tb_component_list where scanner_id=1;"
        (ret, e) = self.util.executeQueryStatement(addr, int(port), user, passwd, dbname, sql)
        if e:
            self.logger.debug("Fail to query versions from db, %s" %(utility.excToString(e)))
            return 0
        else:
            if len(ret[0]) == 2:
                osVer = ret[0][0]
                appVer = ret[0][1]
                if osVer.count(".") < 2 or appVer.count(".") != 3:
                    print "Invalid version format!"
                    return 0
                else:
                    osVerSplit = osVer.split(".")
                    appVerSplit = appVer.split(".")
                    print "%s.%s %s.%s %s" %(osVerSplit[0], osVerSplit[1], appVerSplit[0], appVerSplit[1], appVerSplit[-1])
                    return 0
            else:
                self.logger.debug("The database returns invalid result %s" %(ret))
                return 0

    def printSoapSvcPort(self, addr, port, user, passwd, dbname):
        """
        Print soap service port, default 15505
        
        addr: the address of the database.
        port: the port of the database.
        user: the username of the database.
        passwd: the password of the database.
        dbname: the name of the database.
        
        return: always return 0.
        """
        passwd = self.util.decryptString(passwd)

        sql = "select value from tb_global_setting where section='imss_manager' and name='listen_port';"
        (ret, e) = self.util.executeQueryStatement(addr, int(port), user, passwd, dbname, sql)
        if e:
            self.logger.debug("Fail to query soap port from db, %s" %(utility.excToString(e)))
            return 0
        else:
            if ret:
                print "%s" % (ret[0][0])
                return 0
            else:
                self.logger.debug("The database returns invalid result %s" %(ret))
                return 0

    def checkIpInCompList(self, addr, port, user, passwd, dbname, ip):
        """
        Check if the specified IP address already in tb_component_list.
        
        addr: the address of the database.
        port: the port of the database.
        user: the username of the database.
        passwd: the password of the database.
        dbname: the name of the database.
        ip: the IP address.
        
        return: 0 if it is not in admin db, 1 if it is in admin db, -1 if 
                any error happens,
        """
        passwd = self.util.decryptString(passwd)

        sql = "select scanner_id from tb_component_list where ip_addr = %s"
        (ret, e) = self.util.executeQueryStatement(addr, int(port), user, passwd, dbname, sql, [ip])
        if e:
            self.logger.debug("Fail to check if ip %s is in tb_component list or not, %s" %(ip, utility.excToString(e)))
            return -1
        # Non-empty result.
        if ret:
            self.logger.debug("The IP %s with scanner ID %s is in tb_component_list." %(ip, ret[0][0]))
            return 1
        else:
            self.logger.debug("The IP %s is not in tb_component_list." %(ip))
            return 0

    def getCompStatus(self, scannerId):
        """
        Get the components's status from admin DB.
        
        scannerId: the scanner ID.
        
        return: a list [daemon, policy, euq, ers, foxhunter], return an empty
                list if any error happens.
        """
        sql = "select daemon, policy, euq, nrs, ipprofiler from tb_component_list where scanner_id=%s"
        (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(),
                                                   sql,
                                                   [scannerId])
        if e:
            self.logger.debug("Fail to get components status with scanner ID %d, %s." %(scannerId, utility.excToString(e)))
            return []

        if not ret:
            self.logger.debug("The components status with scanner ID %s is empty." %(scannerId))
            return []

        return [x for x in ret[0]]

    def getAdminDBInfoFromParent(self, updater):
        """
        Get the adminDB information from a specified parent device.
        
        updater: the status updater.
        
        return: 0 if the adminDB info is successfully retrieved. -1 if any
                errro happens.
        """
        # Build an opener that handles cookies.
        cj = cookielib.CookieJar()
        opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(cj))

        url = "https://%s:8445" %(self.conf.getParentAddress())
        # Authentication
        path = "login.imss"
        authData = {"userid" : self.conf.getAdminUser(),
                    "pwdfake" : base64.b64encode(self.conf.getAdminPassword())}
        try:
            resp = opener.open("%s/%s" %(url, path), urllib.urlencode(authData))
        except Exception, e:
            self.logger.debug("Fail to request the path %s, %s" %(path, utility.excToString(e)))
            updater.append(self.res.getWording("msg.requestParentFail"))
            return -1
        # Get adminDB info.
        path = "ws_dbinfo.imss"
        try:
            resp = opener.open("%s/%s" %(url, path))
        except urllib2.HTTPError as e:
            self.logger.debug("Fail to request the path %s, %s" %(path, utility.excToString(e)))
            if e.code == 403:
                updater.append(self.res.getWording("msg.registerNoPermission"))
            else:
                updater.append(self.res.getWording("msg.requestParentFail"))
            return -1
        except Exception as e:
            self.logger.debug("Fail to request the path %s, %s" %(path, utility.excToString(e)))
            updater.append(self.res.getWording("msg.requestParentFail"))
            return -1
        
        # Parse the request result.
        try:
            root = xml.etree.ElementTree.fromstring(resp.read())
            address = root[0].text
            if address == "127.0.0.1":
                address = self.conf.getParentAddress()
            self.conf.setAdminDBAddress(address)
            self.logger.debug('Get adminDB address "%s" from parent.' %(address))
            self.conf.setAdminDBUsername(root[1].text)
            self.logger.debug('Get adminDB user name "%s" from parent.' %(root[1].text))
            self.conf.setAdminDBPassword(root[2].text)
            self.conf.setAdminDBPort(int(root[3].text))
            self.logger.debug('Get adminDB port "%s" from parent.' %(root[3].text))
            self.conf.setAdminDBName(root[4].text)
            self.logger.debug('Get adminDB db name "%s" from parent.' %(root[4].text))
        except Exception, e:
            self.logger.debug("Fail to parse the adminDB info response, %s" %(utility.excToString(e)))
            updater.append(self.res.getWording("msg.adminDbInfoFormatInvalid"))
            return -1

        return 0

    def register(self, login, addr, port, user, passwd, dbname):
        """
        Register the current device to specified admin db.
        
        login: the login user name of Admin UI
        addr: the address of the database.
        port: the port of the database.
        user: the username of the database.
        passwd: the password of the database.
        dbname: the name of the database.
        
        return: 0 if successful, -1 if failed.
        """
        port = int(port)
        # Get the icp of current device.
        icp = self.getIcpName()
        if not icp:
            self.logger.debug("Fail to get icp device name.")
            return -1
        nicInfo = self.util.getNicInfo()
        if not nicInfo.has_key(icp):
            self.logger.debug("The icp %s doesn't exist on this machine." %(icp))
            return -1
        ip = nicInfo[icp][0]
        # Check if this IP already exists in the admin db to be registered to.
        if self.checkIpInCompList(addr, port, user, passwd, dbname, ip) != 0:
            self.logger.debug("The device whose IP is %s already exists in tb_component_list, cannot register to it." %(ip))
            return -1

        # Get the current components status and maintain it after registering.
        status = self.getCompStatus(1)
        if not status:
            self.logger.debug("Cannot get the current component status, use default values (all stopped).")
            self.conf.setEnableCentralController(0)
            self.conf.setEnableScanner(0)
            self.conf.setEnableEuqConsole(0)
            self.conf.setEnableFoxhunter(0)
        else:
            self.conf.setEnableCentralController(0)
            if status[0] == 2:
                self.conf.setEnableScanner(1)
            else:
                self.conf.setEnableScanner(0)
            if status[2] == 2:
                self.conf.setEnableEuqConsole(1)
            else:
                self.conf.setEnableEuqConsole(0)
            if status[4] == 2:
                self.conf.setEnableFoxhunter(1)
            else:
                self.conf.setEnableFoxhunter(0)

        # Set admin db information.
        self.conf.setAdminDBAddress(addr)
        self.conf.setAdminDBPort(port)
        self.conf.setAdminDBUsername(user)
        self.conf.setAdminDBPassword(passwd)
        self.conf.setAdminDBName(dbname)

        # Set to append install.
        self.conf.setFreshInstall(0)

        # register the device
        if self.registerDeviceToAdminDb(TerminalStatusUpdater()) != 0:
            self.logger.debug("Fail to register the device to %s." %(addr))
            return -1

        # If register finished successfully, save the new admin DB info.
        if self.saveInstallData(TerminalStatusUpdater()) != 0:
            return -1

        # Update the issue file.
        if self.updateIssueFile(TerminalStatusUpdater()) != 0:
            return -1

        # move to afterRegister , because this policy event also need L10N!!!!
        # Insert system event if successful.
        #sql = "insert into tb_system_event (scanner_id, time_date, event_id, event_description, admin_name) values (1, current_timestamp, '30000', %s, %s)"
        # Always insert this event for parent.
        #para = [self.res.getWording("msg.childRegistered") %(ip), login]
        #(ret, e) = self.util.executeUpdateStatement(addr, port, user, self.util.decryptString(passwd), dbname, sql, para)
        #if ret != 0:
        #    # Insert system event fail won't cause register fail.
        #    self.logger.debug("Fail to insert child %s register successful system event, %s" %(ip, utility.excToString(e)))

        return 0

    def checkIcpStatus(self, updater):
        """
        Check if the icp is activated.
        
        return: 0 if icp is activated, -1 if icp is not activated, -2 if icp is using dhcp
        """
        icp = self.getIcpName()
        if not icp:
            self.logger.debug("Cannot get the icp name")
            return -1
        else:
            nics = self.util.getNicInfo()
            if not nics:
                self.logger.debug("Cannot get the nic information on this machine")
                return -1
            else:
                if nics.has_key(icp):
                    if nics[icp][3]:
                        self.logger.debug("The icp %s is using DHCP." %(icp))
                        updater.append(self.res.getWording("msg.nicDhcp") %(icp))
                        return -2
                    else:
                        return 0
                else:
                    self.logger.debug("The icp %s is not activated." %(icp))
                    updater.append(self.res.getWording("msg.nicNotActivate") %(icp))
                    return -1

    def mountOrigRoot(self):
        """
        Mount the previous version's root partition to current system.
        
        return: 0 if successful, -1 if failed.
        """
        # Create the mount point first.
        (ret, e) = self.util.createDir(self.conf.getOrigRootMountPoint(), 0755)
        if ret != 0:
            self.logger.debug("Fail to create the mount point %s for previous version's root partition, %s" %(self.conf.getOrigRootMountPoint(), utility.excToString(e)))
            return -1

        # Do mount
        p = subprocess.Popen(["/bin/mount", "-t", self.conf.getOrigRootFs(), self.conf.getOrigRootDisk(), self.conf.getOrigRootMountPoint()],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to mount the previous version's root partition, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1

        self.logger.debug("Previous version's root partition has been mounted to %s." %(self.conf.getOrigRootMountPoint()))
        return 0

    def umountOrigRoot(self):
        """
        Umount the previous version's root partition from current system.
        
        return: 0 if successful, -1 if failed.
        """
        if not os.path.exists(self.conf.getOrigRootMountPoint()):
            self.logger.debug("The original root mount point %s doesn't exit." %(self.conf.getOrigRootMountPoint()))
            return 0

        if not os.path.ismount(self.conf.getOrigRootMountPoint()):
            self.logger.debug("The path %s is not a mount point." %(self.conf.getOrigRootMountPoint()))
            return 0

        # Do umount
        p = subprocess.Popen(["/bin/umount", self.conf.getOrigRootMountPoint()],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to umount the previous version's root partition, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1

        # Remove the mount point.
        (ret, e) = self.util.removePath(self.conf.getOrigRootMountPoint())
        if ret != 0:
            self.logger.debug("Fail to remove the mount point %s for previous version's root partition, %s" %(self.conf.getOrigRootMountPoint(), utility.excToString(e)))
            return -1

        return 0

    def getAppVersionFromImssd(self, imssdPath):
        """
        Try to read the application version from imssd binary.
        
        imssdPath: the path of the imssd binary.
        
        return: (version, build) tuple, e.g., (8.5, 1000), it returns (0.0, 1000)
                if any error happens.
        """
        if not os.path.exists(imssdPath):
            self.logger.debug("The file %s doesn't exit." %(imssdPath))
            return (0.0, 1000)

        p1 = subprocess.Popen(["strings", imssdPath], stdout=subprocess.PIPE)
        p2 = subprocess.Popen(["grep", "Build_Linux"], stdin=p1.stdout, stdout=subprocess.PIPE)
        p1.stdout.close()
        output = p2.communicate()[0]
        result = re.search(r"Version (\d+\.\d+)-Build_Linux_(\d{4})", output)
        if result == None:
            self.logger.debug("The strings output \"%s\" of %s doesn't contains IMSx application informaion." %(output, imssdPath))
            return (0.0, 1000)
        else:
            return (float(result.group(1)), int(result.group(2)))

    def checkBaseImsxVersion(self, updater):
        """
        Get the version of the current IMSx, check whether it can be upgraded.
        """
        origImssdPath = os.path.join(self.conf.getOrigRootMountPoint() + self.conf.getInstallPath(), "imss", "bin", "imssd")
        (ver, build) = self.getAppVersionFromImssd(origImssdPath)
        self.logger.debug("Local IMSx version: %s, build: %s." %(ver, build))
        if ver != self.conf.getBaseVersion() or build < self.conf.getBaseMinBuildNo():
            self.logger.debug("Cannot upgrade from version: %s, build: %s." %(ver, build))
            updater.append(self.res.getWording("msg.baseVersionPrompt"))
            return -1

        return 0

    def confirmImsxRole(self, imssiniPath):
        """
        Confirm the role of IMSx device according to the imss.ini file.
        
        imssiniPath: the path of imss.ini
        
        return: 0 if successful, -1 if failed.
        """
        parser = ConfigParser.SafeConfigParser()
        try:
            parser.readfp(open(imssiniPath, "r"))
            scannerId = parser.getint("imss_manager", "scanner_id")
            # Only parent's scanner id is 1.
            if scannerId == 1:
                self.conf.setRole(0)
                self.logger.debug("This device is a parent.")
            else:
                self.conf.setRole(1)
                self.logger.debug("This device is a child.")
        except Exception, e:
            self.logger.debug("Fail to parse ini file %s, %s" %(imssiniPath, utility.excToString(e)))
            return -1

        return 0


    def confirmDatabaseConf(self, odbciniPath):
        """
        Confirm the database configuration according to odbc.ini.
        
        odbciniPath: the path of odbc.ini
        
        return: 0 if successful, -1 if failed.
        """
        parser = ConfigParser.SafeConfigParser()
        try:
            parser.readfp(open(odbciniPath, "r"))
            self.conf.setAdminDBAddress(parser.get("IMSS", "Servername"))
            self.conf.setAdminDBPort(parser.getint("IMSS", "Port"))
            self.conf.setAdminDBName(parser.get("IMSS", "Database"))
            self.conf.setAdminDBUsername(parser.get("IMSS", "UserName"))
            self.conf.setAdminDBPassword(parser.get("IMSS", "Password"))
        except Exception, e:
            self.logger.debug("Fail to parse ini file %s, %s" %(odbciniPath, utility.excToString(e)))
            return -1

        return 0

    def checkClusterStatus(self, updater):
        """
        Check the components' (parent and children) status before upgrade.
        
        return: 0 if status are all OK to upgrade, -1 if not OK.
        """
        # As a parent, need to confirm the manager service on all children are
        # stopped.
        if self.conf.getRole() == 0:
            # Get manager listen port.
            managerPort = 15505
            sql = "SELECT value from tb_global_setting where section='imss_manager' and name='listen_port';"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to query manager listen port from db, %s, use default value." %(utility.excToString(e)))
            else:
                managerPort = int(ret[0][0])
            self.logger.debug("The manager listen port is %d." %(managerPort))

            sql = "select ip_addr from tb_component_list where scanner_id!=1;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to query children ip addresses from db, %s" %(utility.excToString(e)))
                return -1
            else:
                childRunning = False
                for rec in ret:
                    ip = rec[0]
                    if self.util.isConnectable(ip, managerPort)[0]:
                        self.logger.debug("[%s]:%d can be connected, imsx service is still running on this machine." %(ip, managerPort))
                        updater.append(self.res.getWording("msg.componentRunningPrompt") %(ip))
                        childRunning = True
                if childRunning:
                    return -1

        # As a child, need to confirm the parent is upgraded already, i.e. the 
        # parent must be upgraded before children.
        else:
            sql = "select app_ver from tb_component_list where scanner_id=1;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to query app version from db, %s" %(utility.excToString(e)))
                return -1
            else:
                if len(ret[0]) == 1:
                    appVer = ret[0][0]
					#imsva9.0 version format is  9.0.0.xxxx
					#imss7.1 version format is 7.1
                    self.logger.debug("Query app version from db, %s" % (appVer))
                    if appVer.count(".") < 1 or appVer.count(".") > 3:
                        self.logger.debug("The format of app version %s is invalid." %(appVer))
                        return -1
                    else:
                        appVerSplit = appVer.split(".")
                        ver = float(".".join(appVerSplit[0:2]))
                        if ver != self.conf.getVersion():
                            updater.append(self.res.getWording("msg.wrongUpgradeSequence"))
                            return -1
                else:
                    self.logger.debug("The database returns invalid result %s" %(ret))
                    return -1

        return 0

    def getSMTPDport(self, updater):
        """
        Get smtpd listen port before upgrade.
        
        return: 0 if status are all OK to upgrade, -1 if not OK.
        """
        sql = "select fieldvalue from tb_postfixconfig where fieldname='default_transport';"
        (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                   self.util.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(),
                                                   sql)
        if e:
            self.logger.debug("Fail to query default_transport from db, %s" %(utility.excToString(e)))
            return -1
        else:
            if len(ret[0]) == 1:
                return (int)(ret[0][0])
            else:
                self.logger.debug("The database returns invalid result %s" %(ret))
                return -1

        return -1

    def migrateAdminDB(self, updater):
        """
        Upgrade the DB schema and then migrate the settings in DB.
        
        return: 0 if successful, -1 if failed.
        """
        updater.append(self.res.getWording("msg.migrateDb"))
        
        migToolPath = "%s/imss/script/imp_exp.sh" %(self.conf.getInstallPath())
        cmd = "%s -p %s %s %s" %(migToolPath, self.conf.getAdminDbExportPath(), self.conf.getUpgradePrevSqlPath(), self.conf.getUpgradePostSqlPath())

        self.logger.debug("Try to migrate the adminDB.")
        p = subprocess.Popen(cmd,
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
        p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to migrate the adminDB, tool exited with code %d. Command:%s" %(p.returncode,cmd))
            return -1

        # For IMSVA, import local users to adminDB.
        if self.conf.getProduct() == 1:
            if self.importLocalUsers() != 0:
                self.logger.debug("Fail to import local users to adminDB.")
                return -1
        
        # Generate the DLPComplianceTemplateFullXML record in tb_named_obj.
        ret = subprocess.call(["/bin/sh", "%s/imss/script/UpdateDlpTemplate.sh" %(self.conf.getInstallPath())],
                              stdout = open("/dev/null"),
                              stderr = open("/dev/null"))
        if ret != 0:
            self.logger.debug("Fail to update DLPComplianceTemplateFullXML in tb_named_obj.")
            return -1
        
        return 0

    def migrateLocalConfigurations(self, updater):
        """
        Migrate the local configurations.
        
        return: 0 if successful, -1 if failed.
        """
        # First copy the ogiginal files to new installation's folder.
        # In IMSVA, we unified imss, ipprofiler, nrs to the same folder, so
        # we have to move files in these folders to imss.
        #
        # A list of (src, dst) tuples. 
        imssHome = "%s/imss" %(self.conf.getInstallPath())
        origImssHome = self.conf.getOrigRootMountPoint() + imssHome
        fileList = [("%s/bin/*bookmark" %(origImssHome), "%s/bin/" %(imssHome)),
                    ("%s/config/Agent.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/database.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/euqodbc.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/foxdns.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/imss.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/imss.ini.db" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/krb5.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/MsgTracing.conf" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/odbc.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/odbcinst.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/pslist.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/rtstat.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/slave.nfo" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/tmfbe_guid" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/LocalDisable.ERS" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/postfix/etc/*" %(origImssHome), "%s/postfix/etc/" %(imssHome)),
                    ("%s/MsgTracing/*Bookmark.txt" %(origImssHome), "%s/MsgTracing/" %(imssHome)),
                    ("%s/MsgTracing/*Timestamp.txt" %(origImssHome), "%s/MsgTracing/" %(imssHome)),
                    ("%s/UI/adminUI/ROOT/reports/*" %(origImssHome), "%s/UI/adminUI/ROOT/reports/" %(imssHome)),
                    ("%s/etc/hosts*" %(self.conf.getOrigRootMountPoint()), "/etc/"),
                    ("%s/etc/issue" %(self.conf.getOrigRootMountPoint()), "/etc/"),
                    ("%s/var/log/maillog*" %(self.conf.getOrigRootMountPoint()), "/var/log/"),
                    ("%s/config/fox_dns_server.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/foxproxy.ini" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/foxproxy.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/fox_tls_forced_domain.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/normal_dns_server.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/rbl_check.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/rbl_info_regex.list" %(origImssHome), "%s/config/" %(imssHome)),
                    ("%s/config/ret_replace.list" %(origImssHome), "%s/config/" %(imssHome))]
        for (src, dst) in fileList:
            # Check if the source exist before copying, otherwise cp will issue an error.
            # Rely on the return value of ls to identify whether source exists. If ls returns
            # no error, means source exists and cp will be called, otherwise return a successful
            # code directly.
            p = subprocess.Popen("ls -U %s > /dev/null 2>&1 && /bin/cp -af %s %s || true" %(src, src, dst),
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to copy %s to %s, stdout: %s, stderr: %s" %(src, dst, stdout, stderr))
                return -1

        # Migrate Configuration files.

        # Migrate foxproxy.ini.
        path = "%s/imss/config/foxproxy.ini" %(self.conf.getInstallPath())
        rules = [(r"^backend_server_port\s*=.*$", "backend_server_port=10021"),
                 (r"^has_foxlib_installed\s*=.*$", ""),
                 (r"^proxy_port\s*=.*$", "proxy_port=10020"),
                 (r"^# Use this parameter to exclude null message.*", ""),
                 (r"^# If a SMTP client closes an established connection without sending out anything.*", ""),
                 (r"^# the session data of this connection will be classified as a null message.*", ""),
                 (r"^incomplete_msg_exclude_null_msg\s*=.*$", ""),
                 (r"^# This is for RBL check against DNS.*", ""),
                 (r"^# Normal RBL check is delagated to the backend MTA.*", ""),
                 (r"^# 0 \-> off.*", ""),
                 (r"^# 1 \-> on.*", ""),
                 (r"^recipient_rbl_check\s*=.*$", ""),
                 (r"^fox_tls_forced_domain_check\s*=.*$", ""),
                 (r"^fox_tls_forced_domain_list\s*=.*$", "")]
        (ret, e) = self.util.replaceInFile(rules, path)
        if ret != 0:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Migrate foxdns.ini.
        path = "%s/imss/config/foxdns.ini" %(self.conf.getInstallPath())
        contents = []
        try:
            with open(path, "r") as f:
                for line in f:
                    contents.append(line)

            contents.append("\nschedule_domain_resolve_enable=1\n")
            contents.append("#schedule_domain_resolve_SchedType = 0 means monthly, 1 means weekly, 2 means daily\n")
            contents.append("schedule_domain_resolve_SchedType=2\n")
            contents.append("schedule_domain_resolve_monthDay=1\n")
            contents.append("schedule_domain_resolve_monthHour=1\n")
            contents.append("schedule_domain_resolve_monthMinute=0\n")
            contents.append("schedule_domain_resolve_weekDay=0\n")
            contents.append("schedule_domain_resolve_weekHour=1\n")
            contents.append("schedule_domain_resolve_weekMinute=0\n")
            contents.append("schedule_domain_resolve_dayHour=1\n")
            contents.append("schedule_domain_resolve_dayMinute=0\n")

            with open(path, "w") as f:
                f.write("".join(contents))
        except Exception, e:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Migrate slave.nfo.
        path = "%s/imss/config/slave.nfo" %(self.conf.getInstallPath())
        contents = []
        try:
            contents.append("[master]\n")
            with open(path, "r") as f:
                for line in f:
                    if re.search(r"^master_ip\s*=\s*$", line):
                        contents.append("master_ip=127.0.0.1\n")
                    else:
                        contents.append(line)

            with open(path, "w") as f:
                f.write("".join(contents))
        except Exception, e:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Migrate imss.ini.
        path = "%s/imss/config/imss.ini" %(self.conf.getInstallPath())
        contents = []
        try:
            with open(path, "r") as f:
                for line in f:
                    if re.search(r"^product_version\s*=.*$", line):
                        contents.append("product_version=9.1\n")
                    elif re.search(r"^# If not specified\, the proxy service sets it to the first IP address", line):
                        contents.append("# If not specified, the proxy service sets the IP address to the first IP address\n")
                    elif re.search(r"^# If not specified or specified as \"all\" or 0\.0\.0\.0", line):
                        continue
                    elif re.search(r"^# then proxy service binds to INADDR_ALL", line):
                        contents.append("# If not specified or specified as \"all\", the proxy service receives\n")
                        contents.append("# packets from all interfaces, including IPv4 or IPv6 clients.\n")
                        contents.append("# If specified as \"0.0.0.0\", the  proxy service receives packets\n")
                        contents.append("# from all interfaces, but is limited to IPv4 clients only.\n")
                    else:
                        contents.append(line)
                        
            # IMSVA 9.1-000531, remove tmfbe url and cert from imss.ini, this should
            # only be done when upgrading from 9.0 to 9.1.
            removeTmfbe = ('# 15.5\n'
                           '# Feedback server of tmfbe\n'
                           '#\n'
                           '# [Valid Range]  : Valid domain name\n'
                           '# [Default Value]: When the key is commented or its value is empty, use "imsva900-en.fbs20.trendmicro.com" for en-US, \n'
                           '#                  use "imsva900-jp.fbs20.trendmicro.com" for JP\n'
                           '#\n'
                           '#tmfbe_host=imsva900-en.fbs20.trendmicro.com\n'
                           '\n'
                           '# 15.6\n'
                           '# Client certificate of tmfbe\n'
                           '#\n'
                           '# [Valid Range]  : Valid tmfbe client certificate string\n'
                           '# [Default Value]: When the key is commented or its value is empty, use "/opt/trend/imss/config/imsva900.cert"\n'
                           '#\n'
                           '#tmfbe_client_cert=\n'
                           )
            newContent = "".join(contents).replace(removeTmfbe, "")
            with open(path, "w") as f:
                f.write(newContent)
        except Exception, e:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Migrate Agent.ini.
        path = "%s/imss/config/Agent.ini" %(self.conf.getInstallPath())
        rules = [(r"^Agent_BuildNumber\s*=.*$", "Agent_BuildNumber=2159")]
        (ret, e) = self.util.replaceInFile(rules, path)
        if ret != 0:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Migrate main.cf.
        postconfPath = "%s/imss/postfix/usr/sbin/postconf" %(self.conf.getInstallPath())
        # Configurations will be overwritten.
        overwriteConfs = [("import_environment", "MAIL_CONFIG MAIL_DEBUG MAIL_LOGTAG TZ XAUTHORITY DISPLAY LANG=C")]
        for key, value in overwriteConfs:
            p = subprocess.Popen("%s -e \"%s=%s\"" %(postconfPath, key, value),
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to update %s in main.cf, stdout: %s, stderr: %s" %(key, stdout, stderr))
                return -1

        appendConfs = [("smtp_tls_policy_maps", "")]
        for key, value in appendConfs:
            p = subprocess.Popen("%s -h %s" %(postconfPath, key),
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to get the value of %s in main.cf, stdout: %s, stderr: %s" %(key, stdout, stderr))
                return -1
            else:
                origValue = stdout.strip()
                if origValue:
                    p = subprocess.Popen("%s -e \"%s=%s\"" %(postconfPath, key, value),
                                         shell = True,
                                         stdout = subprocess.PIPE,
                                         stderr = subprocess.PIPE)
                    (stdout, stderr) = p.communicate()
                    if p.returncode != 0:
                        self.logger.debug("Fail to update %s in main.cf, stdout: %s, stderr: %s" %(key, stdout, stderr))
                        return -1

        # Migrate master.cf.
        port = self.conf.getSmtpPort()
        path = "%s/imss/postfix/etc/postfix/master.cf" %(self.conf.getInstallPath())
        try:
            contents = []
            with open(path, "r") as f:
                for line in f:
                    if re.search(r"^#smtp\s+inet\s+.*$", line):
                        contents.append("%d      inet  n       -       n       -       200     smtpd\n" % (port))
                        contents.append("    -o content_filter= \n")
                        contents.append("    -o smtpd_proxy_filter=inet:127.0.0.1:10020\n")
                        contents.append("    -o local_recipient_maps=hash:/opt/trend/imss/postfix/etc/postfix/empty\n")
                    elif re.search(r"^smtp\s+inet\s+n\s+\-\s+n\s+\-\s+200\s+smtpd", line):
                        continue
                    elif re.search(r"^\d+\s+inet\s+n\s+\-\s+n\s+\-\s+200\s+smtpd", line):
                        continue
                    elif re.search(r"^0\.0\.0\.0\.\:\d+\s+inet\s+n\s+\-\s+n\s+\-\s+200\s+smtpd", line):
                        continue
                    elif re.search(r"^\:\:\:\d+\s+inet\s+n\s+\-\s+n\s+\-\s+200\s+smtpd", line):
                        continue
                    elif re.search(r"^#IMSVA\: content filter loop back smtpd.*$", line):
                        contents.append("#IMSVA: smtpd for imss before-queue content filter\n");
                        contents.append("tls-usage-cleanup   unix  n       -       n       -       0       cleanup\n");
                        contents.append("	-o	header_checks=pcre:/opt/trend/imss/postfix/etc/tls_header_checks\n");
                        contents.append("	-o	nested_header_checks=\n");
                        contents.append("	-o	mime_header_checks=\n");
                        contents.append("\n");
                        contents.append("localhost:10021	inet	n	-	n	-	100	smtpd\n");
                        contents.append("	-o	cleanup_service_name=tls-usage-cleanup\n");
                        contents.append("	-o  smtpd_authorized_xforward_hosts=127.0.0.0/8\n");
                        contents.append("	-o	smtpd_timeout=$imss_timeout\n");
                        contents.append("	-o	local_recipient_maps=\n");
                        contents.append("	-o	myhostname=IMSVA\n");
                        contents.append("	-o	smtpd_junk_command_limit=99999\n");
                        contents.append("	-o	smtpd_client_restrictions=\n");
                        contents.append("	-o	smtpd_tls_security_level=none\n");
                        contents.append("	-o	queue_minfree=512000000\n");
                        contents.append("	-o	mynetworks=127.0.0.0/8\n");
                        contents.append("	-o	mynetworks_style=host\n");
                        contents.append("	-o	smtpd_sender_restrictions=\n");
                        contents.append("	-o	smtpd_recipient_restrictions=permit_mynetworks,reject_unauth_destination\n");
                        contents.append("	-o	import_environment=$default_import_env\n");
                        contents.append("\n");
                        contents.append(line)
                    elif re.search(r"^localhost:10027\s+inet\s+.*$", line):
                        contents.append(line)
                        contents.append("#\t-o\tsmtpd_milters=inet:127.0.0.1:8891\n")
                        contents.append("#\t-o\tnon_smtpd_milters=$smtpd_milters\n")
                        contents.append("#\t-o\tmilter_default_action=tempfail\n")
                        contents.append("#\t-o\tmilter_protocol=2\n")
                    else:
                        contents.append(line)

            contents.append("\n");
            contents.append("# smtp clients for outgoing TLS.\n");
            contents.append("smtp_none      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=none\n");
            contents.append("smtp_may_med      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=may\n");
            contents.append("    -o  smtp_tls_ciphers=medium\n");
            contents.append("smtp_may_high      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=may\n");
            contents.append("    -o  smtp_tls_ciphers=high\n");
            contents.append("smtp_encrypt_med      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=encrypt\n");
            contents.append("    -o  smtp_tls_mandatory_ciphers=medium\n");
            contents.append("smtp_encrypt_high      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=encrypt\n");
            contents.append("    -o  smtp_tls_mandatory_ciphers=high\n");
            contents.append("smtp_verify_med      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=verify\n");
            contents.append("    -o  smtp_tls_mandatory_ciphers=medium\n");
            contents.append("smtp_verify_high      unix  -       -       n       -       -       smtp\n");
            contents.append("    -o  smtp_tls_security_level=verify\n");
            contents.append("    -o  smtp_tls_mandatory_ciphers=high\n");

            with open(path, "w") as f:
                f.write("".join(contents))
        except Exception, e:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        return 0

    def upgradeCron(self):
        """
        Upgrade the crontab.
        
        return: 0 if successful, -1 if fail to backup the crontab, -2 if fail to edit the crontab.
        """
        if self.conf.getProduct() == 1:
            # First backup the crontab file.
            cronPath = "/var/spool/cron/root"
            cronBakPath = self.conf.getCronBackupPath()
            p = subprocess.Popen("/bin/cp -af %s %s" %(cronPath, cronBakPath),
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to backup the crontab, stdout: %s, stderr: %s" %(stdout, stderr))
                return -1

            # Upgrade the cron file.
            rules = [(r"^.+mailtrace\.sh.*$", "")]
            (ret, e) = self.util.replaceInFile(rules, cronPath)
            if ret != 0:
                self.logger.debug("Fail to modify the crontab, %s" %(utility.excToString(e)))
                return -2
        else:
            # TODO, for IMSS
            pass

        if self.exportCronSetting() != 0:
            return -2

        return 0

    def restoreCron(self):
        """
        Restore the crontab to previous version.
        
        return: 0 if successful, -1 if failed.
        """
        if self.conf.getProduct() == 1:
            cronPath = "/var/spool/cron/root"
            cronBakPath = self.conf.getCronBackupPath()
            if not os.path.exists(cronBakPath):
                self.logger.debug("Cannot find the backup crontab.")
                return -1

            subprocess.call("rm -rf %s && mv %s %s" %(cronPath, cronBakPath, cronPath),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
        else:
            # TODO, IMSS
            pass

        return 0

    def startDryrun(self):
        """
        Start the dryrun.
        
        return: 0 if successful, -1 if failed.
        """
        # Check if the current device is available to dryrun.
        enabledFlag = self.conf.getDryrunEnabledFlag()
        startedFlag = self.conf.getDryrunStartedFlag()
        stoppedFlag = self.conf.getDryrunStoppedFlag()

        if os.path.exists(startedFlag):
            self.updater.append(self.res.getWording("msg.dryrunRunning"))
            self.logger.debug("Dryrun is already started on this system.")
            return 0

        if os.path.exists(stoppedFlag):
            self.updater.append(self.res.getWording("msg.dryrunFinished"))
            self.logger.debug("Dryrun has been finished on this system.")
            return -1

        if not os.path.exists(enabledFlag):
            self.updater.append(self.res.getWording("msg.dryrunNotReady"))
            self.logger.debug("The system is not ready for dryrun.")
            return -1

        # Get the role of current device.
        imssiniPath = os.path.join(self.conf.getInstallPath(), ".imss/config/imss.ini")
        if self.confirmImsxRole(imssiniPath) != 0:
            self.updater.append(self.res.getWording("msg.dryrunStartError"))
            self.logger.debug("Fail to identify the role of this device.")
            return -1

        # Get the database settting.
        odbciniPath = os.path.join(self.conf.getInstallPath(), ".imss/config/odbc.ini")
        if self.confirmDatabaseConf(odbciniPath) != 0:
            self.updater.append(self.res.getWording("msg.dryrunStartError"))
            self.logger.debug("Fail to get the database setting.")
            return -1

        # As a parent, need to make sure all children have upgraded.
        if self.conf.getRole() == 0:
            allUpgrade = True
            sql = "select app_ver from tb_component_list where scanner_id != 1;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to query app version from db, %s" %(utility.excToString(e)))
                self.updater.append(self.res.getWording("msg.dryrunStartError"))
                return -1
            else:
                for item in ret:
                    appVer = item[0]
                    if appVer.count(".") != 3:
                        self.logger.debug("The format of app version %s is invalid." %(appVer))
                        allUpgrade = False
                        break
                    else:
                        appVerSplit = appVer.split(".")
                        ver = float(".".join(appVerSplit[0:2]))
                        if ver != self.conf.getVersion():
                            allUpgrade = False
                            break
            if not allUpgrade:
                self.updater.append(self.res.getWording("msg.groupUpgradeNotFinishPrompt"))
                return -1

        # Shutdown the crond
        subprocess.call(["service", "crond", "stop"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        # Set dry run stop in crontab.
        dryrunStopTime = datetime.datetime.now() + datetime.timedelta(hours = self.conf.getDryrunTime())
        try:
            with open("/var/spool/cron/root", "a") as f:

                f.write("%d %d * * * %s/stop_dry_run.sh 2>&1\n" % (dryrunStopTime.minute, dryrunStopTime.hour, self.conf.getInstallPkgPath()))
        except Exception, e:
            self.logger.debug("Fail to write stop dry run in crontab, %s" %(utility.excToString(e)))
            self.updater.append(self.res.getWording("msg.dryrunStartError"))
            return -1

        # Start the crond
        subprocess.call(["service", "crond", "start"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        # Unhide the imss home folder.
        try:
            os.rename(os.path.join(self.conf.getInstallPath(), ".imss"), os.path.join(self.conf.getInstallPath(), "imss"))
        except Exception, e:
            self.logger.debug("Fail to restore the imss home folder: %s. Force to mv .imss to imss" %(utility.excToString(e)))
            #SEGTT-314891: start manager to load firewall,may cause some services start after imssctl.sh stop
            p = subprocess.Popen("rm -rf %s && /bin/mv -f %s %s" %(os.path.join(self.conf.getInstallPath(), "imss"), os.path.join(self.conf.getInstallPath(), ".imss"), os.path.join(self.conf.getInstallPath(), "imss")),
                                shell = True,
                                stdout = subprocess.PIPE,
                                stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Force to mv .imss to imss failed, stdout: %s, stderr: %s" %(stdout, stderr))
                self.updater.append(self.res.getWording("msg.dryrunStartError"))
                return -1
        #SEGTT-314891 If some services is still running, imssctl.sh start cannot start some services...
        self.stopApp()
        self.startApp(self.updater)

        # Remove dry run enable flag and create started flag.
        self.util.removePath(self.conf.getDryrunEnabledFlag())
        open(self.conf.getDryrunStartedFlag(), "a").close()
        self.updater.append(self.res.getWording("msg.dryrunStartPrompt") %(self.conf.getDryrunTime()))
        self.logger.debug("Dryrun has started.")

        return 0

    def stopDryrun(self):
        """
        Stop the dry run.
        
        return: 0 if successful, -1 if failed.
        """
        if os.path.exists(self.conf.getDryrunStoppedFlag()):
            self.logger.debug("Dry run is already stopped.")
            self.updater.append(self.res.getWording("msg.dryrunStopped"))
            return 0
        
        if not os.path.exists(self.conf.getDryrunStartedFlag()):
            self.logger.debug("Dry run is not started yet.")
            self.updater.append(self.res.getWording("msg.dryrunNotStart"))
            return -1

        imssiniPath = "%s/imss/config/imss.ini" %(self.conf.getInstallPath())
        if self.confirmImsxRole(imssiniPath) != 0:
            self.logger.debug("Fail to get the role of current device.")
            self.updater.append(self.res.getWording("msg.dryrunStopError"))
            return -1

        # Stop the application.
        self.stopApp()

        # Start the adminDB so that children can connect it to check status.
        if self.conf.getRole() == 0:
            ctlPostgreSQL("start", self.conf)

        # Hide the imss folder again.
        self.logger.debug("Try to hide the current imss folder.")
        try:
            os.rename(os.path.join(self.conf.getInstallPath(), "imss"), os.path.join(self.conf.getInstallPath(), ".imss"))
        except Exception, e:
            self.logger.debug("Fail to hide the imss folder, %s" %(utility.excToString(e)))
            self.updater.append(self.res.getWording("msg.dryrunStopError"))
            return -1
        
        self.util.removePath(self.conf.getDryrunStartedFlag())
        open(self.conf.getDryrunStoppedFlag(), "a").close()
        self.logger.debug("The dryrun is stopped.")
        self.updater.append(self.res.getWording("msg.dryrunStopped"))
        
        return 0

    def removeDryRunCronJob(self):
        # Stop crond
        subprocess.call(["service", "crond", "stop"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        # Remove the stop dry run script from crontab.
        rules = [(r"^.+stop_dry_run\.sh.*$", "")]
        crontabPath = "/var/spool/cron/root"
        (ret, e) = self.util.replaceInFile(rules, crontabPath)
        if ret != 0:
            self.logger.debug("Fail to remove dry run stop script from crontab, %s" %(utility.excToString(e)))
            return -1

        # Start crond
        subprocess.call(["service", "crond", "start"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        return 0

    def importLocalUsers(self):
        """
        Add local linux users to adminDB, thus these users cannot be added to
        admin accounts via web console, this is to prevent important local users
        being overwritten by admin accounts.
        
        return: 0 if successful, -1 if failed.
        """
        # Truncate the table first.
        sql = "truncate table tb_admin_sysusers"
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql)
        if ret != 0:
            self.logger.debug("Fail to truncate tb_admin_sysusers, %s" %(utility.excToString(e)))
            return -1

        sql = "insert into tb_admin_sysusers (user_id, user_name) values (%s, %s)"
        for user in pwd.getpwall():
            para = [user.pw_uid, user.pw_name]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Fail to add user %s into tb_admin_sysusers, %s" %(user.pw_name, utility.excToString(e)))
                return -1

        return 0
    
    def checkDependencies(self, updater):
        """
        Check whether the dependency packages are installed or not.
        
        return: 0 if successful, -1 if failed.
        """
        for name in self.conf.getDependencies():
            self.logger.debug("Checking dependency package %s..." %(name))
            if not self.util.isRpmInstalled(name):
                self.logger.debug("Dependency %s is not installed." %(name))
                updater.append(self.res.getWording("msg.dependencyNotInstall") %(name))
                return -1
        
        return 0
        
    def removePostfixFoxlibSetting(self):
        """
        Remove the foxlib settings in postfix config.
        """
        if(self.util.isRpmInstalled("postfix")):
            # Identify if the postfix is 64bit
            is64bitPostfix = False
            p = subprocess.Popen("rpm -q postfix | grep x86_64 | wc -l",
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode == 0:
                is64bitPostfix = stdout.strip() == "1"
            else:
                self.logger.debug("Fail to check if postfix is 64bit, stderr: %s" %(stderr))
            
            foxlibEnv = "LD_PRELOAD=%s/imss/%s/libTmFoxSocketLib.so TM_FOX_PROXY_LIST=%s/imss/config/foxproxy.list LD_LIBRARY_PATH=%s/imss/lib TM_FOX_PROXY_CONNECT_PORT=2500" %(self.conf.getInstallPath(), "lib64" if is64bitPostfix else "lib", self.conf.getInstallPath(), self.conf.getInstallPath())
            
            # Get the current import_environment setting of postfix
            p = subprocess.Popen(["postconf", "-h", "import_environment"],
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode == 0:
                oldEnv = stdout.strip()
                self.logger.debug("The import_environment setting of postfix is: %s" %(oldEnv))
                if foxlibEnv not in oldEnv:
                    self.logger.debug("Fox lib already not in postfix setting, no change.")
                else:
                    newEnv = oldEnv.replace(foxlibEnv, "")
                    self.logger.debug("The postfix import_environment will be set to [%s]" %(newEnv))
                    subprocess.call(["postconf", "-e", "import_environment=%s" %(newEnv)],
                                    stdout = open("/dev/null"),
                                    stderr = open("/dev/null"))
                    
                    # Check if postfix is running
                    p = subprocess.Popen("ps -ef | grep master | grep -v grep | wc -l",
                                         shell = True,
                                         stdout = subprocess.PIPE,
                                         stderr = subprocess.PIPE)
                    (stdout, stderr) = p.communicate()
                    if p.returncode == 0 and int(stdout) > 0:
                        self.logger.debug("Postfix is running, need to restart it")
                        subprocess.call(["%s/imss/script/postfixctl.sh" %(self.conf.getInstallPath()), "restart"],
                                        stdout = open("/dev/null"),
                                        stderr = open("/dev/null"))
                    elif p.returncode != 0:
                        self.logger.debug("Fail to check if postfix is running, stderr: %s" %(stderr))
                        return -1
                self.logger.debug("Finished applying postfix changes.")
            else:
                self.logger.debug("Fail to get the import_environment setting of postfix, stderr: %s" %(stderr))
                return -1
        
        return 0

class SilentInstallWelcome(Step):
    """
    The first step of silent install process.
    """
    def apply(self):
        self.updater.append(self.res.getWording("msg.silentInstallStart"))
        return 0

    def rollback(self):
        # No need to rollback this step.
        return 0

class SilentInstallPreCheck(Step):
    """
    The step is to check whether IMSx can be installed, if the check fails, 
    installation cannot continue.
    """

    def apply(self):
        # Check installed rpms.
        ret = self.checkRpmForInstallation(self.updater)
        if ret != 0:
            return -1

        # Check database configuration.
        ret = self.checkDatabaseConfiguration(self.updater)
        if ret != 0:
            return -1

        # Check whether IMSx needed ports (except foxhunter) are available or not.
        ret = self.checkAvailablePorts(self.updater)
        if ret != 0:
            return -1
           
        # Check whether IMSx foxhunter are available or not.
        if self.conf.getEnableIpProfiler() or self.conf.getEnableErs():
            ret = self.checkFoxHunterPort(self.updater)
            if ret != 0:
                return -1

        # Check install path, if it is invalid, installation cannot continue.
        ret = self.checkInstallPath(self.updater)
        if ret != 0:
            return -1
        
        # Check whether all the dependencies are installed or not
        ret = self.checkDependencies(self.updater)
        if ret != 0:
            return -1;

        # Check icp status, the icp must be activated and icp must not be DHCP.
        ret = self.checkIcpStatus(self.updater)
        if ret != 0:
            return -1

        return 0

    def rollback(self):
        # No need to rollback for pre check step.
        return 0;

class SilentInstallSysCheck(Step):
    """
    This step is to check whether the system requirement is met before installing
    IMSx, if the check fails, installation can still continue.
    """
    def apply(self):
        # Check memory, if below minimum level, the installation can still
        # continue, but will give the user a waring.
        ret = 0

        result = self.checkAvailableMemSize(self.updater)
        if result != 0:
            ret = -1
            
        result = self.checkTotalDiskSize(self.updater)
        if result != 0:
            ret = -1

        # For IMSS, check if IMSx users and groups already exist on the machine.
        # If so, need to show some warnings.
        # IMSVA doesn't need this because users and groups are added when 
        # installing the imsva-system rpm.
        if self.conf.getProduct() == 0:
            result = self.checkUsersAndGroups(self.updater)
            if result != 0:
                ret = -1

        # For IMSS, check if install folder already exist.
        if self.conf.getProduct() == 0:
            result = self.checkInstallFolderContent(self.updater)
            if result != 0:
                ret = -1

        return ret

    def rollback(self):
        # No need to rollback this step.
        return 0

class SilentInstallConfirmCompStatus(Step):
    """
    This step is to confirm the component enable status of IMSx.
    """
    def apply(self):
        # Confirm component enable status
        self.confirmCompEnableStatus()
        return 0

    def rollback(self):
        return 0

class SilentInstallUsersAndGroups(Step):
    """
    This step is to install the IMSx needed users and groups.
    """
    def apply(self):
        if self.installUsersAndGroups(self.updater) != 0:
            return -1

        return 0

    def rollback(self):
        if self.removeUsersAndGroups(self.updater) != 0:
            return -1

        return 0

class SilentInstallRpms(Step):
    """
    This step is to install the IMSx related rpms.
    """
    def apply(self):
        # Install IMSX rpms.
        if self.installImsxRpms(self.updater) != 0:
            return -1

        return 0

    def rollback(self):
        if self.uninstallImsxRpms(self.updater) != 0:
            return -1

        if self.removeImsxFiles(self.updater) != 0:
            return -1

        return 0

class SilentInstallReplaceConstants(Step):
    """
    This step is to replace the constants (e.g., _PKG_IMSS_HOME_) in scripts and configurations.
    """
    def apply(self):
        # Replace constants (e.g., install path, db password...) in scripts and
        # configuration files according to the real situation.
        ret = self.replaceConstants(self.updater)
        if ret != 0:
            return -1

        return 0

    def rollback(self):
        # No need to rollback this step.
        return 0

class SilentInstallDatabase(Step):
    """
    This step is to install and init the database.
    """
    def apply(self):
        if self.installDatabase(self.updater) != 0:
            return -1

        return 0

    def rollback(self):
        # Stop the database for bundled database type.
        if self.conf.getAdminDBType() == 0:
            ctlPostgreSQL("stop", self.conf)
        return 0

class SilentInstallRegisterDevice(Step):
    """
    This step is to register the current device to admin DB.
    """
    def apply(self):
        # If append install, need to get the adminDB information from parent.
        if self.conf.getFreshInstall() == 0:
            ret = self.getAdminDBInfoFromParent(self.updater)
            if ret != 0:
                return -1
        # Register this device to admin DB.
        ret = self.registerDeviceToAdminDb(self.updater)
        if ret != 0:
            return -1

        # For IMSS append install, after register to master's admin db, then reallocate tb_euq_table_info.
        if self.conf.getFreshInstall() == 0 and self.conf.getProduct() == 0:
            if self.removeUseleseEuqDb() == -1:
                return -1
            self.reallocEuqTable()

            # get parent address and soap port for slave.nfo
            parentAddress = self.getParentAddress()
            soapPort = self.getSoapPort()
            
            # modify slave.nfo
            path = "%s/imss/config/slave.nfo" %(self.conf.getInstallPath())
            contents = []
            try:
                contents.append("[master]\n")
                contents.append("master_ip=")
                contents.append(parentAddress)
                contents.append("\n")
                contents.append("soap_port=")
                contents.append(soapPort)
                contents.append("\n")

                with open(path, "w") as f:
                    f.write("".join(contents))
            except Exception, e:
                self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
                return -1
        
        return 0

    def rollback(self):
        ret = self.unregisterDeviceFromAdminDb(self.updater)
        if ret != 0:
            return -1

        return 0

class SilentInstallService(Step):
    """
    This step is to install the IMSx service to the machine.
    """
    def apply(self):
        # Configure system init for IMSx
        ret = self.installService(self.updater)
        if ret != 0:
            return -1

        # initialize LDAPS certificate Database
        p = subprocess.Popen("%s/database/init_ldap_db.sh %s" % (self.conf.getInstallPkgPath(), self.conf.getInstallPath()),
                             shell = True,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to initialize database for LDAPS certificate, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1

        return 0

    def rollback(self):
        # No need to rollback this step, the service is uninstalled in the rollback
        # of installing rpm step.
        return 0

class SilentInstallCron(Step):
    """
    This step is to install the cron jobs to the machine.
    """
    def apply(self):
        # Update cron file
        ret = self.installCron(self.updater)
        if ret != 0:
            return -1

        return 0

    def rollback(self):
        ret = self.uninstallCron(self.updater)
        if ret != 0:
            return -1

        return 0

class SilentInstallCleanup(Step):
    """
    This step is to remove temp files and backup install configs.
    """
    def apply(self):
        # Remove temp files.
        ret = self.removeInstallTempFiles(self.updater)
        if ret != 0:
            return -1

        # Backup installation files.
        ret = self.saveInstallData(self.updater)
        if ret != 0:
            return -1

        return 0

    def rollback(self):
        # No rollback for this step
        return 0

class SilentInstallUpdateIssue(Step):
    """
    This step is to update the /etc/issue file on IMSVA.
    """
    def apply(self):
        self.updateIssueFile(self.updater)
        return 0

    def rollback(self):
        return 0

class SilentInstallStartApp(Step):
    """
    This step is to start the IMSx services.
    """
    def apply(self):
        # Start the application
        self.startApp(self.updater)
        return 0

    def rollback(self):
        self.stopAppExceptDB()
        return 0

class SilentInstallImportDefAuComp(Step):
    """
    This step is to import the default AU engines and patterns.
    """
    def apply(self):
        if self.conf.getFreshInstall() != 1:
            self.logger.debug("Skip this step because only fresh install need to perform this step.")
            return 0
        ret = self.importDefAuComp(self.updater)
        if ret != 0:
            return -1

        return 0

    def rollback(self):
        # No rollback for this step.
        return 0
    
class SilentInstallFoxhunterConfig(Step):
    """
    This step is to configure foxhunter, this step is only needed for IMSS. 
    """
    def apply(self):
        if self.conf.getProduct() == 0:
            # Enable ERS and IP profiler if the user enables. By default, they are both disabled.
            if self.conf.getEnableErs() == 1:
                sql = "update tb_global_setting set value='yes' where section='ers' and name='ers_enabled'"
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
                if ret != 0:
                    self.logger.debug("Fail to enable ERS in database, %s" %(utility.excToString(e)))
                
                # Update the conf version for ERS so that foxdns can sync the config to local.
                sql = "update tb_tables_version set version=version+1 where tb_name='ers'"
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
                if ret != 0:
                    self.logger.debug("Fail to update ERS version in database, %s" %(utility.excToString(e)))
            
            if self.conf.getEnableIpProfiler() == 1:
                sql = "update t_foxhuntersetting set enable=true"
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
                if ret != 0:
                    self.logger.debug("Fail to enable IP profiler in database, %s" %(utility.excToString(e)))
                    
                # Update the conf version for IP profiler so that foxdns can sync the config to local.
                sql = "update tb_tables_version set version=version+1 where tb_name='ip_profiler'"
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                            self.conf.getAdminDBPort(),
                                                            self.conf.getAdminDBUsername(),
                                                            self.util.decryptString(self.conf.getAdminDBPassword()),
                                                            self.conf.getAdminDBName(),
                                                            sql)
                if ret != 0:
                    self.logger.debug("Fail to update IP profiler version in database, %s" %(utility.excToString(e)))
            
            # Determine if the user has postfix installed.
            if(self.util.isRpmInstalled("postfix")):
                self.logger.debug("Found postfix installed on this machine.")
                
                # Make foxhunter aware the foxlib exists
                self.util.replaceInFile([(r"has_foxlib_installed=0", r"has_foxlib_installed=1")], "%s/imss/config/foxproxy.ini" %(self.conf.getInstallPath()))
                # Config changed, need to restart foxhunter if either IP profiler or ERS is enabled.
                if self.conf.getEnableErs() == 1 or self.conf.getEnableIpProfiler() == 1:
                    subprocess.call(["%s/imss/script/foxproxyd" %(self.conf.getInstallPath()), "restart"],
                                    stdout = open("/dev/null"),
                                    stderr = open("/dev/null"))
                
                # Identify if the postfix is 64bit
                is64bitPostfix = False
                p = subprocess.Popen("rpm -q postfix | grep x86_64 | wc -l",
                                     shell = True,
                                     stdout = subprocess.PIPE,
                                     stderr = subprocess.PIPE)
                (stdout, stderr) = p.communicate()
                if p.returncode == 0:
                    is64bitPostfix = stdout.strip() == "1"
                else:
                    self.logger.debug("Fail to check if postfix is 64bit, stderr: %s" %(stderr))
                
                foxlibEnv = "LD_PRELOAD=%s/imss/%s/libTmFoxSocketLib.so TM_FOX_PROXY_LIST=%s/imss/config/foxproxy.list LD_LIBRARY_PATH=%s/imss/lib TM_FOX_PROXY_CONNECT_PORT=2500" %(self.conf.getInstallPath(), "lib64" if is64bitPostfix else "lib", self.conf.getInstallPath(), self.conf.getInstallPath())               
                
                # Get the current import_environment setting of postfix
                p = subprocess.Popen(["postconf", "-h", "import_environment"],
                                     stdout = subprocess.PIPE,
                                     stderr = subprocess.PIPE)
                (stdout, stderr) = p.communicate()
                if p.returncode == 0:
                    oldEnv = stdout.strip()
                    self.logger.debug("The import_environment setting of postfix is: %s" %(oldEnv))
                    if "libTmFoxSocketLib" in oldEnv:
                        self.logger.debug("Fox lib already in postfix setting, no change.")
                    else:
                        newEnv = oldEnv + " " + foxlibEnv
                        self.logger.debug("The postfix import_environment will be set to [%s]" %(newEnv))
                        subprocess.call(["postconf", "-e", "import_environment=%s" %(newEnv)],
                                        stdout = open("/dev/null"),
                                        stderr = open("/dev/null"))
                        
                        # Check if postfix is running
                        p = subprocess.Popen("ps -ef | grep master | grep -v grep | wc -l",
                                             shell = True,
                                             stdout = subprocess.PIPE,
                                             stderr = subprocess.PIPE)
                        (stdout, stderr) = p.communicate()
                        if p.returncode == 0 and int(stdout) > 0:
                            self.logger.debug("Postfix is running, need to restart it")
                            subprocess.call(["%s/imss/script/postfixctl.sh" %(self.conf.getInstallPath()), "restart"],
                                            stdout = open("/dev/null"),
                                            stderr = open("/dev/null"))
                        elif p.returncode != 0:
                            self.logger.debug("Fail to check if postfix is running, stderr: %s" %(stderr))
                            return -1
                    self.logger.debug("Finished applying postfix changes.")
                else:
                    self.logger.debug("Fail to get the import_environment setting of postfix, stderr: %s" %(stderr))
                    return -1
        
        return 0
    
    def rollback(self):
        # Revert the changes to postfix config.
        if self.removePostfixFoxlibSetting() < 0:
            return -1
        
        return 0

class UpgradeWelcom(Step):
    """
    This should be the first step of the whole upgrade process.
    """
    def apply(self):
        uuid_str = ""

        try:
            with open("/etc/fstab", "r") as f:
                for line in f:
                    if re.search(r"^UUID=[\d\w\-]+\s+\/boot", line):
                        uuid_str = line
                        break
            items = uuid_str.split(" ")
            uuid_str = items[0]
        except Exception, e:
            self.logger.debug("Fail to read UUID of /boot in %s, %s" %(path, utility.excToString(e)))
            return -1

        self.conf.setBootUUID(uuid_str)
        return 0

    def rollback(self):
        # Mount the original root if it is not mounted.
        if not os.path.exists(self.conf.getOrigRootMountPoint()):
            if self.mountOrigRoot() != 0:
                return -1

        # update /boot uuid in old fstab
        contents = []
        try:
            path = self.conf.getOrigRootMountPoint() + "/etc/fstab"
            with open(path, "r") as f:
                for line in f:
                    if re.search(r"^UUID=[\d\w\-]+\s+\/boot", line):
                        pos = line.find(" ")
                        contents.append(self.conf.getBootUUID() + line[pos:])
                    else:
                        contents.append(line)

            with open(path, "w") as f:
                f.write("".join(contents))
        except Exception, e:
            self.logger.debug("Fail to edit %s, %s" %(path, utility.excToString(e)))
            return -1

        # Copy the backup boot folder to current boot folder.
        p = subprocess.Popen("/bin/cp -af " + self.conf.getOrigRootMountPoint() + "/boot-imsva-9.0-backup-for-9.1/* /boot",
                             shell = True,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to restore original boot folder, stdout:%s, stderr: %s" %(stdout, stderr))
            return -1

        if self.umountOrigRoot() != 0:
            return -1

        return 0

class UpgradeInit(Step):
    """
    This step is to perform initialize steps.
    """
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        self.adminDbBackedUp = True

    def apply(self):
        self.adminDbBackedUp = False

        # Get the current installed path.
        # IMSVA has a fixed path.
        if self.conf.getProduct() == 1:
            self.conf.setInstallPath("/opt/trend")
        # TODO, IMSS need to check rpm -q output.
        else:
            pass

        # Mount the original root partition.
        if self.mountOrigRoot() != 0:
            return -1

        # Confirm the role of base IMSx.
        imssiniPath = os.path.join(self.conf.getOrigRootMountPoint() + self.conf.getInstallPath(), "imss", "config", "imss.ini")
        if self.confirmImsxRole(imssiniPath) != 0:
            self.logger.debug("Fail to get the role of base IMSx.")
            return -1

        # Confirm the database configuration of the base IMSx.
        odbciniPath = os.path.join(self.conf.getOrigRootMountPoint() + self.conf.getInstallPath(), "imss", "config", "odbc.ini")
        if self.confirmDatabaseConf(odbciniPath) != 0:
            self.logger.debug("Fail to get the database configurations of base IMSx.")
            return -1

        # Shutdown the crond
        subprocess.call(["service", "crond", "stop"],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"))

        # Create the /var/imss link if it is IMSVA.
        if self.conf.getProduct() == 1:
            pgDataRootPathLink = os.path.normpath("/var/imss")
            pgDataRootPath = os.path.normpath("/var/app_data/imss/db")
            self.logger.debug("Try to create a link from %s to %s." %(pgDataRootPath, pgDataRootPathLink))
            if os.path.lexists(pgDataRootPathLink):
                (ret, e) = self.util.removePath(pgDataRootPathLink)
                if ret != 0:
                    self.logger.debug("Cannot delete %s, %s" %(pgDataRootPathLink, utility.excToString(e)))
                    return -1
            try:
                os.symlink(pgDataRootPath, pgDataRootPathLink)
            except Exception, e:
                self.logger.debug("Fail to create a link from %s to %s." %(pgDataRootPath, pgDataRootPathLink))
                return -1

        # Backup the adminDB data folder
        adminDBDataPath = "/var/app_data/imss/db/pgdata"
        adminDBDataBakPath = self.conf.getAdminDBBackupFolder()
        self.logger.debug("Try to backup the adminDB data folder %s to %s." %(adminDBDataPath, adminDBDataBakPath))
        if os.path.lexists(adminDBDataBakPath):
            self.logger.debug("Cannot create the backup folder %s because it already exists." %(adminDBDataBakPath))
            return -1
        p = subprocess.Popen("/bin/cp -af %s %s" %(adminDBDataPath, adminDBDataBakPath),
                             shell = True,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to copy the adminDB data folder, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1
        self.adminDbBackedUp = True

        imssHome = os.path.join(self.conf.getInstallPath(), "imss")
        tempImssHome = self.conf.getOrigRootMountPoint() + imssHome
        # IMSVA
        if self.conf.getProduct() == 1:
            # Because the old IMSVA is in original root, we need to 
            # create a IMSS_HOME link to it to start the adminDB and
            # export setting.
            try:
                hiddenImssHome = os.path.join(self.conf.getInstallPath(), ".imss")
                os.rename(imssHome, hiddenImssHome)
            except Exception, e:
                self.logger.debug("Fail to rename the folder %s to %s, %s" %(imssHome, hiddenImssHome, utility.excToString(e)))
                return -1

            try:
                os.symlink(tempImssHome, imssHome)
            except Exception, e:
                self.logger.debug("Fail to create the symlink from %s to %s, %s" %(tempImssHome, imssHome, utility.excToString(e)))
                return -1

        # Start the adminDB.
        ctlPostgreSQL("start", self.conf)

        # Export the configuration from adminDB if it is a parent or central controller.                
        backup_flag = "backupEuq"
        if self.conf.getRole() == 0:
            backup_flag = "backupAll"

            # Export the configurations.
            exportPath = self.conf.getAdminDbExportPath()
            toolPath = "%s/imss/script/imp_exp.sh" %(self.conf.getInstallPath())

            # Export the setting from adminDB.
            self.updater.append(self.res.getWording("msg.exportDBStartPrompt"))
            p = subprocess.Popen([toolPath, "-e", exportPath],
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to export from adminDB, stdout: %s, stderr: %s" %(stdout, stderr))
                self.updater.append(self.res.getWording("msg.exportDBFailPrompt"))
                return -1
            else:
                self.logger.debug("The previous settings in adminDB has been exported to %s." %(exportPath))
                self.updater.append(self.res.getWording("msg.exportDBEndPrompt"))

        # dump adminDB before upgrade
        toolPath = "%s/database/dbimpexp.sh" % (self.conf.getInstallPkgPath())
        self.updater.append(self.res.getWording("msg.backupDBStartPrompt"))
        p = subprocess.Popen([toolPath, backup_flag], stdout = subprocess.PIPE, stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to dump database(%s), stdout: %s, stderr: %s" %(backup_flag, stdout, stderr))
            self.updater.append(self.res.getWording("msg.backupDBFailPrompt"))
            return -1
        else:
            self.logger.debug("database(%s) dump to /var/app_data/imss/pg92" % (backup_flag))
            self.updater.append(self.res.getWording("msg.backupDBEndPrompt"))

        # IMSVA
        if self.conf.getProduct() == 1:
            ctlPostgreSQL("stop", self.conf)
            # Restore the IMSS_HOME
            try:
                os.remove(imssHome)
                os.rename(os.path.join(self.conf.getInstallPath(), ".imss"), imssHome)
            except Exception, e:
                self.logger.debug("Fail to restore the %s folder after exporting adminDB, %s" %(imssHome, utility.excToString(e)))
                return -1

        return 0

    def rollback(self):
        # Restore the backup adminDB.
        if self.adminDbBackedUp:
            # Stop the DB first.
            dbctlPath = "%s/imss/script/dbctl.sh" %(self.conf.getInstallPath())
            if os.path.exists(dbctlPath):
                ctlPostgreSQL("stop", self.conf)
            # remove dump file first
            p = subprocess.Popen("rm -rf /var/app_data/imss/pg92",
                                 shell = True,
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug("Fail to remove database dump file, stdout: %s, stderr: %s" %(stdout, stderr))

            adminDBDataPath = "/var/app_data/imss/db/pgdata"
            adminDBDataBakPath = self.conf.getAdminDBBackupFolder()
            self.logger.debug("Try to restore the adminDB data from folder %s to %s." %(adminDBDataBakPath, adminDBDataPath))
            if os.path.isdir(adminDBDataBakPath):
                p = subprocess.Popen("rm -rf %s && mkdir -p /var/app_data/imss/db && /bin/mv -f %s %s" %(adminDBDataPath, adminDBDataBakPath ,adminDBDataPath),
                                     shell = True,
                                     stdout = subprocess.PIPE,
                                     stderr = subprocess.PIPE)
                (stdout, stderr) = p.communicate()
                if p.returncode != 0:
                    self.logger.debug("Fail to restore the adminDB data folder, stdout: %s, stderr: %s" %(stdout, stderr))
                    return -1
            else:
                self.logger.debug("The adminDB backup folder %s doesn't exist." %(adminDBDataBakPath))
                return -1

        return 0

class UpgradePreCheck(Step):
    """
    This step is to check whether the system is available for upgrade.
    """
    def apply(self):
        self.updater.append(self.res.getWording("msg.upgradePreCondCheckStartPrompt"))

        # Check the base IMSx version and build.
        if self.checkBaseImsxVersion(self.updater) != 0:
            return -1

        # Need to temporary restore the original imss home and start adminDB
        # on parent so that we can check the service status of children. 
        imssHome = os.path.join(self.conf.getInstallPath(), "imss")
        tempImssHome = self.conf.getOrigRootMountPoint() + imssHome
        if self.conf.getRole() == 0:
            try:
                os.rename(imssHome, os.path.join(self.conf.getInstallPath(), ".imss"))
            except Exception, e:
                self.logger.debug("Fail to rename the IMSS_HOME folder, %s" %(utility.excToString(e)))
                return -1
            try:
                os.symlink(tempImssHome, imssHome)
            except Exception, e:
                self.logger.debug("Fail to create the symlink from %s to %s, %s" %(tempImssHome, imssHome, utility.excToString(e)))
                return -1
            dbctlPath = "%s/imss/script/dbctl.sh" %(self.conf.getInstallPath())
            ctlPostgreSQL("start", self.conf)
        if self.checkClusterStatus(self.updater) != 0:
            return -1

        # get smtp listen port of postfix from database
        port = self.getSMTPDport(self.updater)
        self.conf.setSmtpPort(port)

        if self.conf.getRole() == 0:
            ctlPostgreSQL("stop", self.conf)
            # Restore the IMSS_HOME
            try:
                os.remove(imssHome)
                os.rename(os.path.join(self.conf.getInstallPath(), ".imss"), imssHome)
            except Exception, e:
                self.logger.debug("Fail to restore the %s folder after exporting adminDB, %s" %(imssHome, utility.excToString(e)))
                return -1

        self.updater.append(self.res.getWording("msg.upgradeStartPrompt"))
        return 0

    def rollback(self):
        # Stop the adminDB for parent.
        if self.conf.getRole() == 0:
            dbctlPath = "%s/imss/script/dbctl.sh" %(self.conf.getInstallPath())
            if os.path.exists(dbctlPath):
                ctlPostgreSQL("stop",self.conf)
        return 0

class UpgradeInstallNewVersion(Step):
    """
    This step is to install the new IMSx application to the system.
    """
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        # Specify whether the crontab is backed up successfully.
        self.crontabBackedup = True

    def apply(self):
        self.crontabBackedup = False

        if self.installImsxRpms(self.updater) != 0:
            return -1

        if self.replaceConstants(self.updater) != 0:
            return -1

        # Upgrade the crontab.
        ret = self.upgradeCron()
        if ret != 0:
            self.logger.debug("Fail to update the crontab")
            if ret != -1:
                self.crontabBackedup = True
            return -1
        else:
            self.crontabBackedup = True

        # initialize LDAPS certificate Database
        p = subprocess.Popen("%s/database/init_ldap_db.sh %s" % (self.conf.getInstallPkgPath(), self.conf.getInstallPath()),
                             shell = True,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to initialize database for LDAPS certificate, stdout: %s, stderr: %s" %(stdout, stderr))

        # Install services
        if self.installService(self.updater) != 0:
            self.logger.debug("Fail to install imsx services.")
            return -1

        return 0

    def rollback(self):
        # Need to restore crontab only if the original crontab is backed up.
        if self.crontabBackedup:
            if self.restoreCron() != 0:
                self.logger.debug("Fail to restore the crontab.")
                return -1

        return 0

class UpgradeLocalConfigurations(Step):
    """
    This step is to upgrade the local configurations files.
    """
    def apply(self):
        if self.migrateLocalConfigurations(self.updater) != 0:
            return -1

        return 0

    def rollback(self):
        # No need to rollback this step.
        return 0

class UpgradeAdminDB(Step):
    """
    This step is to update the schema and data in adminDB.
    """
    def apply(self):
        # Create new database before migration
        self.updater.append(self.res.getWording("msg.installDatabaseStart"))
        ret = self.installBundledDatabase()
        if ret != 0:
            self.updater.append(self.res.getWording("msg.installDatabaseFail"))
            return -1
        else:
            self.updater.append(self.res.getWording("msg.installDatabaseSuccess"))

        # Start the adminDB
        ctlPostgreSQL("start", self.conf)

        restore_flag = "restoreEuq"
        if self.conf.getRole() == 0:
            restore_flag = "restoreAll"

        # restore data
        toolPath = "%s/database/dbimpexp.sh" % (self.conf.getInstallPkgPath())
        self.updater.append(self.res.getWording("msg.restoreDBStartPrompt"))
        p = subprocess.Popen([toolPath, restore_flag], stdout = subprocess.PIPE, stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to restore database(%s), stdout: %s, stderr: %s" %(restore_flag, stdout, stderr))
            self.updater.append(self.res.getWording("msg.restoreDBFailPrompt"))
            return -1
        else:
            self.logger.debug("database(%s) restored from /var/app_data/imss/pg92" % (restore_flag))
            self.updater.append(self.res.getWording("msg.restoreDBEndPrompt"))

        # Only parent need to upgrade adminDB.
        if self.conf.getRole() == 0:
            # migrate data
            if self.migrateAdminDB(self.updater) != 0:
                return -1

            # Start the manager to apply the firewall settings so that children can
            # access the parent's adminDB. 
            subprocess.call("%s start" %(os.path.join(self.conf.getInstallPath(), "imss/script/S99MANAGER")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))

            # If it is a parent, need to import default engines and patterns.
            importRes = 0
            if self.importDefAuComp(self.updater) != 0:
                self.logger.debug("Fail to import the default engines and patterns.")
                importRes = -1

            subprocess.call("%s stop" %(os.path.join(self.conf.getInstallPath(), "imss/script/imssctl.sh")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
            subprocess.call("%s start" %(os.path.join(self.conf.getInstallPath(), "imss/script/dbctl.sh")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))

            if importRes != 0:
                return -1

            #From 9.0 to 9.1, need to remove previous vsapi/atse engine version, to prevent user rolling back the engine, new feature "Connected Threat Defense(Better Together)" need the new engine to support--Niel.
            #Also, we need to remove previous tmufe/tmase engine version, becase new feature "Time of Click Protection" need the new engines.
            #    vsapi engine: type=0, so the backup type=100 / atse engine: type=9, so the backup type=109  /tmufe engine: type=8, so backup type=108 / tmase engine: type=5, so backup type=105
            #1)In actually, the large object refernced by the old engine stored in databse should be removed by using "select lo_unlink(oid)", oid is the value stored in column "content" in tb_active_update.
            #  we can use "SELECT DISTINCT (loid) FROM pg_largeobject" to list all large objects.
            sql = "select content from tb_active_update where type=100 or type=109 or type=105 or type=108;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                #This is not critical, because the large object is just has one copy, and it doesn't occupy much disk[sizeof(libvsapi.so)+sizeof(libatse.so) + sizeof(libtmufeng.so) + sizeof(libtmaseng.so)], 
                #so it can be OK if we leave them in DB when error occurs here.
                self.logger.debug("Fail to query content oid for vsapi/atse engine from tb_active_update, %s" %(utility.excToString(e)))
            else:
                for item in ret:
                    sql = "select lo_unlink(%s);"
                    para = [item[0]]
                    (ret2, e2) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                                  self.conf.getAdminDBPort(),
                                                                  self.conf.getAdminDBUsername(),
                                                                  self.util.decryptString(self.conf.getAdminDBPassword()),
                                                                  self.conf.getAdminDBName(),
                                                                  sql,
                                                                  para)
                    if ret2 != 0:
                        # This is not critical, if failed, the large object will leave in DB.
                        self.logger.debug("Failed to delete large object from DB, %s" %(utility.excToString(e2)))
                    else:
                        self.logger.debug("Delete large object successfully, oid=%s." %(item[0]));
            #end 1)~~~

            scannerId = self.getLocalScannerId()
            #2) remove the records in tb_active_update   
            sql = "delete from tb_active_update where type=100 or type = 109 or type=105 or type=108;"
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql)
            if ret != 0:
                self.logger.debug("Failed to execute sql command to delete old vsapi/atse engine %d, %s" %(scannerId, utility.excToString(e)))
                return -1
            else:
                self.logger.debug("New feature(Better Together) needs the engine upgraded, delete old vsapi/atse engine version to prevent user rolling back them.")
            #end 2)~~~

        else:
            # Child doesn't need to do adminDB migration.
            pass

        # Both parent and children needs to update the os and app versions in tb_component_list.
        osVer = self.getOsVersion()
        appVer = self.getAppVersion()
        scannerId = self.getLocalScannerId()

        sql = "update tb_component_list set os_ver=%s, app_ver=%s where scanner_id=%s;"
        para = [osVer, appVer, scannerId]
        (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                    self.conf.getAdminDBPort(),
                                                    self.conf.getAdminDBUsername(),
                                                    self.util.decryptString(self.conf.getAdminDBPassword()),
                                                    self.conf.getAdminDBName(),
                                                    sql,
                                                    para)
        if ret != 0:
            self.logger.debug("Failed to execute sql command to update the os and app version for scanner %d, %s" %(scannerId, utility.excToString(e)))
            return -1
        else:
            self.logger.debug("Updated scanner %d in tb_component_list, set ov_ver=%s, app_ver=%s" %(scannerId, osVer, appVer))

        # Update EUQ database password.
        if self.conf.getRole() == 0:
            sql = "select password, db_id from tb_euq_db_info;"
            (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
            if e:
                self.logger.debug("Fail to query euq db info from db, %s" %(utility.excToString(e)))
                return -1

            for item in ret:
                if len(item) != 2:
                    self.logger.debug("Failed to get password and db_id from table")
                    return -1
                plain_passwd = item[0]
                db_id = item[1]
                password = self.util.encryptString(plain_passwd)
                self.logger.debug("The euq db info for db_id: %s." %(db_id))
                sql = "update tb_euq_db_info set password=%s where db_id=%s;"
                para = [password, db_id]
                (ret2, e2) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                              self.conf.getAdminDBPort(),
                                                              self.conf.getAdminDBUsername(),
                                                              self.util.decryptString(self.conf.getAdminDBPassword()),
                                                              self.conf.getAdminDBName(),
                                                              sql,
                                                              para)
                if ret2 != 0:
                    self.logger.debug("Failed to execute sql command to update euq db passwd: %s" %(utility.excToString(e2)))
                    return -1

                self.logger.debug("update euq db password successfully")

        return 0

    def rollback(self):
        # Only need to stop DB here. The DB migration will be put in a transaction
        # and the rollback will be handled by database itself.
        ctlPostgreSQL("stop",self.conf)
        return 0

class UpgradeCleanup(Step):
    """
    This step is to perfrom the cleanup jobs for upgrade.
    """
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        self.logBackedUp = True

    def apply(self):
        self.logBackedUp = False
        # Backup the log folder here so that we can have the db migration logs
        # even after rolling back.
        logFolder = "/var/app_data/imss/log/"
        logBakFolder = self.conf.getLogBackupFolder()
        self.logger.debug("Try to backup the log folder %s to %s." %(logFolder, logBakFolder))
        if os.path.lexists(logBakFolder):
            self.logger.debug("The log backup folder %s already exists." %(logBakFolder))
            return -1
        p= subprocess.Popen("/bin/cp -af %s %s" %(logFolder, logBakFolder),
                            shell = True,
                            stdout = subprocess.PIPE,
                            stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug("Fail to backup the log folder, stdout: %s, stderr: %s" %(stdout, stderr))
            return -1
        self.logBackedUp = True

        # Save upgrade config.
        if self.saveInstallData(self.updater) != 0:
            return -1

        # Umount the original root partition should be the last action of the 
        # whole upgrade process.
        if self.umountOrigRoot() != 0:
            return -1

        # Resume PostgreSQL configuration which has been modified for inline upgrade
        pgDataRootPath = os.path.normpath("/var/imss")
        if self.conf.getProduct() == 1:
            pgDataRootPath = os.path.normpath("/var/app_data/imss/db")
        pgDataPath = "%s/pgdata" %(pgDataRootPath)
        pgConfPath = "%s/postgresql.conf" %(pgDataPath)

        self.logger.debug("Try to resume PostgreSQL configuration for inline upgrade.")
        rules = [(r"max_locks_per_transaction =.*#", r"#max_locks_per_transaction = 64\t\t#"),
                 (r"#idle_in_transaction_session_timeout =.*#", r"idle_in_transaction_session_timeout = 3600000\t#") ]
        (ret, e) = self.util.replaceInFile(rules, pgConfPath)
        if ret != 0:
            self.logger.debug("Failed to resume PostgreSQL settings in %s, %s" %(pgConfPath, utility.excToString(e)))
            return -1
        
        # Start the adminDB so that it can be used by children.
        if self.conf.getRole() == 0:
            ctlPostgreSQL("start", self.conf)

        # Hide the current IMSS_HOME folder to prevent the user to srart 
        # IMSX directly, the user must use dry_run to start it.
        self.logger.debug("Try to hide the current imss folder.")
        try:
            os.rename(os.path.join(self.conf.getInstallPath(), "imss"), os.path.join(self.conf.getInstallPath(), ".imss"))
        except Exception, e:
            self.logger.debug("Fail to hide the imss folder, %s" %(utility.excToString(e)))
            return -1

        # Start the crond
        subprocess.call(["service", "crond", "start"],
                        stdout = subprocess.PIPE,
                        stderr = subprocess.PIPE)

        # Create the dryrun_ready_flag.
        open(self.conf.getDryrunEnabledFlag(), "a").close()

        # Prompt for dry_run.
        self.updater.append(self.res.getWording("msg.dryrunReadyPrompt") %(self.conf.getDryrunTime()))

        return 0

    def rollback(self):
        # Restore the log folder if it is backed up.
        if self.logBackedUp:
            logFolder = "/var/app_data/imss/log/"
            logBakFolder = self.conf.getLogBackupFolder()
            self.logger.debug("Try to restore the log folder from %s to %s." %(logBakFolder, logFolder))
            if os.path.isdir(logBakFolder):
                p = subprocess.Popen("rm -rf %s && /bin/mv -f %s %s" %(logFolder, logBakFolder, logFolder),
                                     shell = True,
                                     stdout = subprocess.PIPE,
                                     stderr = subprocess.PIPE)
                (stdout, stderr) = p.communicate()
                if p.returncode != 0:
                    self.logger.debug("Fail to restore the log folder, stdout: %s, stderr: %s" %(stdout, stderr))
                    return -1
            else:
                self.logger.debug("The log backup folder %s doesn't exist." %(logBakFolder))
                return -1

        return 0

##########################################################
# IMSS upgrade step
#########################################################

class ImssUpgradeInit(Step):
    """
    This step is to perform initialize steps.
    """
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        self.folder_mgr = g_upd_models.folder_backup_mgr
        self.service_mgr = g_upd_models.service_mgr
        self.db_mgr = g_upd_models.db_mgr
        self.rc_mgr = g_upd_models.rc_mgr
        self.rpm_status = g_upd_models.rpm_status
        self.imssPath = self.rpm_status.getImssInstalledPath()

    def apply(self):
        try:
            ret = self.folder_mgr.initialFolder()
            if ret != 0:
                return -1

            # not critical, need not to check return value
            self.service_mgr.stopImss71Services()
            self.service_mgr.stopCronJobs()
            self.rc_mgr.removeImss71RcLinks()

            # if not install imss imsscctrl and imsseuq. then there is no internal db
            if self.imssPath:
                imssiniPath = os.path.join(self.imssPath, "config", "imss.ini")
                if self.confirmImsxRole(imssiniPath) != 0:
                    self.logger.debug("Fail to get the role of base IMSx.")
                    return -1
            # fixme if only nrs or ipprofiler installed on this server
            else:
                # no imss components  installed. there is no internal DB, need not to do dbbackup
                self.conf.setRole(2)

            self.service_mgr.startDb()
            # Confirm the database configuration of the base IMSx.
            odbciniPath = self.util.getOdbcIniPath(self.rpm_status)
            if not odbciniPath:
                return -1

            if os.path.exists(Const.DEBUG_FLAG_FILE):
                parser = ConfigParser.SafeConfigParser()
                try:
                    parser.readfp(open(odbciniPath, "r"))
                    self.logger.debug("---------ImssUpgradeInit db ini file info, path:%s" % odbciniPath)
                    self.logger.debug(parser.get("IMSS", "Servername"))
                    self.logger.debug(parser.getint("IMSS", "Port"))
                    self.logger.debug(parser.get("IMSS", "Database"))
                    self.logger.debug(parser.get("IMSS", "UserName"))
                    self.logger.debug(parser.get("IMSS", "Password"))
                    self.logger.debug("---------ImssUpgradeInit initial conf db info")
                    Debugger.print_db_info(self.logger, self.conf)
                except Exception, e:
                    self.logger.debug("Fail to parse ini file %s, %s" % (odbciniPath, utility.excToString(e)))
                    return -1

            # read adminDB info into conf
            if self.util.isUsingInternalAdminDB(self.rpm_status):
                if self.confirmDatabaseConf(odbciniPath) != 0:
                    self.logger.debug("Fail to get the database configurations of base IMSx.")
                    return -1

            ret = self.checkClusterStatus(updater=self.updater)
            if ret != 0:
                return -1

            #very important
            self.db_mgr.initial()

            self.updater.append(self.res.getWording("msg.exportDBStartPrompt"))
            ret = self.db_mgr.doMieExport()
            if ret != 0:
                self.updater.append(self.res.getWording("msg.exportDBFailPrompt"))
                return -1
            self.updater.append(self.res.getWording("msg.exportDBEndPrompt"))

            self.updater.append(self.res.getWording("msg.backupDBStartPrompt"))
            ret = self.db_mgr.doPgDumpFromImss71()
            if ret != 0:
                self.updater.append(self.res.getWording("msg.backupDBFailPrompt"))
                return -1
            self.updater.append(self.res.getWording("msg.backupDBEndPrompt"))

            self.service_mgr.stopDb()

            if self.folder_mgr.doRpmFilesBack() != 0:
                return -1

            if self.folder_mgr.doPgDataFolderBack() != 0:
                return -1
        except Exception as e:
            exc_type, exc_obj, exc_tb = sys.exc_info()
            fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
            self.logger.debug("Exception message: %s    %s"%(e.message, str((exc_type, fname, exc_tb.tb_lineno))))
            return -1
        return 0

    def rollback(self):
        self.service_mgr.startCronJobs()
        self.service_mgr.stopDb()
        self.db_mgr.removeMieExport()
        self.db_mgr.removePgdumpFiles()

        if self.folder_mgr.rollbackRpmFiles() != 0:
            return -1
        if self.folder_mgr.rollbackPgDataFolder() != 0:
            return -1
        self.rc_mgr.rollbackImss71RcLinks()
        self.service_mgr.startImss71Services()
        return 0


class ImssUpgradeInstallNewVersion(Step):
    """
    This step is to install the new IMSx application to the system.
    """
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        self.cron_backup_file = os.path.join(self.conf.getBackupFolder(), "origcrontabfile")
        self.rpm_mgr = g_upd_models.rpm_mgr
        self.cron_mgr = CronMgr(conf, logger, updater)
        self.config_mgr = g_upd_models.config_mgr
        self.copy_mgr = g_upd_models.copy_mgr
    def apply(self):
        if os.path.exists(Const.DEBUG_FLAG_FILE):
            self.logger.debug("---------ImssUpgradeInstallNewVersion conf db info")
            Debugger.print_db_info(self.logger, self.conf)
            
        try:
            if self.rpm_mgr.installNewRpm(self.updater) != 0:
                return -1

            if self.copy_mgr.copyFiles() !=0:
                return -1

            if self.config_mgr.doConfigUpgrade() != 0:
                return -1

            self.config_mgr.updatePostfixConf()

            if self.config_mgr.updateMaincf()!=0:
                return -1

            # replace CONSTANTS in config files
            if self.replaceConstants(self.updater) != 0:
                return -1

            if self.cron_mgr.upgradeCron() != 0:
                return -1

            if self.exportCronSetting()!=0:
                self.logger.debug("Fail to run exportCronSetting")

            # initialize LDAPS certificate Database
            p = subprocess.Popen(
                "%s/database/init_ldap_db.sh %s" % (self.conf.getInstallPkgPath(), self.conf.getInstallPath()),
                shell=True,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE)
            (stdout, stderr) = p.communicate()
            if p.returncode != 0:
                self.logger.debug(
                    "Fail to initialize database for LDAPS certificate, stdout: %s, stderr: %s" % (stdout, stderr))

            if self.installService(self.updater) != 0:
                self.logger.debug("Fail to install imsx services.")
                return -1
        except Exception as e:
            exc_type, exc_obj, exc_tb = sys.exc_info()
            fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
            self.logger.debug("Exception message: %s    %s" % (e.message, str((exc_type, fname, exc_tb.tb_lineno))))
            return -1
        return 0

    def rollback(self):
        self.config_mgr.rollbackMaincf()
        self.config_mgr.rollbackPostfixConf()
        if self.uninstallService() != 0:
            self.logger.debug("Fail to uninstall the services")
            return -1
        self.cron_mgr.rollBackCron()
        if self.rpm_mgr.rollBackInstalledNewRpm() != 0:
            return -1
        return 0


class ImssUpgradeAdminDB(Step):
    def __init__(self, conf, res, logger, util, isCritical, updater):
        Step.__init__(self, conf, res, logger, util, isCritical, updater)
        self.db_mgr = g_upd_models.db_mgr
        self.service_mgr = g_upd_models.service_mgr
        self.rpm_mgr = g_upd_models.rpm_mgr
        self.copy_mgr = g_upd_models.copy_mgr
        self.scannerId = str(self.getLocalScannerId())
        self.is_version_update = False
        self.is_new_db_created = False

    """
    This step is to update the schema and data in adminDB.
    """

    def apply(self):
        self.is_new_db_created = False
        if os.path.exists(Const.DEBUG_FLAG_FILE):
            self.logger.debug("---------ImssUpgradeAdminDB conf db info")
            Debugger.print_db_info(self.logger, self.conf)

        try:
            # Create new database before migration
            self.updater.append(self.res.getWording("msg.installDatabaseStart"))
            # fixme db password should be vertify
            ret = self.installBundledDatabase()
            if ret != 0:
                self.updater.append(self.res.getWording("msg.installDatabaseFail"))
                return -1
            else:
                self.is_new_db_created = True
                self.updater.append(self.res.getWording("msg.installDatabaseSuccess"))

            self.service_mgr.stopDb()
            # copy old pg_hba.conf file.
            # no critical
            self.copy_mgr.copyPgConf()

            # Start the adminDB
            self.service_mgr.startDb()
            # restore data
            self.updater.append(self.res.getWording("msg.restoreDBStartPrompt"))
            if self.db_mgr.doPgRestoreToImss91() != 0:
                self.updater.append(self.res.getWording("msg.restoreDBFailPrompt"))
                return -1
            self.updater.append(self.res.getWording("msg.restoreDBEndPrompt"))

            # self.updater.append(self.res.getWording("msg.migrateDb"))
            if self.db_mgr.doEuqSchemaUpdate()!=0:
                return -1

            # Only parent need to upgrade adminDB.
            #migrateAdminDB include import MIE data
            if self.conf.getRole() == 0:
                # migrate data
                if self.migrateAdminDB(self.updater) != 0:
                    return -1

                # If it is a parent, need to import default engines and patterns.
                # depent on manager, need to start it
                subprocess.call("%s start" %(os.path.join(self.conf.getInstallPath(), "imss/script/S99MANAGER")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
                importAu = self.importDefAuComp(self.updater)
                subprocess.call("%s stop" %(os.path.join(self.conf.getInstallPath(), "imss/script/imssctl.sh")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
                subprocess.call("%s start" %(os.path.join(self.conf.getInstallPath(), "imss/script/dbctl.sh")),
                            shell = True,
                            stdout = open("/dev/null"),
                            stderr = open("/dev/null"))
                if importAu !=0:
                    self.logger.debug("Fail to import the default engines and patterns.")
                    return -1

                #From 7.0 to 9.1, need to remove previous vsapi/atse engine version, to prevent user rolling back the engine, new feature "Connected Threat Defense(Better Together)" need the new engine to support--Niel.
                #Also, we need to remove previous tmufe/tmase engine version, becase new feature "Time of Click Protection" need the new engines.
                #    vsapi engine: type=0, so the backup type=100 / atse engine: type=9, so the backup type=109  /tmufe engine: type=8, so backup type=108 / tmase engine: type=5, so backup type=105
                #1)In actually, the large object refernced by the old engine stored in databse should be removed by using "select lo_unlink(oid)", oid is the value stored in column "content" in tb_active_update.
                #  we can use "SELECT DISTINCT (loid) FROM pg_largeobject" to list all large objects.
                sql = "select content from tb_active_update where type=100 or type=109 or type=105 or type=108;"
                (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.util.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
                if e:
                    #This is not critical, because the large object is just has one copy, and it doesn't occupy much disk[sizeof(libvsapi.so)+sizeof(libatse.so) + sizeof(libtmufeng.so) + sizeof(libtmaseng.so)], 
                    #so it can be OK if we leave them in DB when error occurs here.
                    self.logger.debug("Fail to query content oid for vsapi/atse engine from tb_active_update, %s" %(utility.excToString(e)))
                else:
                    for item in ret:
                        sql = "select lo_unlink(%s);"
                        para = [item[0]]
                        (ret2, e2) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                                  self.conf.getAdminDBPort(),
                                                                  self.conf.getAdminDBUsername(),
                                                                  self.util.decryptString(self.conf.getAdminDBPassword()),
                                                                  self.conf.getAdminDBName(),
                                                                  sql,
                                                                  para)
                        if ret2 != 0:
                            # This is not critical, if failed, the large object will leave in DB.
                            self.logger.debug("Failed to delete large object from DB, %s" %(utility.excToString(e2)))
                        else:
                            self.logger.debug("Delete large object successfully, oid=%s." %(item[0]));
                #end 1)~~~

                #2) remove the records in tb_active_update   
                sql = "delete from tb_active_update where type=100 or type = 109 or type=105 or type=108;"
                (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql)
                if ret != 0:
                    self.logger.debug("Failed to execute sql command to delete old vsapi/atse engine %d, %s" %(scannerId, utility.excToString(e)))
                    return -1
                else:
                    self.logger.debug("New feature(Better Together) needs the engine upgraded, delete old vsapi/atse engine version to prevent user rolling back them.")
                #end 2)~~~

            # Both parent and children needs to update the os and app versions in tb_component_list.

            # osVer = self.getOsVersion()
            # appVer = self.getAppVersion()
            # scannerId = self.getLocalScannerId()
            # sql = "update tb_component_list set os_ver=%s, app_ver=%s where scanner_id=%s;"

            appVer = self.getAppVersion()
            osVer = self.getOsVersion()
            sql = "update tb_component_list set os_ver=%s, app_ver=%s where scanner_id=%s;"
            para = [osVer, appVer]
            if int(self.scannerId)>0:
                para.append(self.scannerId)
            else:
                sql = "update tb_component_list set os_ver=%s, app_ver=%s where ip_addr=%s;"
                para.append(self.util.getIpAddr())

            self.logger.debug("Sql:%s  Parameters: %s"%(sql,str(para)))
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to  update scanner os_ver app_ver, Exception:%s" % (utility.excToString(e)))
                return -1
            else:
                self.logger.debug("Update scanner app_ver success")

            self.is_version_update = True
            # Update EUQ database password. Because some old version password is not encrypted.
            if self.conf.getRole() == 0:
                sql = "select password, db_id from tb_euq_db_info;"
                (ret, e) = self.util.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                           self.conf.getAdminDBPort(),
                                                           self.conf.getAdminDBUsername(),
                                                           self.util.decryptString(self.conf.getAdminDBPassword()),
                                                           self.conf.getAdminDBName(),
                                                           sql)
                if e:
                    self.logger.debug("Fail to query euq db info from db, %s" % (utility.excToString(e)))
                    return -1

                for item in ret:
                    if len(item) != 2:
                        self.logger.debug("Failed to get password and db_id from table")
                        return -1
                    plain_passwd = item[0]
                    db_id = item[1]
                    password = self.util.encryptString(plain_passwd)
                    self.logger.debug("The euq db info for db_id: %s." % db_id)
                    sql = "update tb_euq_db_info set password=%s where db_id=%s;"
                    para = [password, db_id]
                    (ret2, e2) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                                  self.conf.getAdminDBPort(),
                                                                  self.conf.getAdminDBUsername(),
                                                                  self.util.decryptString(self.conf.getAdminDBPassword()),
                                                                  self.conf.getAdminDBName(),
                                                                  sql,
                                                                  para)
                    if ret2 != 0:
                        self.logger.debug(
                            "Failed to execute sql command to update euq db passwd: %s" % (utility.excToString(e2)))
                        return -1

            self.logger.debug("update euq db password successfully")
            ret = self.rpm_mgr.removeImss71Rpms()
            if ret!=0:
                self.logger.debug("Fail to remove old rpm package info.")
                return -1
            self.logger.debug("The imss7.1 rpm info is removed from rpmdb")
            #debug code
            # while os.path.exists("/tmp/stop"):
            #     time.sleep(3)
            #     if os.path.lexists("/tmp/roll"):
            #         return -1

            #do some jobs after install is finished.
            post = PostUpgradeAction(self.conf, self.logger, self.util)
            # copy uninstall.sh to imss/backup folder

            #post work maybe fail but it doesn't effect the result of upgrade
            post.doUpgradePostWork(self.getLocalScannerId())

            self.service_mgr.startImss91Services()
            self.logger.debug("Imss 9.1 services has been started")
            self.service_mgr.startCronJobs()
            self.logger.debug("Cron service has been started")

        except Exception as e:
            exc_type, exc_obj, exc_tb = sys.exc_info()
            fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
            self.logger.debug("Exception message: %s    %s" % (e.message, str((exc_type, fname, exc_tb.tb_lineno))))
            return -1
        return 0

    def rollback(self):
        #roll back
        if self.is_version_update:
            sql = "update tb_component_list set app_ver='7.1' where scanner_id=%s;"
            para = [self.scannerId]
            (ret, e) = self.util.executeUpdateStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        self.util.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),
                                                        sql,
                                                        para)
            if ret != 0:
                self.logger.debug("Failed to rollback app version for scanner %s, Exception:%s" % (
                    self.scannerId, utility.excToString(e)))
            else:
                self.logger.debug(
                    "Rollback version updated scanner %s in tb_component_list, set app_ver='7.1'" % (self.scannerId))

        # Only need to stop DB here. The DB migration will be put in a transaction
        # and the rollback will be handled by database itself.
        self.service_mgr.stopImss91Services()
        self.service_mgr.stopDb()
        #clear up /var/imss/pgdata folder
        if os.path.exists(Const.PGDATA_SRC_FOLDER)  and self.is_new_db_created:
            ret = os.system("/bin/rm -rf %s >/dev/null 2>&1"%Const.PGDATA_SRC_FOLDER)
            if ret != 0:
                self.logger.debug("Fail to clear up /var/imss/pgdata")
        return 0


def ctlPostgreSQL(cmd,conf):
    dbctlPath = "%s/imss/script/dbctl.sh" %(conf.getInstallPath())
    subprocess.call([dbctlPath, cmd],
                        stdout = open("/dev/null"),
                        stderr = open("/dev/null"),env=utility.getEnvForRunCmd(conf))