#!/usr/bin/env python
# -*- coding: utf-8 -*-
import ConfigParser
import socket
import subprocess
import psycopg2 as dbapi
import pwd
import grp
import re
import os
import errno
import shutil
import struct

from model import Const


def excToString(e):
        """
        Represent the exception in a user-friendly format, e.g:
        exception: OSError - [Errno 20] Not a directory: 'test'
        
        e: the exception happened.
        
        return: the formatted string.
        """
        return "exception: %s - %s" %(e.__class__.__name__, e)


#put those two env together, because when minimal install, there some no so error
# def getEnvForRunCmd(conf):
#         environ = os.environ.copy()
#         environ["LD_LIBRARY_PATH"] = "%s/imssbase/lib" % conf.getInstallPkgPath()
#         return environ
def getEnvForRunCmd(conf):
        environ = os.environ.copy()
        environ["LD_LIBRARY_PATH"] = "$LD_LIBRARY_PATH:%s/imssbase/lib:%s/imss/lib:%s/imss/PostgreSQL/lib:%s/imss/lib/curl" % \
                                     ( conf.getInstallPkgPath(),conf.getInstallPath(), conf.getInstallPath(), conf.getInstallPath())
        return environ

class Utility(object):
    """
    This class is a collection of utilities, e.g, check network connection, 
    check disk size, etc.
    """
    def __init__(self,config,logger):
        """
        Initialize some internal variables. 
        """
        # The path of binaries that may be called. Actually it is imssbase/bin.
        self.extBinPath = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))),
                                       "imssbase",
                                       "bin")
        self.conf = config
        self.logger = logger
        self.isInternalDB = None
        self.isInternalAdminDB = None

    def isConnectable(self, address, port):
        """
        Test if a host can be connected. This is for IPv4 only.
        
        address: the address of the host.
        port: the port of the host.
        
        return: a tuple (retcode, exception). Retcode True means successful, in 
        this case, exception is None. Retcode False means failed, and exception
        is set.
        """
        ret = True
        exception = None
        
        s = socket.socket(socket.AF_INET)
        # Set 2 seconds timeout for connect attempt.
        s.settimeout(2)
        try:
            s.connect((address, port))
        except Exception, e:
            ret = False
            exception = e
        s.close()
        
        return (ret, exception)

    def generateDBName(self, name):
        return name


    @staticmethod
    def getRpmInstalledPath(rpm):
        rc, rstr = Utility.getResultFromCmd(["rpm -ql %s | tee 2> /dev/null | head -1" % rpm])
        #rc, rstr = Utility.getResultFromCmd(["rpm -q --queryformat %%{instprefixes} %s" % rpm])
        return rstr

    @staticmethod
    def isRpmInstalled( name):
        """
        Test if an rpm is installed.
        
        name: the name of the rpm.
        
        return: True if the rpm is installed, otherwise return False.
        """
        ret = subprocess.call(["rpm", "-q", name],
                              stdout = open("/dev/null"),
                              stderr = open("/dev/null"))
        # If ret code is 0, means the rpm exists.
        if ret == 0:
            return True
        else:
            return False
    
    def installRpm(self, filePath, extraArgs):
        """
        Install an rpm.
        
        filepath: the full file path of the rpm file.
        extraArgs: extra arguments in a list provided to the rpm command.
        
        return: (ret, stdout, stderr) tuple if successful, ret=0, otherwise,
        ret=-1. The stdout and stderr are the output of the rpm command.
        """
        p = subprocess.Popen(["rpm", "-i"] + extraArgs + [filePath], 
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode == 0:
            return (0, stdout, stderr)
        else:
            return (-1, stdout, stderr)
    
    def uninstallRpm(self, name, extraArgs):
        """
        Uninstall an rpm.
        
        name: the name of the installed rpm.
        extraArgs: extra arguments in a list provided to the rpm command.
        
        return: (ret, stdout, stderr) tuple if successful, ret=0, otherwise,
        ret=-1. The stdout and stderr are the output of the rpm command.
        """
        p = subprocess.Popen(["rpm", "-e"] + extraArgs + [name],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode == 0:
            return (0, stdout, stderr)
        else:
            return (-1, stdout, stderr)
        
    def isDatabaseConnectable(self, address, port, user, password, dbname="postgres"):
        """
        Check if an database can be connected or not.
        
        address: address of the database. None means connect to UNIX domain socket.
        port: port of the database.
        user: the user name of the database.
        password: the password of the database. None means no password needed.
        dbname: the name of the database, postgres is the default database.
        
        return: (ret, exception), if database can be connected, ret=0, 
        exception=None, otherwise, ret=-1, exception=the exception reported.
        """
        ret = 0
        exception = None
        connString = ""
        # Compose connection string.
        if address != None:
            connString += "host=%s " %(address)
        connString += "port=%d user=%s dbname=%s" %(port, user, dbname)
        if password != None:
            connString += " password=%s" %(password)
             
        try:
            dbapi.connect(connString)
        except Exception, e:
            ret = -1
            exception = e
            
        return (ret, exception)
    
    def executeSqlFile(self, address, port, user, password, dbname, filePath):
        """
        Execute sql statements in a file.
        
        address: address of the database. None means connect to local UNIX domain socket.
        port: port of the database.
        user: the user name of the database.
        password: the password of the database. None means no password.
        dbname: the name of the database.
        
        return: (returncode, exception) tuple, if success, returncode=0, 
        exception is None. otherwise, returncode=-1, exception is not None.
        """
        connString = ""
        # Compose connection string.
        if address != None:
            connString += "host=%s " %(address)
        connString += "port=%d user=%s dbname=%s" %(port, user, dbname)
        if password != None:
            connString += " password=%s" %(password)

        conn = None
        try:
            with dbapi.connect(connString) as conn:
                # Let the sql script itself control transaction, this is the
                # default of psql.
                conn.autocommit = True
                with conn.cursor() as cur:
                    cur.execute(open(filePath, "r").read())
        except Exception, e:
            return (-1, e)
        finally:
            if conn:
                conn.close()
        
        return (0, None)

    @staticmethod
    def executeQueryStatement(address, port, user, password, dbname, sql, para=None):
        """
        Execute an sql statement that is used to query data from database.
        
        address: address of the database. None means connect to local UNIX domain socket.
        port: port of the database.
        user: the user name of the database.
        password: the password of the database. None means no password.
        dbname: the name of the database.
        sql: the sql statement that will be executed.
        para: a list contains the query parameters.
        
        return: (result, exception) tuple, if no error happens, result is not None, 
        exception is None. If there's any error, exception is specified, and 
        result is None. Result is a list, and it can be empty even if no error happens.
        """
        connString = ""
        # Compose connection string.
        if address != None:
            connString += "host=%s " %(address)
        connString += "port=%s user=%s dbname=%s" %(str(port), user, dbname)
        if password != None:
            connString += " password=%s" %(password)

        conn = None
        try:
            with dbapi.connect(connString) as conn:
                conn.autocommit = True
                with conn.cursor() as cur:
                    cur.execute(sql, para)
                    result = cur.fetchall()
        except Exception, e:
            return (None, e)
        finally:
            if conn:
                conn.close()
        
        return (result, None)
    
    def executeUpdateStatement(self, address, port, user, password, dbname, sql, para=None):
        """
        Execute an sql statement that is used to update data in database.
        
        address: address of the database.
        port: port of the database.
        user: the user name of the database.
        password: the password of the database.
        dbname: the name of the database.
        sql: the sql statement that will be executed.
        para: a list that contains the query parameters.
        
        return: (returncode, exception) tuple, if success, returncode=0, 
        exception is None. otherwise, returncode=-1, exception is not None.
        """
        connString = ""
        # Compose connection string.
        if address != None:
            connString += "host=%s " %(address)
        connString += "port=%s user=%s dbname=%s" %(str(port), user, dbname)
        if password != None:
            connString += " password=%s" %(password)

        conn = None
        try:
            with dbapi.connect(connString) as conn:
                conn.autocommit = True
                with conn.cursor() as cur:
                    cur.execute(sql, para)
        except Exception, e:
            return (-1, e)
        finally:
            if conn:
                conn.close()
        
        return (0, None)
    
    def getDiskFreeSize(self, filePath):
        """
        Get the free size in bytes of the disk where the file lives. If the 
        file doesn't exist, the function will try its parent directory 
        recursively, until the root directory.
        
        filePath: the file path on the disk.
        
        return: the disk free size in bytes. 
        """
        while filePath != "/":
            if os.path.exists(filePath):
                break
            else:
                filePath = os.path.dirname(filePath)
        
        stat = os.statvfs(filePath)
        size = stat.f_bavail * stat.f_frsize
        return size
    
    def getTotalDiskSize(self):
        """
        Get the total disk size in bytes of the machine.
        
        return: the total disk size in bytes, return -1 if any error.
        """
        # Use df command to get total disk size. First remove the first line, then filter the line that have no % sign, get
        # the size column, then add them all.
        cmd = """df --block-size=1024 | sed '1d' | grep '%' | tr -s ' ' | cut -d ' ' -f 2 | awk '{s+=$1} END {printf "%.0f", s}'"""
        p = subprocess.Popen(cmd,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE,
                             shell=True)
        (stdout, stderr) = p.communicate()
        if p.returncode == 0:
            return int(stdout) * 1024
        else:
            return -1
    
    def getMemSize(self):
        """
        Get the total size of memory in bytes on Linux.
        
        return: memory size in bytes.
        """
        meminfo = {"MemTotal" : 0}
        with open("/proc/meminfo") as f:
            for line in f:
                # Get the memory size and transfer to byte format.
                meminfo[line.split()[0][:-1]] = int(line.split()[1]) * 1000
        
        return meminfo["MemTotal"]

    def getIcpName(self):
        """
        Get the interface name of current ICP.

        return: the name of ICP, or "" if failed.
        """
        ret = ""
        # For IMSVA, icp name is saved in a file during OS installation.
        if self.conf.getProduct() == 1:
            try:
                with open("/etc/icp.conf") as f:
                    ret = f.readline().strip()
            except Exception, e:
                self.logger.debug("Cannot get ICP interface, %s" % (excToString(e)))
        # For IMSS, use the first nic in ifconfig output as the icp.
        else:
            nics = self.getNicInfo()
            if not nics:
                self.logger.debug("Cannot get the nic info of this machine.")
                ret = ""
            else:
                ret = nics.keys()[0]
                self.logger.debug("Use %s as ICP." % (ret))

        return ret
    
    def getCurrentUsername(self):
        """
        Get the current login user name.
        
        return: the current login user name.
        """
        return pwd.getpwuid(os.getuid()).pw_name
    
    def isUserExist(self, name):
        """
        Check whether an user exists on the system.
        
        return: True if exist, False if not.
        """
        for user in pwd.getpwall():
            if user.pw_name == name:
                return True
        
        return False
    
    def isGroupExist(self, name):
        """
        Check whether a group exists on the system.
        
        return: True if exist, False if not. 
        """
        for group in grp.getgrall():
            if group.gr_name == name:
                return True
        
        return False
    
    def addUser(self, name, group, home, shell):
        """
        Add an user to local machine.
        
        name: the name of the user.
        group: the group of the user.
        home: the home directory path of the user, empty or None means not to create home directory.
        shell: the login shell of the user.
        
        return: (errorcode, errorstring) return (0, "") if no error, (-1, description)
                if error happens.
        """
        if self.isUserExist(name):
            return (0, "")
        
        binPath = "/usr/sbin/useradd"
        cmd = []
        if home:
            cmd = [binPath, name, "-g", group, "-d", home, "-s", shell]
        else:
            cmd = [binPath, name, "-g", group, "-M", "-s", shell]
        p = subprocess.Popen(cmd,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (_, stderr) = p.communicate()
        if p.returncode != 0:
            return (-1, stderr)
        return (0, "")
        
    def delUser(self, name):
        """
        Delete an user.
        
        name: the name of the user.
        
        return: (errorcode, errorstring) return (0, "") if no error, (-1, description)
                if error happens.
        """
        if not self.isUserExist(name):
            return (0, "")
        
        binPath = "/usr/sbin/userdel"
        p = subprocess.Popen([binPath, "-f", name],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (_, stderr) = p.communicate()
        if p.returncode != 0:
            return (-1, stderr)
        return (0, "")
    
    def addGroup(self, name):
        """
        Add an group to local machine.
        
        name: the name of the group.
        
        return: (errorcode, errorstring) return (0, "") if no error, (-1, description)
                if error happens.
        """
        if self.isGroupExist(name):
            return (0, "")
        
        binPath = "/usr/sbin/groupadd"
        p = subprocess.Popen([binPath, name],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (_, stderr) = p.communicate()
        if p.returncode != 0:
            return (-1, stderr)
        return (0, "")
    
    def delGroup(self, name):
        """
        Delete an group.
        
        name: the name of the group.
        
        return: (errorcode, errorstring) return (0, "") if no error, (-1, description)
                if error happens.
        """
        if not self.isGroupExist(name):
            return (0, "")
        
        binPath = "/usr/sbin/groupdel"
        p = subprocess.Popen([binPath, name],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (_, stderr) = p.communicate()
        if p.returncode != 0:
            return (-1, stderr)
        return (0, "")

    def encryptString(self, msg):
        """
        Encrypt a string.
        
        return: the encrypted string.
        """
        cmdPath = os.path.join(self.extBinPath, "passwd_util")
        p = subprocess.Popen([cmdPath, "-e", msg],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE, env=getEnvForRunCmd(self.conf))
        (stdout, _) = p.communicate()
        return stdout
    
    def decryptString(self, msg):
        """
        Decrypt a string.
        
        return: the decrypted string or the original string if fail to decrypt.
        """
        cmdPath = os.path.join(self.extBinPath, "passwd_util")
        p = subprocess.Popen([cmdPath, "-d", msg],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE, env=getEnvForRunCmd(self.conf))
        (stdout, _) = p.communicate()
        return stdout

    def replaceInFile(self, rules, filePath):
        """
        Replace the patterns in a file. The file will be replaced one line by
        one line.
        
        rules: a list of (pattern, replacement) tuples, each tuple in the list
        specifies a replacement rule, the first element (pattern) of the tuple
        is the pattern you want to match, the second element (replacement) is 
        the content after replacing.
        filePath: the path of file to be modified.
        
        return: (ret, exception) tuple, if successful, ret=0, exception is None,
        otherwise, ret=-1, exception is specified. 
        """
        changedContents = []
        try:                
            with open(filePath, "r") as origFile:
                for line in origFile:
                    # Write the replace results to a list.
                    changedLine = line
                    for (pattern, repl) in rules:
                        # Convert repl to string
                        repl = str(repl)
                        changedLine = re.sub(pattern, repl, changedLine)
                    changedContents.append(changedLine)  
            # Write contents back to original file.
            with open(filePath, "w+") as origFile:
                origFile.write("".join(changedContents))
            return (0, None)
        except Exception, e:
            return (-1, e) 
    
    def createDir(self, path, mode, existOK=True):
        """
        Works like mkdir -p, create the directories recursively.
        
        path: the directory you want to created.
        mode: the mode of the created directory, e.g., 0755(octal)
        existOK: if true, the functions returns OK even if the directory exists,
        otherwise, the fucntion returns fail if the directory exists.
        
        return: (ret, exception) tuple, if successful, ret=0, exception is None,
        otherwise, ret=-1, exception is specified. 
        """
        try:
            os.makedirs(path, mode)
        except OSError, e:
            if existOK and e.errno == errno.EEXIST and os.path.isdir(path):
                return (0, None)
            else:
                return (-1, e)
        return (0, None)
     
    def getLinkDst(self, path):
        """
        Get the destination of the symbolic link.
        
        path: the path of the symbolic link.
        
        return: the path of the destination.
        """
        if os.path.islink(path):
            dst = os.readlink(path)
            if os.path.isabs(dst):
                return dst
            else:
                return os.path.join(os.path.dirname(path), dst)
        else:
            return path

    def removePath(self, path):
        """
        Remove the file or directory. Make use of rm command.
        
        path: the path of a file or directory.
        
        return: (ret, exception) tuple, if successful, ret=0, exception is None,
        otherwise, ret=-1, exception is specified. 
        """
        p = subprocess.Popen("rm -rf %s" %(path),
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE,
                             shell = True)
        (_, stderr) = p.communicate()
        if p.returncode != 0:
            return (-1, Exception(stderr))
        else:
            return (0, None)

    def getIpAddr(self):
        ip = "0"
        icp = self.getIcpName()
        if not icp:
            self.logger.debug("Fail to get icp device name.")
        nicInfo = self.getNicInfo()
        if nicInfo.has_key(icp):
            self.logger.debug("The icp %s exist on this machine." % (icp))
            ip = nicInfo[icp][0]
        return ip

    def getNicInfo(self):
        """
        Get the network interface cards information by parsing the output of
        ifconfig command, python doesn't have a built-in way to do this.
                
        return: a dict, key is device name, value is a list of which the format is 
                [IPv4 address, mask, mac address, is DHCP or not].
        """
        ret = {}
        
        if Utility.isOSVer7orLatter():
            # RHEL7 has no ifconfig in minimal installation, so we have to use another command to get nic info
            p = subprocess.Popen(["ip", "address"],
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, _) = p.communicate()
            if p.returncode != 0:
                return ret
            
            device = None
            ip = None
            mask = None
            mac = None
            
            for line in stdout.split("\n"):
                result = re.search(r"^\d+:\s+(.+):", line)
                if result != None:
                    # We met a new nic, now check the last nic info and put it into the result if possible
                    if device != None and ip != None and mask != None and mac != None:
                        ret[device] = [ip, mask, mac, False]
                    # Clear the last nic values
                    device = None
                    ip = None
                    mask = None
                    mac = None
                    
                    # Begin to handle the new nic
                    device = result.group(1)
                    # Skip loopback
                    if device == "lo":
                        device = None
                    continue
                
                # If not get a device skip to next line
                if device == None:
                    continue
                
                # Check mac address
                result = re.search(r"^\s+link/ether\s+([0-9a-zA-Z:]+)\s+", line)
                if result != None:
                    mac = result.group(1)
                    continue
                
                # Check ip and mask
                result = re.search(r"^\s+inet\s+([0-9\.]+)/([0-9]+)\s+", line)
                if result != None:
                    ip = result.group(1)
                    mask = '.'.join([str((0xffffffff << (32 - int(result.group(2))) >> i) & 0xff) for i in [24, 16, 8, 0]])
                    continue
            
            # Process the last nic
            if device != None and ip != None and mask != None and mac != None:
                ret[device] = [ip, mask, mac, False]
        else:
            p = subprocess.Popen(["ifconfig"],
                                 stdout = subprocess.PIPE,
                                 stderr = subprocess.PIPE)
            (stdout, _) = p.communicate()
            if p.returncode != 0:
                return ret
            
            texts = stdout.strip().split("\n\n")
            for item in texts:
                device = None
                ip = None
                mask = None
                mac = None
        
                result = re.search(r"^(\w+)[\s+|:]", item)
                if result == None:
                    continue
                else:
                    device = result.group(1)
                    # Skip loopback
                    if device == "lo":
                        continue
                
                result = re.search(r"inet\s+(?:addr:)?(\d+\.\d+\.\d+\.\d+)\s", item)
                if result == None:
                    continue
                else:
                    ip = result.group(1)
                    
                result = re.search(r"(?:Mask:|netmask\s)(\d+\.\d+\.\d+\.\d+)\s", item)
                if result == None:
                    continue
                else:
                    mask = result.group(1)
                
                result = re.search(r"(?:HWaddr|ether)\s(\w{2}:\w{2}:\w{2}:\w{2}:\w{2}:\w{2})", item)
                if result == None:
                    continue
                else:
                    mac = result.group(1)
                    
                ret[device] = [ip, mask, mac, False]
        
        # Get whether the nic is using DHCP or not
        for nic, info in ret.iteritems():
            confPath = "/etc/sysconfig/network-scripts/ifcfg-%s" %(nic)
            if os.path.isfile(confPath):
                with open(confPath) as f:
                    if re.search(r'^BOOTPROTO="?dhcp"?\s*$', f.read(), re.MULTILINE|re.IGNORECASE):
                        info[3] = True
        
        return ret
    
    def getLocalCronJobs(self, user):
        """
        Get the cron jobs of an user. This function calls OS crontab command.
        
        user: the name of the user.
        
        return: a tuple, format is (retcode, stdout, stderr), if no error, retcode
                is 0, otherwise it is -1, stdout and stderr are outputs of crontab
                command. If retcode is 0, stdout is the content of the crontab. 
        """
        ret = 0
        p = subprocess.Popen(["crontab", "-l", "-u", "%s" %(user)],
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            ret = -1
        
        return (ret, stdout, stderr)

    def addNewCronJob(self,job):
        cmd = """(crontab -l ; echo "%s") 2>&1 | grep -v "no crontab" | sort | uniq | crontab -"""%job
        p = subprocess.Popen([cmd],
                             shell=True,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE)
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            self.logger.debug(
                "Fail to add a new cron job, stdout: %s, stderr: %s, Command:%s" % (stdout, stderr,cmd))
            return -1
        return 0

    def setLocalCronJobs(self, user, content):
        """
        Set the cron jobs of an user. This function calls OS crontab command.
        
        user: the name of the user.
        content: the new crontab content.
        
        return: a tuple, format is (retcode, stdout, stderr), if no error, retcode
                is 0, otherwise it is -1, stdout and stderr are outputs of crontab
                command.
        """
        ret = 0
        p = subprocess.Popen(["crontab", "-u", "%s" %(user), "-"],
                             stdin = subprocess.PIPE,
                             stdout = subprocess.PIPE,
                             stderr = subprocess.PIPE)
        (stdout, stderr) = p.communicate(content)
        if p.returncode != 0:
            ret = -1
        
        return (ret, stdout, stderr)
        
    def isLocalAddress(self, address):
        """
        Check whether the address belongs to local machine, i.e., the address
        is loopback or the IP address of local network interfaces, or can be
        resolved to local machine's IP.
        """
        nicInfo = self.getNicInfo()
        # Collect all local addresses.
        localAddrs = ["127.0.0.1"]
        for item in nicInfo.values():
            localAddrs.append(item[0])
        # The address may be a hostname, so we have to try resolve it to IP.
        addrs = [address]
        # Currently only handle IPv4 address
        try:
            for item in socket.getaddrinfo(address, None, socket.AF_INET):
                addrs.append(item[4][0])
        except:
            # Ignore exceptions.
            pass
        
        return not frozenset(localAddrs).isdisjoint(frozenset(addrs))

    @staticmethod
    def isOSVer7orLatter():
        # add Centos support for more simple test
        rls_file_path = ""
        match = ""
        matchVer8 = ""
        ret = False

        if os.path.exists("/etc/centos-release"):
            rls_file_path = "/etc/centos-release"
            match = "CentOS Linux release (7|8|9|10)\."
            matchVer8 = "CentOS Linux release (8|9|10)\."
        elif os.path.exists("/etc/redhat-release"):
            rls_file_path = "/etc/redhat-release"
            match = "Red Hat Enterprise Linux Server release (7|8|9|10)\."
            matchVer8 = "Red Hat Enterprise Linux release (8|9|10)\."

        with open(rls_file_path) as f:
                line = f.readline()
                if re.match(match, line) or re.match(matchVer8, line):
                    ret = True
        return ret

    def dottedQuadToNum(self, ip):
        "convert decimal dotted quad string to long integer"
        return struct.unpack('L', socket.inet_aton(ip))[0]

    def checkIPInSameNetwork(self, ip1, netmask, ip2):
        return (self.dottedQuadToNum(ip1) & self.dottedQuadToNum(netmask)) == \
               (self.dottedQuadToNum(ip2) & self.dottedQuadToNum(netmask))

    def getIPFromAllNicsByParentIP(self, parentIP):
        ret = ""
        try:
            for key, value in self.getNicInfo().items():
                if self.checkIPInSameNetwork(value[0], value[1], parentIP) is True:
                    ret = value[0]
                    break
        except:
            pass
        return ret

    @staticmethod
    def getResultFromCmd(args):
        """return empty str if error happens"""
        if type(args) != list:
            return (-1, "Error input cmd")
        p = subprocess.Popen(args,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, shell=True)
        # stdout return all the data until EOF
        (stdout, stderr) = p.communicate()
        if p.returncode != 0:
            # log err
            return (-1, stderr)
        else:
            return (0, stdout.strip())

    # 1. only install one of  imss, imsscctrl, imsseuq. there can be internal DB
    # 2. internalDB can only be checked through  postgres's path
    # 3. this function depends on the path of postgres process.
    # 4. so. if internal postgres is not runing . this function won't work
    def isUsingInternalDB(self, force_check=False):
        if self.isInternalDB != None and force_check == False:
            self.logger.debug("Use previous check result, isInternalDB:"+str(self.isInternalDB))
            return self.isInternalDB
        imss_path = Const.INTERNAL_DB_PATH
        if not imss_path:
            self.isInternalDB = False
            return self.isInternalDB
        # kves   /opt/trend/imss is hard code
        rc, rl = Utility.getResultFromCmd(["""ps axo pid,args|grep postgres|awk -F " "  '{print $1}' """])
        for pid in rl.split("\n"):
            code, path = Utility.getResultFromCmd(["readlink -f /proc/%s/exe" % pid.strip()])
            if code == 0 and path.find(imss_path) == 0:
                self.isInternalDB = True
                return self.isInternalDB
        self.isInternalDB = False
        self.logger.debug("isInternalDB: %s. Command result:%s"%(str(self.isInternalDB), path))
        return self.isInternalDB

    def getOdbcIniPath(self, rpm_status):
        imssPath = rpm_status.getImssInstalledPath()
        odbciniPath = ""
        if imssPath:
            odbciniPath = os.path.join(imssPath, "config", "odbc.ini")
        elif rpm_status.isIppInstalled():
            odbciniPath = os.path.join(rpm_status.getRpmInfo(Const.RPM_IPP, Const.RPM_INSTALL_PATH), "config",
                                       "odbc.ini")
        elif rpm_status.isNrsInstalled():
            # nrs odbc.ini path
            odbciniPath = os.path.join(rpm_status.getRpmInfo(Const.RPM_NRS, Const.RPM_INSTALL_PATH), "odbc.ini")
        if not os.path.exists(odbciniPath):
            self.logger.debug("Fail to get the odbc.ini , the path is not exist, file path: %s" % odbciniPath)
            return ""

        self.logger.debug("odbc.ini path is :%s" % (odbciniPath))
        return odbciniPath


    def isUsingInternalAdminDB(self,rpm_status, force_check=False):
        if self.isInternalAdminDB != None and force_check == False:
            return self.isInternalAdminDB

        odbciniPath = self.getOdbcIniPath(rpm_status)
        if odbciniPath:
            parser = ConfigParser.SafeConfigParser()
            try:
                parser.readfp(open(odbciniPath, "r"))
                self.conf.setAdminDBAddress(parser.get("IMSS", "Servername"))
                self.conf.setAdminDBPort(parser.getint("IMSS", "Port"))
                self.conf.setAdminDBName(parser.get("IMSS", "Database"))
                self.conf.setAdminDBUsername(parser.get("IMSS", "UserName"))
                self.conf.setAdminDBPassword(parser.get("IMSS", "Password"))
            except Exception, e:
                self.logger.debug("Fail to parse ini file %s, %s" % (odbciniPath, self.excToString(e)))
                self.isInternalAdminDB = False
                return self.isInternalAdminDB

        if not self.isUsingInternalDB(force_check):
            self.isInternalAdminDB = False
            return self.isInternalAdminDB

        if "127.0.0.1" in self.conf.getAdminDBAddress():
            self.isInternalAdminDB = True
            return self.isInternalAdminDB

        for val in  self.getNicInfo().values():
            if val[0].strip() in self.conf.getAdminDBAddress():
                self.isInternalAdminDB = True
                return self.isInternalAdminDB

        self.isInternalAdminDB = False
        return self.isInternalAdminDB

    @staticmethod
    def isOSRedHat4(logger):
        ret = False
        try:
            with open("/etc/redhat-release") as f:
                version = f.readline()
                if version.find("Red Hat Enterprise Linux AS release 4") >=0:
                    ret = True
        except  Exception as e:
            logger.error("Exception when check version of OS: %s"+str(e))
        return ret

    def isDbNewVersion(self, ip, port, usr, psw):
        ret,ept = self.executeQueryStatement(ip, port, usr, psw, "postgres", 'select version()')
        if ept == None:
            if str(ret).lower().find(Const.POSTGRESQL_CHECK_VERSION) >= 0:
                return 0
            else:
                self.logger.debug("The db version is not psql 9.2. address:%s version is %s"%(ip,str(ret)))
        else:
            self.logger.debug("Fail to check db version. Address:%s  Exception:%s"%(ip,excToString(ept)))
        return -1

    # check is DB has table tb_global_setting
    #then we will think the DB has been dump and restore
    def isDbHasDumpAndRestore(self,ip, port, usr, psw,dbname):
        sql = "SELECT EXISTS(SELECT * FROM information_schema.tables WHERE table_catalog = '%s' AND table_name = 'tb_global_setting');"%dbname
        # if self.isDbNewVersion(ip, port, usr, psw) !=0:
        (table_exist, ept) = self.executeQueryStatement(ip,
                                                           port,
                                                           usr,
                                                           psw,
                                                           dbname, sql, para=None)
        if ept != None:
            self.logger.debug("Fail to judge if the database has been dumped and restored. Exception:%s"%excToString(ept))
        if str(table_exist).lower().find("t") >=0:
            return 0
        return -1

    #return format:
    #[(server,port,username,db_name)]
    def getNoUpgradeEuqDbList(self):
        ret_str = ""
        #return value
        #(server,port,username,db_name,db_id)
        ret_list = []
        sql = "select server,port,username,password,db_name,db_id from tb_euq_db_info;"
        (euq_db_list, ept) = Utility.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                        self.conf.getAdminDBPort(),
                                                        self.conf.getAdminDBUsername(),
                                                        #this password is read from odbc.ini  imss7.1 that is not encrypt
                                                        self.decryptString(self.conf.getAdminDBPassword()),
                                                        self.conf.getAdminDBName(),sql, para=None)
        if ept == None:
            for db in euq_db_list:
                if self.isDbNewVersion(db[0],db[1],db[2],self.decryptString(db[3] )) != 0:
                    ret_list.append((db[0],db[1],db[2],db[4],db[5]))
        else:
            self.logger.debug("Fail to get euq db list. Exception %s"%excToString(ept))
        return ret_list



    # return format:
    # [(address, version,  installed_components)]
    def getNoUpgradeComponentList(self):
        ret_list = []
        ret_str = ""
        sql = "select  app_ver,ip_addr,is_master,policy,euq,nrs,ipprofiler from tb_component_list"
        (ret, ept) = Utility.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                   self.conf.getAdminDBPort(),
                                                   self.conf.getAdminDBUsername(),
                                                        #this password is read from odbc.ini  imss7.1 that is not encrypt
                                                   self.decryptString(self.conf.getAdminDBPassword()),
                                                   self.conf.getAdminDBName(), sql, para=None)
        if ept == None:
            for row in ret:
                if row[0].strip().find("9.1") !=0:
                    component = ""
                    if row[2]!=0:
                        component+=("imsscctrl,")
                    if row[3]!= 0:
                        component+=("imss,")
                    if row[4] != 0:
                        component+=("imsseuq,")
                    if row[5] != 0:
                        component+=("nrs,")
                    if row[6] != 0:
                        component+=("ipprofiler")
                    ret_list.append((row[1],row[0],component.strip(",")))
        else:
            self.logger.debug("Fail to get no upgrade components. Exception: %s"%excToString(ept))
        return ret_list

    def isDbExist(self, address, port, username, password, dbname):
        sql = "select datname from pg_database where datname=%s;"
        (ret, e) = self.executeQueryStatement(address,
                                                   port,
                                                   username,
                                                   password,
                                                   "postgres",
                                                   sql,
                                                   [dbname])
        if e == None:
            # Non-empty result means exist
            if len(ret)>0:
                return 0
            else:
                return -1
        else:
            self.logger.debug("Failed to check whether db exist, %s" % (excToString(e)))
            return -1

    def checkDbUserRole(self, address, port, username, password):
        """
        Check if user has superuser role

        address: address of the db.
        port: port of the db.
        username: the username can be used to access db.
        password: the password can be used to access db.

        return: 0 if user is superuser, otherwise return -1.
        """
        sql = "select * from pg_roles where rolname=%s and rolsuper='t';"
        (ret, e) = self.executeQueryStatement(address,
                                                   port,
                                                   username,
                                                   password,
                                                   "postgres",
                                                   sql,
                                                   [username])
        if e == None:
            # Non-empty result means exist
            if ret:
                return 0
            else:
                return -1
        else:
            self.logger.debug("Failed to check whether db exist, %s" % (excToString(e)))
            return -1

    # if scanner_id in imss.ini is 1. it's parent
    def isParent(self, rpm_status):
        imssPath = rpm_status.getImssInstalledPath()
        if not imssPath:
            return -1
        imssiniPath = os.path.join(imssPath, "config", "imss.ini")
        try:
            cp = ConfigParser.SafeConfigParser()
            if os.path.exists(imssiniPath):
                cp.readfp(open(imssiniPath, "r"))
                scanner_id = cp.getint("imss_manager", "scanner_id")
                self.logger.debug(
                    "Read from imss.ini: scanner_id:%d " % (scanner_id))
                if scanner_id == 1:
                    return 0
                else:
                    return -1
            else:
                self.logger.debug("Config file is not exist, path is %s" % imssiniPath)
        except Exception as e:
            self.logger.debug("Fail to read scanner_id from imss.ini.Exception: %s" % excToString(e))
            return -1
        return -1

    def isParentErsEnable(self):
        sql = "select value from tb_global_setting where name='ers_enabled';"
        (ret, e) = self.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
        if e:
            self.logger.debug("Fail to query ParentErsEnable from db, %s" %(excToString(e)))
            return 0
        else:
            if len(ret[0]) == 1:
                isEnable = ret[0][0]
                self.logger.debug("Query ParentErsEnable from db, %s" % (isEnable))
                if isEnable == "yes":
                    return 1
                else:
                    return 0
            else:
                self.logger.debug("The database returns invalid result %s" %(ret))
                return 0

    def isParentIpProfilerEnable(self):
        sql = "select \"enable\" from t_foxhuntersetting;"
        (ret, e) = self.executeQueryStatement(self.conf.getAdminDBAddress(),
                                                       self.conf.getAdminDBPort(),
                                                       self.conf.getAdminDBUsername(),
                                                       self.decryptString(self.conf.getAdminDBPassword()),
                                                       self.conf.getAdminDBName(),
                                                       sql)
        if e:
            self.logger.debug("Fail to query IpProfiler from db, %s" %(excToString(e)))
            return 0
        else:
            if len(ret[0]) == 1:
                isEnable = ret[0][0]
                self.logger.debug("Query IpProfiler from db, %s" % (isEnable))
                if isEnable:
                    self.logger.debug("The IpProfiler in parent is enabled.")
                    return 1
                else:
                    return 0
            else:
                self.logger.debug("The database returns invalid result %s" %(ret))
                return 0

    def isOSRedHat9(self):
        rls_file_path = "/etc/redhat-release"
        if os.path.exists(rls_file_path):   
            match = "Red Hat .+ release 9\."
            with open(rls_file_path) as f:
                line = f.readline()
                if re.match(match, line):
                    self.logger.debug("OS is RedHat9")
                    return True
        return False

    def addRedHat9Dependency(self):
        ret =  os.system("/bin/cp -af imssbase/lib/libresolv.so.2 " + self.conf.getInstallPath() +"/imss/UI/php/lib")
        if ret == 0:
            self.logger.debug("Success to copy imssbase/lib/libresolv.so.2 for RedHat9")
            os.system("chown imss:imss " + self.conf.getInstallPath() + "/imss/UI/php/lib/libresolv.so.2")
            os.system("chmod 750 " + self.conf.getInstallPath() + "/imss/UI/php/lib/libresolv.so.2")
        else:
            self.logger.debug("Fail to copy imssbase/lib/libresolv.so.2 for RedHat9")
        return False
