/* Kernel Hook Module for Trend Micro ServerProtect for Linux  */
/* Copyright (C) 2012 Trend Micro Incorporated.                */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/************************Change History*****************************/
/**
** Modify to support SLES kernel version 2.6.27
** Modify Date: 2009/10/30
** Modify By:   errik_zhang
**/

/**
** Modify to Add support for command bypass
** Proc entry: /proc/splx/comm_exc_list
** Modify Date: 2010/01/15
** Modify By: errik_zhang@trendmicro.com
**/

/**
** Modify to resolve the confliction with auditd
** Function ClearAuditContext() To clear audit flag
** Function SetAuditContext() to setback audit flag
** Modify Date: 2010/02/01
**/

/**
** Modify to support kernel version 3.0
** Modify Date: 2012/06/21
** Modify By:   samir_bai@trendmicro.com.cn
**/

/**
** Modify to fix kernel panic issue when file type is FIFO
** Modify Date: 2013/02/28
** Modify By: fred_chen@trendmicro.com.cn
**/

/**
** Modify to support kernel data "struct filename" change
** Modify Date: 2013/12/04
** Modify By:   fred_chen@trendmicro.com.cn 
**/

/**
** Modify to support kernel version 4.x
** Modify Date: 2017/05/22
** Modify By:   subin_qu@trendmicro.com.cn
**/

#include    <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)
#include    <linux/config.h> /* retrieve the CONFIG_* macros */
#if defined(CONFIG_SMP)
#define __SMP__
#endif
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS /* force it on */
#endif

#ifdef MODVERSIONS
#include    <linux/modversions.h>
#endif
#define __NO_VERSION__ /* don't define kernel_version in module.h */
#endif
#include    <linux/module.h>

#include    <linux/kernel.h>

#include    <linux/sched.h>
#include    <linux/slab.h>
#include    <linux/unistd.h>
#include    <linux/ptrace.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
#include    <linux/smp_lock.h>
#endif
#include    <linux/stat.h>
#include    <linux/string.h>
#include    <linux/dcache.h>
#include    <linux/mount.h>
#include    <linux/dcache.h>
#include    <linux/mount.h>
#include    <linux/file.h>
#include    <linux/spinlock.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include    <linux/fs.h>
#include    <linux/err.h>
#include    <linux/fcntl.h>
#include    <linux/namei.h>
#endif
#include    <asm/uaccess.h>
#include    <linux/sys.h>
#include    <splxmod.h>
#include    <asm/atomic.h>
#include    <linux/sched.h>
#include    <linux/namei.h>

#ifdef  NFSD
#include    <linux/sunrpc/svc.h>
#include    <linux/nfsd/nfsd.h>
#endif

#include "hooks.h"
#include "misc.h"
/***************************************************************************************************************
                                              Function Declare
****************************************************************************************************************/
extern Boolean S_ISNORMAL(mode_t st_mode);
extern int getStat(char *, struct stat *);
// Replace kernel API getname/putname with khm_getname/khm_putname
extern char *khm_getname(const char *);
extern void khm_putname(char *);
// Replace end
extern Boolean deleteList(LIST_ITEM *);
extern Boolean deleteListPID(pid_t);
extern int lookup_flags(unsigned int f);
extern Boolean insertList(LIST_ITEM *);
extern Boolean findList(pid_t, unsigned int, LIST_ITEM **);
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
extern int PATH_LOOKUP(const char *path, unsigned flags, struct nameidata *nd);
extern void SPLX_PATH_RELEASE(struct nameidata *nd);
#endif
extern void WAKE_UP_INTERRUPTIBLE(wait_queue_head_t *);
extern Boolean inExcForest(pid_t *, int);
extern int splx_kill_proc(pid_t pid, int sig, int priv);
extern struct audit_context *ClearAuditContext(void);
extern void SetAuditContext(struct audit_context *current_audit);
extern Boolean initialized(void);
extern Boolean  InExcComms(const char *comm);

#ifdef KHM_PART("merge from modreg.c")

#ifdef X86_64
#ifdef IA32_HOOK

asmlinkage long (*IA32_orig_execve)(char *, char __user *__user *,
                                    char __user *__user *, struct pt_regs);
asmlinkage long (*IA32_orig_syscall)(void);

asmlinkage long  (*IA32_orig_compat_do_execve)(char __user *name, compat_uptr_t __user *,
                                    compat_uptr_t __user *, struct pt_regs *);

#endif

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
asmlinkage long (*orig_execve)(char *, char __user *__user *,
                               char __user *__user *, struct pt_regs);
#else
asmlinkage long (*orig_execve)(char *, char __user *__user *,
                               char __user *__user *, struct pt_regs *);

#endif


#else
asmlinkage int (*orig_execve)(struct pt_regs);
#endif

asmlinkage long (*orig_syscall)(void);
int (*orig_do_execve)(char * ,
                      char __user *__user *,
                      char __user *__user *,
                      struct pt_regs * );


#ifdef  NFSD
int (*orig_nfsd_open)(struct svc_rqst *, struct svc_fh *, int, int,struct file *);
void (*orig_nfsd_close)(struct file *);
#endif

#endif
/***************************************************************************************************************
                                              Global Variable Declare
****************************************************************************************************************/

extern int     g_iDbgLevel;
extern pid_t  *exc_pid_ary;
extern int    exc_pid_no;
extern INIT_ARGS kini;
extern int    vsapi_chldn_no;
extern rwlock_t kini_lock;
extern spinlock_t dbg_lock;
extern Boolean inited;
extern rwlock_t denywrite_list_head_lock;

long busy_timeout_HZ = 0;
long scan_timeout_HZ = 0;
DECLARE_WAIT_QUEUE_HEAD(vsapi_chldn_wq);
atomic_t    ref_cnt = ATOMIC_INIT(0);
atomic_t    candidate = ATOMIC_INIT(FALSE);

/***************************************************************************************************************
                                              Function Pointer Definition
****************************************************************************************************************/
 void* orig_openat = NULL;
 void* orig_open = NULL;
 void* orig_close = NULL;
 void* orig_dup2 = NULL;
 void* orig_dup3 = NULL;
 void* orig_exit = NULL;
 void* orig_exit_group = NULL;
 void* orig_getpgid = NULL;
 void* orig_write = NULL;
 void* orig_pwrite = NULL;
 void* orig_pwrite64 = NULL;
 void* orig_writev = NULL;
 void* orig_pwritev = NULL;
 void* orig_umount = NULL;
void* orig_umount2 = NULL;
void* orig_unlink = NULL;
  void* orig_unlinkat = NULL;
#ifdef IA32_HOOK
void* IA32_orig_openat = NULL ;
void* IA32_orig_open = NULL;
void* IA32_orig_close = NULL;
void* IA32_orig_dup2 = NULL;
void* IA32_orig_dup3 = NULL;
void* IA32_orig_exit = NULL;
void* IA32_orig_exit_group = NULL;
void* IA32_orig_getpgid = NULL;
void* IA32_orig_write = NULL;
void* IA32_orig_pwrite = NULL;
void* IA32_orig_pwrite64 = NULL;
void* IA32_orig_writev = NULL;;
void* IA32_orig_pwritev = NULL;
void* IA32_orig_umount = NULL;
void* IA32_orig_umount2 = NULL;
void* IA32_orig_unlink = NULL;
void* IA32_orig_unlinkat = NULL;

#endif

/***************************************************************************************************************
                                         Function Definition
****************************************************************************************************************/

#if KHM_PART("OPEN")

#ifndef AT_FDCWD
#define AT_FDCWD		-100
#endif

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage long open_hook_comm(Boolean bIA32, OPENTYPE hook_func, void *orig_system_call, const struct pt_regs *regs)
#else
asmlinkage long open_hook_comm(Boolean bIA32, OPENTYPE hook_func, void *orig_system_call, int dirfd, const char *filename, int flags, mode_t  mode)
#endif
{
    char cpsMethod[32] = {0};
    struct inode  *inode = NULL;
    mode_t  st_mode = 0;
    int ret = 0;
    LIST_ITEM   *ip = NULL;
    char    *comm = NULL;
    Boolean found;
    int vsapi_ret;
    int action;
    int clen;
    int vsapi_chld_pid;
    struct stat statbuf;
    char *tmp = NULL;
/*
#if RHEL_RELEASE_CODE >= 1541
    struct filename *filename_s;
#endif
*/
    int fd = 0;
    int error;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
#else
    struct nameidata nd;
#endif
    int namei_flags;
    Boolean scan_open = FALSE;
    pid_t   cur_pid = 0;
    long timeout;
#ifdef X86_64
    void *temp_ip;
#endif
    struct audit_context *current_audit = NULL;
    
	#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
    int dirfd = 0;
    char *filename = NULL;
	int flags = 0;
	mode_t mode = 0;

    if (!bIA32)
    {     
        if (OPENAT_HOOK == hook_func)
        {
        	dirfd = regs->di;
            filename = (char *)(regs->si);
        	flags = regs->dx;
        	mode = regs->r10;
        }
        else
        {
            filename = (char *)(regs->di);
        	flags = regs->si;
        	mode = regs->dx;
        }
    }
    else
    {
        if (OPENAT_HOOK == hook_func)
        {
            dirfd = regs->bx;
            filename = (char *)(regs->cx);
            flags = regs->dx;
            mode = regs->si;
        }
        else
        {
            filename = (char *)(regs->bx);
            flags = regs->cx;
            mode = regs->dx;
        }
    }
	#endif
	
    CP_DBG_LVL;
    MOD_INC_REF_COUNT;

    sprintf(cpsMethod,"%s%s",(!bIA32 ? "" : "ia32_"),((OPEN_HOOK == hook_func) ? "openHook" : "openatHook"));
    
    cur_pid = current->pid;
    //First check whether the command is in the exclusion list
    if (InExcComms(current->comm))
    {
        MOD_DEC_REF_COUNT;
       /// DPRINTK(3, "%d: %s: not to scan because command [%s] in the command exclusion list\n", cur_pid, cpsMethod, current->comm);
        
    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
    return ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (OPEN_HOOK == hook_func)
        return ((pfn_orig_open)orig_system_call)(filename, flags, mode);
    else
        return ((pfn_orig_openat)orig_system_call)(dirfd,filename, flags, mode);         
    #endif

    }

    //Add for avoid conflict with auditd
    current_audit = ClearAuditContext();
    //Add end
    tmp = khm_getname(filename);
    fd = PTR_ERR(tmp);
    if (IS_ERR(tmp))
    {
        MOD_DEC_REF_COUNT;
        SetAuditContext(current_audit);
        return fd;        
    }

    DPRINTK(2, "%d: %s: command = [%s], filename =[%s],flags =[%#x].\n", cur_pid, cpsMethod, current->comm, tmp,flags & O_ACCMODE);

    if (!initialized())
    {
        // DPRINTK(2, "%d: %s: vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod);
        DPRINTK(3, "%d: %s: command = [%s], filename =[%s],flags =[%#x]. vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod, current->comm, tmp,flags & O_ACCMODE);
        goto jump;
    }

    read_lock(&kini_lock);
    // DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (inExcForest(exc_pid_ary, exc_pid_no))
    {
        DPRINTK(3, "%d: %s: inExcForest() returned true, goto jump for file[%s]\n", cur_pid, cpsMethod,tmp);
        read_unlock(&kini_lock);
        goto jump;
    }
    read_unlock(&kini_lock);

    error = getStat(tmp, &statbuf);
    if (error)
    {
        memset(&statbuf, 0, sizeof(statbuf));
    }
    
    st_mode = statbuf.st_mode;
    DPRINTK(3, "%d: %s: vsapi_chldn_no=%d st_mode=%x,file=%s.\n", cur_pid, cpsMethod, vsapi_chldn_no, st_mode,tmp);

    if (!S_ISNORMAL(st_mode))
    {
        DPRINTK(3, "%d: %s: not regular file or link, goto jump for file[%s].\n", cur_pid, cpsMethod,tmp);
        goto jump;
    }

    namei_flags = flags;
    if ((namei_flags + 1) & O_ACCMODE)
        namei_flags++;
    if (namei_flags & O_TRUNC)
        namei_flags |= 2;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    error = kern_path(tmp, lookup_flags(namei_flags), &pPath);
#else
    error = PATH_LOOKUP(tmp, lookup_flags(namei_flags), &nd);
#endif

    if ((flags & O_ACCMODE) == O_RDWR || (flags & O_ACCMODE) == O_WRONLY || (flags & O_ACCMODE) == O_RDONLY)
    {
        DPRINTK(3, "%d: %s: file[%s] denywrite: Has write attribute\n", cur_pid, cpsMethod,tmp);
        read_lock(&denywrite_list_head_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        if (needToDenyWrite(pPath.dentry, pPath.mnt, flags))
        {
#else
        if (needToDenyWrite(nd.DENTRY, nd.MNT, flags))
        {
#endif
            read_unlock(&denywrite_list_head_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
            if (!error) path_put(&pPath);
#else
            if (!error) SPLX_PATH_RELEASE(&nd);
#endif

            DPRINTK(3, "%d: %s: if (!found), file=[%s] return -EACCES\n", cur_pid, cpsMethod,tmp);

            khm_putname(tmp);
            tmp = NULL;

            ret = -EACCES;
            MOD_DEC_REF_COUNT;
            SetAuditContext(current_audit);
            return (ret);
        }
        read_unlock(&denywrite_list_head_lock);
    }
    else
        DPRINTK(2, "%d: %s: denywrite: file[%s] has no write attribute\n", cur_pid, cpsMethod,tmp);

    if (error)
    {
        DPRINTK(1, "%d: %s: path lookup file[%s] failed, goto jump\n", cur_pid, cpsMethod,tmp);
        goto jump;
    }

    read_lock(&kini_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    inode = pPath.dentry->d_inode;
    scan_open = needToScanThisOpen(pPath.dentry, pPath.mnt, flags, inode);
#else
    inode = nd.DENTRY->d_inode;
    scan_open = needToScanThisOpen(nd.DENTRY, nd.MNT, flags, inode);
#endif
    read_unlock(&kini_lock);

    DPRINTK(3, "%d: %s: scan_open: %d\n", cur_pid, cpsMethod, scan_open);

    if (!scan_open)
    {
        DPRINTK(2, "%d: %s: no need to scan this file[%s] open, goto jump\n", cur_pid, cpsMethod,tmp);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }

    khm_putname(tmp);
    tmp = NULL;

    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_ATOMIC);
    if (ip == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: ip is NULL.\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    memset(ip, 0, sizeof(LIST_ITEM));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.scan_args.full_pn = (char *) pPath.dentry->d_name.name;
#else
    ip->info.scan_args.full_pn = (char *) nd.DENTRY->d_name.name;
#endif
    clen = strlen(current->comm);
    comm = (char *)kmalloc(clen + 1, GFP_ATOMIC);
    if (comm == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: comm is NULL.\n", cur_pid, cpsMethod);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    memset(comm, 0, clen + 1);
    ip->info.scan_args.comm = comm;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
#ifdef CONFIG_UIDGID_STRICT_TYPE_CHECKS
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,14,0)
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
    ip->info.scan_args.comm_uid = current->SPLX_UID;
#endif
#endif
    ip->info.scan_args.flags = flags;

    #ifdef X86_64
    temp_ip =  ip;
    ip->info.scan_args.u_lip = (((unsigned long)temp_ip) & 0xffffffff00000000) >> 32;
    temp_ip =  ip;
    ip->info.scan_args.d_lip = ((unsigned long)temp_ip) & 0x0000ffffffffffff;
    #else
    ip->info.scan_args.lip = ip;
    #endif
    
    ip->info.scan_args.inode = inode->i_ino;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = TRUE;
    atomic_set(&(ip->info.cond), FALSE);
	atomic_set(&(ip->info.ref_count), 1);
	init_waitqueue_head(&(ip->info.wait_queue));

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
    ip->info.path = &pPath;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.dentry = pPath.dentry;
    ip->info.mnt = pPath.mnt;
#else
    ip->info.dentry = nd.DENTRY;
    ip->info.mnt = nd.MNT;
#endif

    INIT_LIST_HEAD(&ip->item_list);
    read_lock(&kini_lock);
    timeout = scan_timeout_HZ;
    read_unlock(&kini_lock);
    if (!initialized())
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
        ip = NULL;
        goto jump;
    }
    DPRINTK(3, "%d: %s: start to scan this open file[%s].\n", cur_pid, cpsMethod,ip->info.scan_args.full_pn);
    found = insertList(ip);
    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
        ip = NULL;
        MOD_DEC_REF_COUNT;
        DPRINTK(2, "%d: %s: if (!found), return -EACCES\n", cur_pid, cpsMethod);
        SetAuditContext(current_audit);
        return (-EACCES);
    }
    atomic_set(&candidate, TRUE);
    wake_up_interruptible(&vsapi_chldn_wq);
    /* scan this open() only when opened in RDONLY/RDWR mode */
    //  if (timeout > 0) {
    //  /* to be implemented */
    //    } else {
    DPRINTK(2, "%d: %s: sleep on open_wq,command=%s, file=%s.\n", cur_pid, cpsMethod,ip->info.scan_args.comm,ip->info.scan_args.full_pn);
    wait_event(ip->info.wait_queue, atomic_read(&(ip->info.cond)));
    //  }
    DPRINTK(1, "%d: %s: other pid %d, command=%s, filename=%s.\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.comm , ip->info.scan_args.full_pn);

    if (ip->info.vsapi_busy == FALSE)
    {
        vsapi_ret = ip->info.scan_args.vsapi_ret;
        action = ip->info.scan_args.action;
        if (vsapi_ret == VIRUS_FOUND)
        {
            /* should be revised here */
            removeCache(inode);
            switch (action)
            {
            case CLEAN:
                DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                break;
            case DELETE:
                DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                break;
            case MOVE:
                DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                break;
            case RENAME:
                DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                break;
            case BYPASS:
                DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                break;
            case DENYACCESS:
                DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(2, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
                MOD_DEC_REF_COUNT;
                DPRINTK(2, "%d: %s: return -EACCES\n", cur_pid, cpsMethod);
                SetAuditContext(current_audit);
                return (ret);
            default:
                DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
                MOD_DEC_REF_COUNT;
                DPRINTK(2, "%d: %s: return -EACCES\n", cur_pid, cpsMethod);
                SetAuditContext(current_audit);
                return (ret);
            }
        }
        else if (vsapi_ret == NO_VIRUS)
        {
            /*
            * only perfectly clean files can be
            * added to the cache
            */
            if ((flags & O_ACCMODE) == O_RDONLY)
            {
                addCache(inode);
            }
        }
        else
        {
            DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
            found = deleteList(ip);
            if (!found)
                DPRINTK(2, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
            ip = NULL;
            /* permission denied */
            ret = -EACCES;
            MOD_DEC_REF_COUNT;
            DPRINTK(2, "%d: %s: if (!found), return -EACCES\n", cur_pid, cpsMethod);
            SetAuditContext(current_audit);
            return (ret);
        }
    }
    else     /* VSAPI time out */
    {
        vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
        if (vsapi_chld_pid > 0)
        {
            /* from kernel/signal.c { */
            struct  siginfo info;
            info.si_signo = SIGTERM;
            info.si_errno = 0;
            info.si_code = SI_USER;
            info.si_pid = ip->info.scan_args.vsapi_chld_pid;
            info.si_uid = 0;
            splx_kill_proc(info.si_pid, info.si_signo, (long)&info);
            /* from kernel/signal.c } */
        }
    }


    if (ip != NULL)
    {
        found = deleteList(ip);
        if (!found)
            DPRINTK(2, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
    }
jump:
    /* SPNT 5.x scans first and then open here if RDONLY */
    if (tmp != NULL)
    {
        khm_putname(tmp);
        tmp = NULL;
    }
    SetAuditContext(current_audit);

    // If the file is a FIFO, it may be blocked in orig_open(),
    // if KHM is unloaded at this time point.
    // A kernel panic may happen when the orig_open() return
    // in another word, this change may lead to unload fail of splxmod
    // when the orig_open() blocked in openning a FIFO file.


    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
        ret = ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (OPEN_HOOK == hook_func)
        ret = ((pfn_orig_open)orig_system_call)(filename, flags, mode);
    else
        ret = ((pfn_orig_openat)orig_system_call)(dirfd,filename, flags, mode);         
    #endif

    DPRINTK(3, "%d: %s: get outta %s.\n", cur_pid, cpsMethod,cpsMethod);
    MOD_DEC_REF_COUNT;
    return (ret);
}

#endif


#if KHM_PART("CLOSE")

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage long close_hook_comm(Boolean bIA32, CLOSETYPE hook_func, void *orig_system_call,const struct pt_regs *regs)
#else
asmlinkage long close_hook_comm(Boolean bIA32, CLOSETYPE hook_func, void *orig_system_call, int oldfd, int newfd, int flag)
#endif
{

    char cpsMethod[32] = {0};
    unsigned int    flags = 0;
    pid_t   cur_pid = 0;
    struct inode *inode = NULL;
    mode_t      st_mode = 0;
    LIST_ITEM      *ip = NULL;
    char        *comm = NULL;
    int clen;
    long    ret;
    int vsapi_ret;
    int action;
    Boolean found;
    int vsapi_chld_pid;
    struct file *f = NULL;
    int closing_fd = 0;
    
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path    pPath;
#endif
    struct dentry *dentry = NULL;
    struct vfsmount *vfsmnt = NULL;
    long timeout;
#ifdef X86_64
    void *temp_ip;
#endif
    Boolean scan_close;

	#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
	int oldfd = 0;
    int newfd = 0;
    int flag = 0;
    
    if (bIA32)
    {
        if (CLOSE_HOOK == hook_func)
        {
            oldfd = regs->bx;
        }
        else if (DUP2_HOOK == hook_func)
        {
            oldfd    = regs->bx;
            newfd = regs->cx;
        }   
        else
        {
            oldfd    = regs->bx;
            newfd = regs->cx;
            flag  = regs->dx;
        } 
    }
    else
    {
        if (CLOSE_HOOK == hook_func)
        {
            oldfd = regs->di;
        }
        else if (DUP2_HOOK == hook_func)
        {
            oldfd    = regs->di;
            newfd = regs->si;
        }   
        else
        {
            oldfd    = regs->di;
            newfd = regs->si;
            flag  = regs->dx;
        } 
    }
	#endif
	
    CP_DBG_LVL;
    MOD_INC_REF_COUNT;

    sprintf(cpsMethod,"%s%s",(!bIA32 ? "" : "ia32_"),((CLOSE_HOOK == hook_func) ? "closeHook" : ((DUP2_HOOK == hook_func) ? "dup2Hook" : "dup3Hook")));
    closing_fd = (CLOSE_HOOK == hook_func) ? oldfd : newfd;
        
    cur_pid = current->pid;

    //First check whether the command is in the exclusion list
    if (InExcComms(current->comm))
    {
        MOD_DEC_REF_COUNT;
       /// DPRINTK(3, "%d: %s: not to scan command [%s] because it in the command exclusion list\n", cur_pid, cpsMethod, current->comm);

        #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
        return ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
        #else
        if (CLOSE_HOOK == hook_func)
            return ((pfn_orig_close)orig_system_call)(oldfd);
        else if (DUP2_HOOK == hook_func)
            return ((pfn_orig_dup2)orig_system_call)(oldfd,newfd);     
        else
            return ((pfn_orig_dup3)orig_system_call)(oldfd,newfd, flag); 
        #endif
		
    }

    if ((f = fget(closing_fd)))
    {
        // do dget and mntget here, or else they might be destroyed after invoking the original close.

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        pPath = f->f_path;
        path_get(&pPath);
        dentry = pPath.dentry;
        vfsmnt = pPath.mnt;
#else
        dentry = dget(f->f_dentry);
        vfsmnt = mntget(f->f_vfsmnt);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0)
        inode = f->f_dentry->d_inode;
        st_mode = f->f_dentry->d_inode->i_mode;
#else
        inode = f->f_path.dentry->d_inode;
        st_mode = f->f_path.dentry->d_inode->i_mode;
#endif
        flags = f->f_flags;
        fput(f);
    }

    
    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
        ret = ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (CLOSE_HOOK == hook_func)
        ret = ((pfn_orig_close)orig_system_call)(oldfd);
    else if (DUP2_HOOK == hook_func)
        ret = ((pfn_orig_dup2)orig_system_call)(oldfd,newfd);     
    else
        ret = ((pfn_orig_dup3)orig_system_call)(oldfd,newfd, flag); 
    #endif

    
    //TT216607
    if (!f)
    {
        DPRINTK(2, "%d: %s: get outta %s, failed to get struct file, ret=[%ld].\n", cur_pid, cpsMethod,cpsMethod,ret);
        MOD_DEC_REF_COUNT;
        return ret;
    }
    else if (ret < 0 || !f->f_op->open)     // it's possible we got file struct (a valid fd), however, other errors occur in original close.
    {
        DPRINTK(2, "%d: %s: get outta %s, f->f_op->open=[%p], ret=[%ld].\n", cur_pid, cpsMethod,cpsMethod,f->f_op->open,ret);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        MOD_DEC_REF_COUNT;
        return ret;
    }
    else
    {
        DPRINTK(3, "%d: %s: orig_close(inode=%p) ret %ld\n", cur_pid, cpsMethod, inode,ret);
    }
    
    if (!initialized())
    {
        DPRINTK(2, "%d: %s: vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }

    read_lock(&kini_lock);
    DPRINTK(3, "%d: %s: command=[%s], fd=[%u] vsapi_chldn_no[%d].\n", cur_pid, cpsMethod, current->comm, closing_fd, vsapi_chldn_no);
    // DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (inExcForest(exc_pid_ary, exc_pid_no))
    {

        DPRINTK(2, "%d: %s: inExcForest() returned true, goto jump\n", cur_pid, cpsMethod);
        read_unlock(&kini_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }
    read_unlock(&kini_lock);

    DPRINTK(3, "%d: %s: st_mode %x\n", cur_pid, cpsMethod, st_mode);
    //Need to revise: fix a bug that panic happens when scan file with st_mode 180
    //Date: 2009-11-23
    if (!S_ISNORMAL(st_mode))
    {
        DPRINTK(3, "%d: %s: not regular file or link, goto jump\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }

    read_lock(&kini_lock);
    scan_close = needToScanThisClose(dentry, vfsmnt, flags, inode);
    read_unlock(&kini_lock);

    if (((flags & O_ACCMODE) == O_WRONLY) || ((flags & O_ACCMODE) == O_RDWR))
    {
        removeCache(inode);
        DPRINTK(2, "%d: %s: remove file [%s], inode [%p] from Cache.\n", cur_pid, cpsMethod,  (char *)dentry->d_name.name,inode);
    }

   // DPRINTK(3, "%d: %s: scan_close=%d, file=%s\n", cur_pid, cpsMethod, scan_close,(char *)dentry->d_name.name);


    if (!scan_close)
    {
        DPRINTK(3, "%d: %s: no need to scan this file[%s] close, goto jump\n", 
            cur_pid, cpsMethod,(char *)dentry->d_name.name);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }

    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_ATOMIC);
    if (ip == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: ip is NULL\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }
    memset(ip, 0, sizeof(LIST_ITEM));
    ip->info.scan_args.full_pn = (char *)dentry->d_name.name;
    clen = strlen(current->comm);
    comm = (char *)kmalloc(clen + 1, GFP_ATOMIC);
    if (comm == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: comm is NULL\n", cur_pid, cpsMethod);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;   
    }
        
    memset(comm, 0, clen + 1);
    ip->info.scan_args.comm = comm;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
    ip->info.scan_args.fd = closing_fd;
#ifdef CONFIG_UIDGID_STRICT_TYPE_CHECKS
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,14,0)
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
    ip->info.scan_args.comm_uid = current->SPLX_UID;
#endif
#endif
    ip->info.scan_args.flags = flags;

    #ifdef X86_64
    temp_ip =  ip;
    ip->info.scan_args.u_lip = (((unsigned long)temp_ip) & 0xffffffff00000000) >> 32;
    temp_ip =  ip;
    ip->info.scan_args.d_lip = ((unsigned long)temp_ip) & 0x0000ffffffffffff;
    #else
    ip->info.scan_args.lip = ip;
    #endif
    
    ip->info.scan_args.inode = inode->i_ino;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = TRUE;
	atomic_set(&(ip->info.ref_count), 1);
	init_waitqueue_head(&(ip->info.wait_queue));
    atomic_set(&(ip->info.cond), FALSE);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
    ip->info.path = &pPath;
#else
    ip->info.dentry = dentry;
    ip->info.mnt = vfsmnt;
#endif

    INIT_LIST_HEAD(&ip->item_list);
    read_lock(&kini_lock);
    timeout = scan_timeout_HZ;
    read_unlock(&kini_lock);

    if (!initialized())
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        goto jump;
    }
    DPRINTK(3, "%d: %s: start to scan this close, command=%s,filename=%s.\n", cur_pid, cpsMethod, ip->info.scan_args.comm,ip->info.scan_args.full_pn);
	
    found = insertList(ip);

    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        dput(dentry);
        mntput(vfsmnt);
#endif
        MOD_DEC_REF_COUNT;
        DPRINTK(1, "%d: %s: if (!found), return -EACCES\n", cur_pid, cpsMethod);
        return (-EACCES);
    }

    atomic_set(&candidate, TRUE);
    wake_up_interruptible(&vsapi_chldn_wq);
    DPRINTK(2, "%d: %s: sleep on close_wq, command=%s,filename=%s.\n", cur_pid, cpsMethod, ip->info.scan_args.comm,ip->info.scan_args.full_pn);
    wait_event(ip->info.wait_queue, atomic_read(&(ip->info.cond)));
    DPRINTK(1, "%d: %s: other pid %d, command=%s, filename=%s.\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.comm, ip->info.scan_args.full_pn);

    if (ip->info.vsapi_busy == FALSE)
    {
        vsapi_ret = ip->info.scan_args.vsapi_ret;
        action = ip->info.scan_args.action;
        if (vsapi_ret == VIRUS_FOUND)
        {
            /* should be revised here */
            //removeCache(inode);
            switch (action)
            {
            case CLEAN:
                DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                break;
            case DELETE:
                DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                break;
            case MOVE:
                DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                break;
            case RENAME:
                DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                break;
            case DENYACCESS:
                DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                break;
            case BYPASS:
                DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                break;
            default:
                DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                break;
            }
        }
        else if (vsapi_ret == NO_VIRUS)
        {
            addCache(inode);
            DPRINTK(3, "%d: %s: add file [%s] inode [%p] into Cache.\n", cur_pid, cpsMethod, (char *)dentry->d_name.name,inode);
           
        }
        else
        {
            DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
        }
    }
    else
    {

        vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
        if (vsapi_chld_pid > 0)
        {
            /* from kernel/signal.c { */
            struct  siginfo info;

            info.si_signo = SIGTERM;
            info.si_errno = 0;
            info.si_code = SI_USER;
            info.si_pid = ip->info.scan_args.vsapi_chld_pid;
            info.si_uid = 0;
            splx_kill_proc(info.si_pid, info.si_signo, (long)&info);

            /* from kernel/signal.c } */

        }
    }

    if (ip != NULL)
    {
        found = deleteList(ip);
        if (!found)
            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
    }

jump:
    DPRINTK(3, "%d: %s: get outta %s\n", cur_pid, cpsMethod,cpsMethod);
    MOD_DEC_REF_COUNT;
    return (ret);
}

#endif

#if KHM_PART("write")

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage long write_hook_comm(Boolean bIA32, WRITETYPE hook_func, void *orig_system_call,const struct pt_regs *regs)
#else
asmlinkage long write_hook_comm(Boolean bIA32, WRITETYPE hook_func, void *orig_system_call, int fd, void *buf, int count, off_t offset)
#endif
{
    char cpsMethod[32] = {0};
    pid_t   cur_pid = 0;
    struct inode *inode = NULL;
    mode_t      st_mode = 0;
    long    ret = 0;
    struct file *f = NULL;
    unsigned int    flags = 0;
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path    pPath;
    #endif
    struct dentry *dentry = NULL;
    struct vfsmount *vfsmnt = NULL;
    char *prefix = !bIA32 ? "" : "ia32_";
    char filename[256] = {0};
    int is_removeCache = FALSE;
	#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
	int fd = !bIA32 ? regs->di : regs->bx;
	#endif
	
    CP_DBG_LVL;
    MOD_INC_REF_COUNT;

  
    switch (hook_func)
    {
    case WRITE_HOOK:
        sprintf(cpsMethod, "%s%s", prefix, "writeHook"); 
        break;
    case WRITEV_HOOK:
        sprintf(cpsMethod, "%s%s", prefix, "writevHook"); 
        break;
    case PWRITEV_HOOK:
        sprintf(cpsMethod, "%s%s", prefix, "pwritevHook"); 
        break;
    case PWRITE64_HOOK:
        sprintf(cpsMethod, "%s%s", prefix, "pwrite64Hook"); 
        break;
    default:
        sprintf(cpsMethod,"%s%s", prefix, "writeHook"); 
        break;
    }
    

    cur_pid = current->pid;

    if (InExcComms(current->comm))
    {    
        goto EXIT_LABEL;        
    }

    if ((f = fget(fd)))
    {
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        pPath = f->f_path;
        path_get(&pPath);
        dentry = pPath.dentry;
        vfsmnt = pPath.mnt;
        #else
        dentry = dget(f->f_dentry);
        vfsmnt = mntget(f->f_vfsmnt);
        #endif

        strncpy(filename,(char *)dentry->d_name.name,sizeof(filename));
        filename[sizeof(filename) - 1] = '\0';
    
        #if LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0)
        inode = f->f_dentry->d_inode;
        st_mode = f->f_dentry->d_inode->i_mode;
        #else
        inode = f->f_path.dentry->d_inode;
        st_mode = f->f_path.dentry->d_inode->i_mode;
        #endif
        flags = f->f_flags;
        fput(f);

        #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
        #else
        dput(dentry);
        mntput(vfsmnt);
        #endif
    }
        
    if (!S_ISNORMAL(st_mode))
    {
        goto EXIT_LABEL;
    }
      
    removeCache(inode);
    is_removeCache = TRUE;
    
EXIT_LABEL:
    
#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
    ret = ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
#else
    switch (hook_func)
    {
        case WRITE_HOOK:
        {
            ret = ((pfn_orig_write)orig_system_call)(fd, (char*)buf, count);
            break;
        }
        case WRITEV_HOOK:
        {
            ret = ((pfn_orig_writev)orig_system_call)(fd, (struct iovec *)buf, count);
            break;
        }
        case PWRITE64_HOOK:
        {
            #ifdef X86_64
            if (bIA32)
                ret = ((pfn_orig_pwriteIA32)orig_system_call)(fd, (char*)buf, count, offset & 0xFFFFFFFF, (offset >> 32) & 0xFFFFFFFF);
            else
                ret = ((pfn_orig_pwrite64)orig_system_call)(fd, (char*)buf, count, offset);
            #else
            ret = ((pfn_orig_pwrite)orig_system_call)(fd, (char*)buf, count, offset);
            #endif
            
            break;
        }
        case PWRITEV_HOOK:
        {
            #ifdef X86_64
            ret = ((pfn_orig_pwritev64)orig_system_call)(fd, (struct iovec *)buf, count,  offset & 0xFFFFFFFF, (offset >> 32) & 0xFFFFFFFF); 
            #else
            ret = ((pfn_orig_pwritev)orig_system_call)(fd, (struct iovec *)buf, count, offset); 
            #endif
            
            break;
        }
        default:
        {   
            DPRINTK(1, "%d: %s: wrong hook_func=[%u].\n", cur_pid, cpsMethod, hook_func);
            break;
        }
            
    }
#endif  
    if (is_removeCache)
        DPRINTK(2, "%d: %s: command=[%s], fd=[%u], filename=[%s],ret=[%ld].\n", cur_pid, cpsMethod, current->comm, fd, filename,ret);        

    MOD_DEC_REF_COUNT;
    return ret;
}

#endif


#if KHM_PART("EXIT")

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage long exit_hook_comm(EXITTYPE hook_func, void *orig_system_call,const struct pt_regs *regs)
#else
asmlinkage long exit_hook_comm(EXITTYPE hook_func, void *orig_system_call,int errcode)
#endif
{
    const char *cpsMethod = (EXIT_HOOK == hook_func) ? "exitHook" : "exitgroupHook";
    pid_t   cur_pid = 0;
    Boolean found;
    CP_DBG_LVL;

    MOD_INC_REF_COUNT;
    cur_pid = current->pid;
    DPRINTK(2, "%d: %s: get into %s\n", cur_pid, cpsMethod,cpsMethod);
    found = deleteListPID(cur_pid);
    if (!found)
        DPRINTK(3, "%d: %s: deleteListPID not found\n", cur_pid, cpsMethod);
    MOD_DEC_REF_COUNT;


    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
    return ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (EXIT_HOOK == hook_func)
        return ((pfn_orig_exit)orig_system_call)(errcode);
    else
        return ((pfn_orig_exit_group)orig_system_call)(errcode);
    #endif
}

#endif

#if KHM_PART("umount")

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
int splxmod_user_path_at(int dfd, const char __user *name, unsigned flags,
		 struct nameidata *nd)
{
	char *tmp = khm_getname(name);
	int err = PTR_ERR(tmp);
	if (!IS_ERR(tmp)) {

		BUG_ON(flags & LOOKUP_PARENT);

		err = path_lookup(tmp, flags, nd);
		khm_putname(tmp);
	}

   // DPRINTK(1,"splxmod_user_path_at return err=%d.\n", err);
	return err;
}
#else 
int splxmod_user_path_at(int dfd, const char __user *name, unsigned flags,
		 struct path *path)
{
	char *tmp = khm_getname(name);
	int err = PTR_ERR(tmp);
    
	if (!IS_ERR(tmp)) {

		BUG_ON(flags & LOOKUP_PARENT);

        err = kern_path(tmp, flags, path);
        #if 0
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        err = kern_path(tmp, flags, path);
        #else
    	struct nameidata nd;
		err = path_lookup(tmp, flags, &nd);
    	if (!err)
			*path = nd.path; 
        #endif
        #endif
		khm_putname(tmp);
	}

    //DPRINTK(1,"splxmod_user_path_at return err=%d.\n", err);
    
	return err;
}
        

#endif
         

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage int umount_hook_comm(Boolean bIA32, UMOUNTTYPE hook_func, void *orig_system_call,const struct pt_regs *regs)
#else
asmlinkage int umount_hook_comm(Boolean bIA32, UMOUNTTYPE hook_func, void *orig_system_call,const char *name, int flags)
#endif
{
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
    #else
    struct nameidata nd;
    #endif
    int retval = 0;
    int lookup_flags = 0;

    CP_DBG_LVL;
    MOD_INC_REF_COUNT;

    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
	char *name;
    int flags;
    if (bIA32)
    {
        name  = (char*)(regs->bx);
        flags = regs->cx;
    }
    else
    {
	    name = (char*)(regs->di);
        flags = regs->si;
    }
    #endif
    

    #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,34)
	if (flags & ~(MNT_FORCE | MNT_DETACH | MNT_EXPIRE | UMOUNT_NOFOLLOW))
	{
        MOD_DEC_REF_COUNT;
		return -EINVAL;
	}
    
	if (!(flags & UMOUNT_NOFOLLOW))
		lookup_flags |= LOOKUP_FOLLOW;
    #endif

    #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
	retval = splxmod_user_path_at(AT_FDCWD, name, lookup_flags, &nd);
    #else
    retval = splxmod_user_path_at(AT_FDCWD, name, lookup_flags, &pPath);
    #endif

	if (retval)
        goto out;

    #if 0
	mnt = real_mount(path.mnt);
	retval = -EINVAL;
	if (path.dentry != path.mnt->mnt_root)
		goto dput_and_out;
	if (!check_mnt(mnt))
		goto dput_and_out;
	if (mnt->mnt.mnt_flags & MNT_LOCKED)
		goto dput_and_out;
	retval = -EPERM;
	if (flags & MNT_FORCE && !capable(CAP_SYS_ADMIN))
		goto dput_and_out;
    #endif
    
    #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
    removeCacheBySuperblock(nd.DENTRY->d_sb);
    #else 
    removeCacheBySuperblock(pPath.dentry->d_sb);
    #endif

    
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    path_put(&pPath);
    #else
    SPLX_PATH_RELEASE(&nd);
    #endif

    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
    retval = ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (UMOUNT2_HOOK == hook_func)
        retval = ((pfn_orig_umount2)orig_system_call)(name,flags);
    else
        retval = ((pfn_orig_umount)orig_system_call)(name);
    #endif  

out:  

    DPRINTK(1,"umount_hook_comm return retval=%d.\n", retval);
    
    MOD_DEC_REF_COUNT;
    return retval;
}

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
asmlinkage int unlink_hook_comm(Boolean bIA32, UNLINKTYPE hook_func, void *orig_system_call,const struct pt_regs *regs)
#else
asmlinkage int unlink_hook_comm(Boolean bIA32, UNLINKTYPE hook_func, void *orig_system_call, int dirfd, const char *pathname, int flags)
#endif
{
    const char *cpsMethod = (UNLINK_HOOK == hook_func) ? "uninkHook" : "unlinkatHook";
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
    #else
    struct nameidata nd;
    #endif
    int retval = 0;
    int lookup_flags = LOOKUP_FOLLOW;

    CP_DBG_LVL;
    MOD_INC_REF_COUNT;

    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
    int dirfd;
	char *pathname;
    int flags;
    if (bIA32)
    {
        if (UNLINKAT_HOOK == hook_func)
        {
            dirfd = regs->bx;
            pathname  = (char*)(regs->cx);
            flags = regs->dx;
        }
        else
        {
             pathname  = (char*)(regs->bx);
        }
    }
    else
    {
        if (UNLINKAT_HOOK == hook_func)
        {
            dirfd = regs->di;
    	    pathname = (char*)(regs->si);
            flags = regs->dx;
        }
        else
        {
            pathname = (char*)(regs->di);
        }
    }
    #endif
    


    #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
	retval = splxmod_user_path_at(AT_FDCWD, pathname, lookup_flags, &nd);
    #else
    retval = splxmod_user_path_at(AT_FDCWD, pathname, lookup_flags, &pPath);
    #endif

	if (retval)
	{
        DPRINTK(1,"%s %s return retval=%d.\n", cpsMethod,pathname,retval);
        goto out;
	}

    
    #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
    removeCache(nd.DENTRY->d_inode);
    #else 
    removeCache(pPath.dentry->d_inode);
    #endif

    
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    path_put(&pPath);
    #else
    SPLX_PATH_RELEASE(&nd);
    #endif

    #ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER  
    retval = ((pfn_orig_syscall_wrapper)orig_system_call)(regs);
    #else
    if (UNLINK_HOOK == hook_func)
        retval = ((pfn_orig_unlink)orig_system_call)(pathname);
    else
        retval = ((pfn_orig_unlinkat)orig_system_call)(dirfd,pathname,flags);
    #endif  

out:  

    DPRINTK(1,"%s %s return retval=%d.\n", cpsMethod,pathname,retval);
    
    MOD_DEC_REF_COUNT;
    return retval;
}


#endif

#if KHM_PART("EXEC")
#ifdef X86_64
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
asmlinkage long execveHook_X86_64(char __user *name, char __user *__user *argv,
                                  char __user *__user *envp, struct pt_regs regs)
#else
asmlinkage long execveHook_X86_64(char __user *name, char __user *__user *argv,
                                  char __user *__user *envp, struct pt_regs *regs)
#endif

{
    const char *cpsMethod = "execveHook";
    struct inode  *inode = NULL;
    mode_t  st_mode;
    int ret;
    pid_t   cur_pid = 0;
    Boolean found;
    int error;
    LIST_ITEM   *ip = NULL;
    char    *comm = NULL;
    int vsapi_ret;
    int action;
    int clen;
    int vsapi_chld_pid;
    struct stat statbuf;
    char *filename;
/*
#if RHEL_RELEASE_CODE >= 1541
    struct filename *filename_s = NULL;
#endif
*/
    long timeout;
    Boolean scan_execve = FALSE;
    void *temp_ip;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
#else
    struct nameidata nd;
#endif
    struct audit_context *current_audit = NULL;
    //  mm_segment_t oldfs = get_fs();
    CP_DBG_LVL;

    MOD_INC_REF_COUNT;
    //Add for avoid conflict with auditd
    current_audit = ClearAuditContext();
    //Add end

    cur_pid = current->pid;
    DPRINTK(3, "%d: %s: get into execveHook\n", cur_pid, cpsMethod);
    found = deleteListPID(cur_pid);
    if (!found)
        DPRINTK(3, "%d: %s: deleteListPID not found\n", cur_pid, cpsMethod);
/*
#if RHEL_RELEASE_CODE >= 1541
    filename_s = getname(name);
    error = PTR_ERR(filename_s);
    if (IS_ERR(filename_s))
    {
        goto out;
    }

    filename = (char *)filename_s->name;
#else
    filename = getname(name);
    error = PTR_ERR(filename);
    if (IS_ERR(filename))
    {
        goto out;
    }
#endif
*/
    filename = khm_getname(name);
    error = PTR_ERR(filename);
    if(IS_ERR(filename))
    {
        goto jump;
    }

    DPRINTK(3, "%d: %s: get into execveHook filename %s\n", cur_pid, cpsMethod, filename);

    //First check whether the command is in the exclusion list
    if (InExcComms(current->comm))
    {
        DPRINTK(3, "%d: %s: not to scan because command [%s] in the command exclusion list\n", cur_pid, cpsMethod, current->comm);
        goto jump;
    }

    if (!initialized())
    {
        DPRINTK(2, "%d: %s: vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod);
        goto jump;
    }

    read_lock(&kini_lock);
    DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (inExcForest(exc_pid_ary, exc_pid_no))
    {
        read_unlock(&kini_lock);
        goto jump;
    }
    read_unlock(&kini_lock);

    error = getStat(filename, &statbuf);

    if (error)
    {
        DPRINTK(2, "%d: %s: get stat for file %s error %d\n", cur_pid, cpsMethod, filename, error);
        goto jump;
    }
   // inode = statbuf.st_ino;
  //  DPRINTK(2, "%d: %s: inode %ld\n", cur_pid, cpsMethod, inode);
    st_mode = statbuf.st_mode;
   // DPRINTK(2, "%d: %s: st_mode %x\n", cur_pid, cpsMethod, st_mode);
    if (!S_ISNORMAL(st_mode))
    {
        DPRINTK(2, "%d: %s: not regular file or link\n", cur_pid, cpsMethod);
        goto jump;
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW | LOOKUP_POSITIVE, &nd);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    error = kern_path(filename, LOOKUP_FOLLOW, &pPath);
#else
    //TT215640: Crash happens on NFSv4 share directory
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW, &nd);
#endif
    if (error)
        goto jump;

    read_lock(&kini_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    inode = pPath.dentry->d_inode;
    scan_execve = needToScanThisExecve(pPath.dentry, pPath.mnt, inode);
#else
    inode = nd.DENTRY->d_inode;
    scan_execve = needToScanThisExecve(nd.DENTRY, nd.MNT, inode);
#endif
    read_unlock(&kini_lock);

    if (!scan_execve)
    {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }

    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_ATOMIC);
    if (ip == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: ip is NULL\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    memset(ip, 0, sizeof(LIST_ITEM));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.scan_args.full_pn = (char *)pPath.dentry->d_name.name;
#else
    ip->info.scan_args.full_pn = (char *)nd.DENTRY->d_name.name;
#endif
    clen = strlen(current->comm);
    comm = (char *)kmalloc(clen + 1, GFP_ATOMIC);
    if (comm == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: comm is NULL\n", cur_pid, cpsMethod);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    ip->info.scan_args.comm = comm;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
#ifdef CONFIG_UIDGID_STRICT_TYPE_CHECKS
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,14,0)
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
    ip->info.scan_args.comm_uid = current->SPLX_UID;
#endif
#endif

    temp_ip =  ip;
    ip->info.scan_args.u_lip = (((unsigned long)temp_ip) & 0xffffffff00000000) >> 32;
    temp_ip =  ip;
    ip->info.scan_args.d_lip = ((unsigned long)temp_ip) & 0x0000ffffffffffff;
    
    ip->info.scan_args.inode = inode->i_ino;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = TRUE;
    atomic_set(&(ip->info.cond), FALSE);
	atomic_set(&(ip->info.ref_count), 1);
	init_waitqueue_head(&(ip->info.wait_queue));

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
    ip->info.path = &pPath;
#elif LINUX_VERSION_CODE >= 0x30000
    ip->info.dentry = pPath.dentry;
    ip->info.mnt = pPath.mnt;
#else
    ip->info.dentry = nd.DENTRY;
    ip->info.mnt = nd.MNT;
#endif

    INIT_LIST_HEAD(&ip->item_list);
    read_lock(&kini_lock);
    timeout = scan_timeout_HZ;
    read_unlock(&kini_lock);
    if (!initialized())
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
        ip = NULL;
        goto jump;
    }
    DPRINTK(2, "%d: %s: start to scan this execve\n", cur_pid, cpsMethod);
    found = insertList(ip);
    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;

        MOD_DEC_REF_COUNT;
        SetAuditContext(current_audit);
        return (-EACCES);
    }
    atomic_set(&candidate, TRUE);
    wake_up_interruptible(&vsapi_chldn_wq);
    /* scan this execve() */
    //    if (timeout > 0) {
    //  /* to be implemented */
    //    } else {
    DPRINTK(2, "%d: %s: sleep on execve_wq\n", cur_pid, cpsMethod);
    wait_event(ip->info.wait_queue, atomic_read(&(ip->info.cond)));
    //    }

    DPRINTK(3, "%d: %s: other pid %d, fd %d, filename %s\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.fd, ip->info.scan_args.full_pn);

    if (ip->info.vsapi_busy == FALSE)
    {
        vsapi_ret = ip->info.scan_args.vsapi_ret;
        action = ip->info.scan_args.action;
        if (vsapi_ret == VIRUS_FOUND)
        {
            /* should be revised here */
            removeCache(inode);
            //DPRINTK(3, "%d: %s: .remove file [%s] from Cache.\n", cur_pid, cpsMethod,  (char *)(ip->info.dentry->d_name.name));
            switch (action)
            {
            case CLEAN:
                DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                break;
            case DELETE:
                DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                break;
            case MOVE:
                DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                break;
            case RENAME:
                DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                break;
            case BYPASS:
                DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                break;
            case DENYACCESS:
                DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;

                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            default:
                DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;
                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            }
        }
        else if (vsapi_ret == NO_VIRUS)
        {
            /*
            * only perfectly clean files can be
            * added to the cache
            */
            addCache(inode);
        }
        else
        {
            DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
            found = deleteList(ip);
            if (!found)
                DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
            ip = NULL;
            /* permission denied */
            ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
            putname(filename_s);
            filename_s = NULL;
            filename = NULL;
#else
            putname(filename);
            filename = NULL;
#endif
*/
            khm_putname(filename);
            filename = NULL;

            MOD_DEC_REF_COUNT;
            SetAuditContext(current_audit);
            return (ret);
        }
    }
    else     /* VSAPI time out */
    {
        vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
        if (vsapi_chld_pid > 0)
        {
            /* from kernel/signal.c { */
            struct  siginfo info;
            info.si_signo = SIGTERM;
            info.si_errno = 0;
            info.si_code = SI_USER;
            info.si_pid = ip->
                          info.scan_args.vsapi_chld_pid;
            info.si_uid = 0;
            splx_kill_proc(info.si_pid, info.si_signo, (long)&info);
            /* from kernel/signal.c } */
        }
    }

    if (ip != NULL)
    {
        found = deleteList(ip);
        if (!found)
            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
    }
jump:
    //First relase filename and set back audit flag.
    if (filename != NULL)
    {
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;
    }
    SetAuditContext(current_audit);
    //Re-do filename
/*
#if RHEL_RELEASE_CODE >= 1541
    filename_s = getname(name);
    filename = (char *)filename_s->name;
#else
    filename = getname(name);
#endif
*/
    filename = khm_getname(name);
    error = PTR_ERR(filename);
    if (IS_ERR(filename))
        goto out;
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
    error = orig_do_execve(filename, argv, envp, &regs);
#else
    error = orig_do_execve(filename, argv, envp, regs);
#endif

#ifndef CONFIG_UTRACE
    if (error == 0)
    {
        task_lock(current);
        current->ptrace &= ~PT_DTRACE;
        task_unlock(current);
    }
#endif
    DPRINTK(3, "%d: %s: get outta execveHook\n", cur_pid, cpsMethod);
    if (filename != NULL)
    {
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;
    }
out:
    MOD_DEC_REF_COUNT;
    return error;
}


#ifdef IA32_HOOK

asmlinkage long IA32_execveHook(char __user *name, compat_uptr_t __user *argv,
                                compat_uptr_t __user *envp, struct pt_regs *regs)
{
    const char *cpsMethod = "IA32_execveHook";
    struct inode  *inode = NULL;
    mode_t  st_mode;
    int ret;
    pid_t   cur_pid = 0;
    Boolean found;
    long error;
    LIST_ITEM   *ip = NULL;
    char    *comm = NULL;
    int vsapi_ret;
    int action;
    int clen;
    int vsapi_chld_pid;
    struct stat statbuf;
    char *filename;

    long timeout;
    Boolean scan_execve = FALSE;
    void *temp_ip;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
#else
    struct nameidata nd;
#endif
    struct audit_context *current_audit = NULL;
    //  mm_segment_t oldfs = get_fs();
    CP_DBG_LVL;

    MOD_INC_REF_COUNT;
    //Add for avoid conflict with auditd
    current_audit = ClearAuditContext();
    //Add end

    cur_pid = current->pid;
    DPRINTK(3, "%d: %s: get into IA32_execveHook\n", cur_pid, cpsMethod);
    found = deleteListPID(cur_pid);
    if (!found)
        DPRINTK(3, "%d: %s: deleteListPID not found\n", cur_pid, cpsMethod);

    filename = khm_getname(name);
    error = PTR_ERR(filename);
    if(IS_ERR(filename))
    {
        goto jump;
    }

    DPRINTK(1, "%d: %s: get into IA32_execveHook filename %s\n", cur_pid, cpsMethod, filename);

    //First check whether the command is in the exclusion list
    if (InExcComms(current->comm))
    {
        DPRINTK(3, "%d: %s: not to scan because command [%s] in the command exclusion list\n", cur_pid, cpsMethod, current->comm);
        goto jump;
    }

    if (!initialized())
    {
        DPRINTK(2, "%d: %s: vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod);
        goto jump;
    }

    read_lock(&kini_lock);
    DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (inExcForest(exc_pid_ary, exc_pid_no))
    {
        read_unlock(&kini_lock);
        goto jump;
    }
    read_unlock(&kini_lock);

    error = getStat(filename, &statbuf);
    if (error)
    {
        DPRINTK(2, "%d: %s: stat error %ld\n", cur_pid, cpsMethod, error);
        goto jump;
    }
    //inode = statbuf.st_ino;
    //DPRINTK(2, "%d: %s: inode %ld\n", cur_pid, cpsMethod, inode);
    st_mode = statbuf.st_mode;
    //DPRINTK(2, "%d: %s: st_mode %x\n", cur_pid, cpsMethod, st_mode);
    if (!S_ISNORMAL(st_mode))
    {
        DPRINTK(2, "%d: %s: not regular file or link\n", cur_pid, cpsMethod);
        goto jump;
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW | LOOKUP_POSITIVE, &nd);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    error = kern_path(filename, LOOKUP_FOLLOW, &pPath);
#else
    //TT215640: Crash happens on NFSv4 share directory
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW, &nd);
#endif
    if (error)
        goto jump;

    read_lock(&kini_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    inode = pPath.dentry->d_inode;
    scan_execve = needToScanThisExecve(pPath.dentry, pPath.mnt, inode);
#else
    inode = nd.DENTRY->d_inode;
    scan_execve = needToScanThisExecve(nd.DENTRY, nd.MNT, inode);
#endif
    read_unlock(&kini_lock);

    if (!scan_execve)
    {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }

    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_ATOMIC);
    if (ip == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: ip is NULL\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    memset(ip, 0, sizeof(LIST_ITEM));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.scan_args.full_pn = (char *)pPath.dentry->d_name.name;
#else
    ip->info.scan_args.full_pn = (char *)nd.DENTRY->d_name.name;
#endif
    clen = strlen(current->comm);
    comm = (char *)kmalloc(clen + 1, GFP_ATOMIC);
    if (comm == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: comm is NULL\n", cur_pid, cpsMethod);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    ip->info.scan_args.comm = comm;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
#ifdef CONFIG_UIDGID_STRICT_TYPE_CHECKS
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,14,0)
    ip->info.scan_args.comm_uid = __kuid_val(current->SPLX_UID);
#else
    ip->info.scan_args.comm_uid = current->SPLX_UID;
#endif
#endif
    temp_ip =  ip;
    ip->info.scan_args.u_lip = (((unsigned long)temp_ip) & 0xffffffff00000000) >> 32;
    temp_ip =  ip;
    ip->info.scan_args.d_lip = ((unsigned long)temp_ip) & 0x0000ffffffffffff;
    ip->info.scan_args.inode = inode->i_ino;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = TRUE;
    atomic_set(&(ip->info.cond), FALSE);
	atomic_set(&(ip->info.ref_count), 1);
	init_waitqueue_head(&(ip->info.wait_queue));
    
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
    ip->info.path = &pPath;
#elif LINUX_VERSION_CODE >= 0x30000
    ip->info.dentry = pPath.dentry;
    ip->info.mnt = pPath.mnt;
#else
    ip->info.dentry = nd.DENTRY;
    ip->info.mnt = nd.MNT;
#endif

    INIT_LIST_HEAD(&ip->item_list);
    read_lock(&kini_lock);
    timeout = scan_timeout_HZ;
    read_unlock(&kini_lock);
    if (!initialized())
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
        ip = NULL;
        goto jump;
    }
    DPRINTK(2, "%d: %s: start to scan this execve\n", cur_pid, cpsMethod);
    found = insertList(ip);
    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);

        khm_putname(filename);
        filename = NULL;

        MOD_DEC_REF_COUNT;
        SetAuditContext(current_audit);
        return (-EACCES);
    }
    atomic_set(&candidate, TRUE);
    wake_up_interruptible(&vsapi_chldn_wq);
    /* scan this execve() */
    //    if (timeout > 0) {
    //  /* to be implemented */
    //    } else {
    DPRINTK(2, "%d: %s: sleep on ia32_execve_wq\n", cur_pid, cpsMethod);
    wait_event(ip->info.wait_queue, atomic_read(&(ip->info.cond)));
    //    }

    DPRINTK(3, "%d: %s: other pid %d, fd %d, filename %s\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.fd, ip->info.scan_args.full_pn);

    if (ip->info.vsapi_busy == FALSE)
    {
        vsapi_ret = ip->info.scan_args.vsapi_ret;
        action = ip->info.scan_args.action;
        if (vsapi_ret == VIRUS_FOUND)
        {
            /* should be revised here */
            removeCache(inode);
            switch (action)
            {
            case CLEAN:
                DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                break;
            case DELETE:
                DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                break;
            case MOVE:
                DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                break;
            case RENAME:
                DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                break;
            case BYPASS:
                DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                break;
            case DENYACCESS:
                DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;

                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            default:
                DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;

                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            }
        }
        else if (vsapi_ret == NO_VIRUS)
        {
            /*
            * only perfectly clean files can be
            * added to the cache
            */
            addCache(inode);
        }
        else
        {
            DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
            found = deleteList(ip);
            if (!found)
                DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
            ip = NULL;
            /* permission denied */
            ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
            putname(filename_s);
            filename_s = NULL;
            filename = NULL;
#else
            putname(filename);
            filename = NULL;
#endif
*/
            khm_putname(filename);
            filename = NULL;

            MOD_DEC_REF_COUNT;
            SetAuditContext(current_audit);
            return (ret);
        }
    }
    else     /* VSAPI time out */
    {
        vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
        if (vsapi_chld_pid > 0)
        {
            /* from kernel/signal.c { */
            struct  siginfo info;
            info.si_signo = SIGTERM;
            info.si_errno = 0;
            info.si_code = SI_USER;
            info.si_pid = ip->
                          info.scan_args.vsapi_chld_pid;
            info.si_uid = 0;
            splx_kill_proc(info.si_pid, info.si_signo, (long)&info);
            /* from kernel/signal.c } */
        }
    }

    if (ip != NULL)
    {
        found = deleteList(ip);
        if (!found)
            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
    }
jump:
    // Rlease filename then set audit flag back.
    if (filename != NULL)
    {

        khm_putname(filename);
        filename = NULL;
    }
    SetAuditContext(current_audit);
    // Re-do filename

    filename = khm_getname(name);
    error = PTR_ERR(filename);
    if(IS_ERR(filename))
    {
        goto out;
    }

    error = IA32_orig_compat_do_execve(filename, argv, envp, regs);

#ifndef CONFIG_UTRACE
    if (error == 0)
    {
        task_lock(current);
        current->ptrace &= ~PT_DTRACE;
        task_unlock(current);
    }
#endif

    DPRINTK(3, "%d: %s: get outta IA32_execveHook\n", cur_pid, cpsMethod);
    if (filename != NULL)
    {

        khm_putname(filename);
        filename = NULL;

    }
out:
    MOD_DEC_REF_COUNT;
    return error;
}
#endif

#else

asmlinkage int execveHook(struct pt_regs regs)
{

    const char *cpsMethod = "execveHook";
    struct inode   *inode = NULL;
    mode_t    st_mode;
    int    ret;
    pid_t   cur_pid = 0;
    Boolean found;
    int error;
    LIST_ITEM    *ip = NULL;
    char    *comm = NULL;
    int    vsapi_ret;
    int    action;
    int    clen;
    int    vsapi_chld_pid;
    struct stat    statbuf;
    char *filename;
/*
#if RHEL_RELEASE_CODE >= 1541
    struct filename *filename_s = NULL;
#endif
*/
    long timeout;
    struct audit_context *current_audit;
    Boolean scan_execve = FALSE;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    struct path pPath;
#else
    struct nameidata nd;
#endif

    CP_DBG_LVL;

    MOD_INC_REF_COUNT;

    //Add for avoid conflict with auditd
    current_audit = ClearAuditContext();
    //Add end

    cur_pid = current->pid;

    DPRINTK(3, "%d: %s: get into execveHook\n", cur_pid, cpsMethod);
    found = deleteListPID(cur_pid);
    if (!found)
        DPRINTK(3, "%d: %s: deleteListPID not found\n", cur_pid, cpsMethod);

/*
#if RHEL_RELEASE_CODE >= 1541
    filename_s = getname((char __user *) regs.bx);
    error = PTR_ERR(filename_s);
    if (IS_ERR(filename_s))
    {
        goto out;
    }

    filename = (char *)filename_s->name;
#else

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)

#ifdef S390
    filename = getname((char *) regs.orig_gpr2);
#else
    filename = getname((char *) regs.ebx);
#endif

#else
    filename = getname((char __user *) regs.bx);
#endif
    error = PTR_ERR(filename);
    if (IS_ERR(filename))
    {
        goto jump;        
    }
#endif
*/
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)

#ifdef S390
    filename = khm_getname((char *) regs.orig_gpr2);
#else
    filename = khm_getname((char *) regs.ebx);
#endif

#else
    filename = khm_getname((char __user *) regs.bx);
#endif

    error = PTR_ERR(filename);
    if (IS_ERR(filename))
    {
        goto jump;        
    }

    DPRINTK(3, "%d: %s: get into execveHook filename %s\n", cur_pid, cpsMethod, filename);

    //First check whether the command is in the exclusion list
    if (InExcComms(current->comm))
    {
        DPRINTK(2, "%d: %s: not to scan because command [%s] in the command exclusion list\n", cur_pid, cpsMethod, current->comm);
        goto jump;
    }

    if (!initialized())
    {
        DPRINTK(2, "%d: %s: vsapi_chldn_no=0 || !inited, goto jump\n", cur_pid, cpsMethod);
        goto jump;
    }

    read_lock(&kini_lock);
    DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (inExcForest(exc_pid_ary, exc_pid_no))
    {
        read_unlock(&kini_lock);
        goto jump;
    }
    read_unlock(&kini_lock);

    error = getStat(filename, &statbuf);
    if (error)
    {
        DPRINTK(2, "%d: %s: stat error %d\n", cur_pid, cpsMethod, error);
        goto jump;
    }
    //inode = statbuf.st_ino;
   // DPRINTK(2, "%d: %s: inode %ld\n", cur_pid, cpsMethod, inode);
    st_mode = statbuf.st_mode;
    //DPRINTK(2, "%d: %s: st_mode %x\n", cur_pid, cpsMethod, st_mode);
    if (!S_ISNORMAL(st_mode))
    {
        DPRINTK(2, "%d: %s: not regular file or link\n", cur_pid, cpsMethod);
        goto jump;
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW | LOOKUP_POSITIVE, &nd);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    error = kern_path(filename, LOOKUP_FOLLOW, &pPath);
#else
    //TT215640: Crash happens on NFSv4 share directory
    error = PATH_LOOKUP(filename, LOOKUP_FOLLOW, &nd);
#endif
    if (error)
        goto jump;

    read_lock(&kini_lock);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    inode = pPath.dentry->d_inode;
    scan_execve = needToScanThisExecve(pPath.dentry, pPath.mnt, inode);
#else
    inode = nd.DENTRY->d_inode;
    scan_execve = needToScanThisExecve(nd.DENTRY, nd.MNT, inode);
#endif
    read_unlock(&kini_lock);

    if (!scan_execve)
    {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }

    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_ATOMIC);
    if (ip == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: ip is NULL\n", cur_pid, cpsMethod);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    memset(ip, 0, sizeof(LIST_ITEM));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.scan_args.full_pn = (char *) pPath.dentry->d_name.name;
#else
    ip->info.scan_args.full_pn = (char *) nd.DENTRY->d_name.name;
#endif
    clen = strlen(current->comm);
    comm = (char *)kmalloc(clen + 1, GFP_ATOMIC);
    if (comm == NULL)
    {
        WPRINTK("SPLXMOD: %d: %s: comm is NULL\n", cur_pid, cpsMethod);
        kfree(ip);
        ip = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        goto jump;
    }
    ip->info.scan_args.comm = comm;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
    ip->info.scan_args.comm_uid = current->SPLX_UID;
    ip->info.scan_args.lip = ip;
    ip->info.scan_args.inode = inode->i_ino;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = TRUE;
    atomic_set(&(ip->info.cond), FALSE);
    atomic_set(&(ip->info.ref_count), 1);
	init_waitqueue_head(&(ip->info.wait_queue));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
    ip->info.dentry = pPath.dentry;
    ip->info.mnt = pPath.mnt;
#else
    ip->info.dentry = nd.DENTRY;
    ip->info.mnt = nd.MNT;
#endif
    INIT_LIST_HEAD(&ip->item_list);
    read_lock(&kini_lock);
    timeout = scan_timeout_HZ;
    read_unlock(&kini_lock);
    if (!initialized())
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
        ip = NULL;
        goto jump;
    }
    DPRINTK(2, "%d: %s: start to scan this execve\n", cur_pid, cpsMethod);
    found = insertList(ip);
    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            ip->info.scan_args.full_pn = NULL;
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)
        path_put(&pPath);
#else
        SPLX_PATH_RELEASE(&nd);
#endif
        kfree(ip);
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;

        MOD_DEC_REF_COUNT;
        SetAuditContext(current_audit);
        return (-EACCES);
    }
    atomic_set(&candidate, TRUE);
    wake_up_interruptible(&vsapi_chldn_wq);
    /* scan this execve() */
    //    if (timeout > 0) {
    //    /* to be implemented */
    //    } else {
    DPRINTK(2, "%d: %s: sleep on execve_wq\n", cur_pid, cpsMethod);
    wait_event(ip->info.wait_queue, atomic_read(&(ip->info.cond)));
    //    }

    DPRINTK(3, "%d: %s: other pid %d, fd %d, filename %s\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.fd, ip->info.scan_args.full_pn);

    if (ip->info.vsapi_busy == FALSE)
    {
        vsapi_ret = ip->info.scan_args.vsapi_ret;
        action = ip->info.scan_args.action;
        if (vsapi_ret == VIRUS_FOUND)
        {
            /* should be revised here */
            removeCache(inode);
            switch (action)
            {
            case CLEAN:
                DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                break;
            case DELETE:
                DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                break;
            case MOVE:
                DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                break;
            case RENAME:
                DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                break;
            case BYPASS:
                DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                break;
            case DENYACCESS:
                DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;

                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            default:
                DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
                putname(filename_s);
                filename_s = NULL;
                filename = NULL;
#else
                putname(filename);
                filename = NULL;
#endif
*/
                khm_putname(filename);
                filename = NULL;

                MOD_DEC_REF_COUNT;
                SetAuditContext(current_audit);
                return (ret);
            }
        }
        else if (vsapi_ret == NO_VIRUS)
        {
            /*
            * only perfectly clean files can be
            * added to the cache
            */
            addCache(inode);
        }
        else
        {
            DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
            found = deleteList(ip);
            if (!found)
                DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
            ip = NULL;
            /* permission denied */
            ret = -EACCES;
/*
#if RHEL_RELEASE_CODE >= 1541
            putname(filename_s);
            filename_s = NULL;
            filename = NULL;
#else
            putname(filename);
            filename = NULL;
#endif
*/
            khm_putname(filename);
            filename = NULL;

            MOD_DEC_REF_COUNT;
            SetAuditContext(current_audit);
            return (ret);
        }
    }
    else     /* VSAPI time out */
    {
        vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
        if (vsapi_chld_pid > 0)
        {
            /* from kernel/signal.c { */
            struct    siginfo info;
            info.si_signo = SIGTERM;
            info.si_errno = 0;
            info.si_code = SI_USER;
            info.si_pid = ip->
                          info.scan_args.vsapi_chld_pid;
            info.si_uid = 0;
            splx_kill_proc(info.si_pid, info.si_signo, (int)&info);
            /* from kernel/signal.c } */
        }
    }

    if (ip != NULL)
    {
        found = deleteList(ip);
        if (!found)
            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
    }
jump:
    //First release the filename then set audit flag back. Do that not to impact original execve
#ifndef S390
    if (filename != NULL)
    {
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;
    }
#endif
    SetAuditContext(current_audit);
    //Re-do: filename
/*
#if RHEL_RELEASE_CODE >= 1541
    filename_s = getname((char __user *) regs.bx);
    error = PTR_ERR(filename_s);
    if (IS_ERR(filename_s))
    {
        goto out;
    }

    filename = (char *)filename_s->name;
#else
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#ifdef S390
    filename = getname((char *) regs.orig_gpr2);
#else
    filename = getname((char *) regs.ebx);
#endif
#else
    filename = getname((char __user *) regs.bx);
#endif
#endif
*/
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#ifdef S390
    filename = khm_getname((char *) regs.orig_gpr2);
#else
    filename = khm_getname((char *) regs.ebx);
#endif
#else
    filename = khm_getname((char __user *) regs.bx);
#endif
    error = PTR_ERR(filename);
    if (IS_ERR(filename))
    {
        goto out;
    }

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#ifdef NAS
    __asm__ volatile ("int $0x80"
                      : "=a" (error)
                      : "0" (NR_syscalls-1),
                      "b" ((long)regs.ebx),
                      "c" ((long)regs.ecx),
                      "d" ((long)regs.edx));
#else
#ifdef S390
    error = do_execve(filename, (char **) regs.gprs[3], (char **) regs.gprs[4], &regs);
    if (error == 0)
    {
        current->ptrace &= ~PT_DTRACE;
    }
#else
    error = orig_do_execve(filename, (char __user * __user *) regs.ecx, (char __user * __user *) regs.edx, &regs);

    if (error == 0)
    {
#ifndef CONFIG_UTRACE
        task_lock(current);
        current->ptrace &= ~PT_DTRACE;
        task_unlock(current);
#endif
        /* Make sure we don't return using sysenter.. */
        set_thread_flag(TIF_IRET);
    }
#endif
#endif
#else

    error = orig_do_execve(filename,
                           (char __user * __user *) regs.cx,
                           (char __user * __user *) regs.dx,
                           &regs);

    if (error == 0)
    {
#ifndef CONFIG_UTRACE
        task_lock(current);
        current->ptrace &= ~PT_DTRACE;
        task_unlock(current);
#endif

        /* Make sure we don't return using sysenter.. */
        set_thread_flag(TIF_IRET);
    }

#endif

#ifndef S390
    if (filename != NULL)
    {
/*
#if RHEL_RELEASE_CODE >= 1541
        putname(filename_s);
        filename_s = NULL;
        filename = NULL;
#else
        putname(filename);
        filename = NULL;
#endif
*/
        khm_putname(filename);
        filename = NULL;
    }
#endif

    DPRINTK(2, "%d: %s: error [%d]\n", cur_pid, cpsMethod, error);
    DPRINTK(1, "%d: %s: get outta execveHook\n", cur_pid, cpsMethod);
out:
    MOD_DEC_REF_COUNT;
    return error;
}

#ifdef  NFSD
int nfsdOpenHook(struct svc_rqst *rqstp, struct svc_fh *fhp, int type,
                 int access, struct file *filp)
{
    const char *cpsMethod = "nfsdOpenHook";
    pid_t cur_pid = 0;
    struct inode   *inode = NULL;
    mode_t    st_mode;
    int    ret;
    LIST_ITEM    *ip = NULL;
    char    *cp = NULL;
    Boolean    found;
    int    vsapi_ret;
    int    action;
    int    clen;
    int    vsapi_chld_pid;
    int    error;
    char    *filename;
    int    flags;
    struct dentry   *dentry;
    struct vfsmount *vfsmnt;
    char *page = (char *) __get_free_page(GFP_KERNEL);
    CP_DBG_LVL;

    cur_pid = current->pid;
    error = fh_verify(rqstp, fhp, type, access | MAY_OWNER_OVERRIDE);
    if (error)
        return error;
    dentry = fhp->fh_dentry;
    filename = dentry->d_name.name;
    if (access & MAY_WRITE)
    {
        flags = O_WRONLY | O_LARGEFILE;
    }
    else
    {
        flags = O_RDONLY | O_LARGEFILE;
    }
    DPRINTK(1, "%d: %s: get into nfsdOpenHook filename %s, flags %x\n", cur_pid, cpsMethod, filename, flags);

    spin_lock(&my_lock);
    DPRINTK(2, "%d: %s: vsapi_chldn_no %d\n", cur_pid, cpsMethod, vsapi_chldn_no);
    if (vsapi_chldn_no == 0) goto jump;

    if (inExcForest(exc_pid_ary, exc_pid_no)) goto jump;

    spin_unlock(&my_lock);
    vfsmnt = fhp->fh_export->ex_mnt;
    if (page)
    {
        unsigned long len;
        char *fpn = d_path(dentry, vfsmnt, page, PAGE_SIZE);
        len = PAGE_SIZE + page - fpn;
        if (len <= MAX_PATH_LEN_DEF)
        {
            cp = (char *)kmalloc(len + 1, GFP_KERNEL);
            if (cp == NULL)
                printk(KERN_WARNING "SPLXMOD: cp is NULL\n", cur_pid, cpsMethod);
            strncpy(cp, fpn, len + 1);
            DPRINTK(2, "%d: %s: full pathname %s\n", cur_pid, cpsMethod, cp);
        }
        free_page((unsigned long) page);
    }
    ip = (LIST_ITEM *)kmalloc(sizeof(LIST_ITEM), GFP_KERNEL);
    if (ip == NULL)
        printk(KERN_WARNING "SPLXMOD: ip is NULL\n", cur_pid, cpsMethod);
    memset(ip, 0, sizeof(LIST_ITEM));
    ip->info.scan_args.full_pn = cp;
    clen = strlen(current->comm);
    cp = (char *)kmalloc(clen + 1, GFP_KERNEL);
    if (cp == NULL)
        printk(KERN_WARNING "SPLXMOD: cp is NULL\n", cur_pid, cpsMethod);
    ip->info.scan_args.comm = cp;
    strncpy(ip->info.scan_args.comm, current->comm, clen + 1);
    ip->info.scan_args.comm_pid = current->pid;
    ip->info.scan_args.comm_uid = current->SPLX_UID;
    ip->info.scan_args.flags = flags;
    ip->info.scan_args.fd = filp;
    ip->info.scan_args.lip = ip;
    ip->info.vsapi_busy = FALSE;
    ip->info.candid = FALSE;
    init_waitqueue_head(&(ip->info.other_proc_wq));
    ip->next = NULL;
    spin_lock(&my_lock);
    found = insertList(ip);
    spin_unlock(&my_lock);
    if (!found)
    {
        if (ip->info.scan_args.full_pn != NULL)
            kfree(ip->info.scan_args.full_pn);
        if (ip->info.scan_args.comm != NULL)
            kfree(ip->info.scan_args.comm);
        kfree(ip);
        return (-EACCES);
    }

    spin_lock(&my_lock);
    inode = dentry->d_inode;
    ip->info.scan_args.inode = inode->i_ino;
    DPRINTK(2, "%d: %s: inode %p\n", cur_pid, cpsMethod, inode);
    st_mode = dentry->d_inode->i_mode;
    DPRINTK(2, "%d: %s: st_mode %x\n", cur_pid, cpsMethod, st_mode);
    if (S_ISDIR(st_mode) || S_ISCHR(st_mode) || S_ISBLK(st_mode) ||    S_ISFIFO(st_mode) || S_ISSOCK(st_mode))
    {
        DPRINTK(2, "%d: %s: not regular file or link\n", cur_pid, cpsMethod);
        found = deleteList(ip);
        if (!found)
            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
        ip = NULL;
        goto jump;
    }

    /* scan this open() only when opened in RDONLY/RDWR mode */
    if (needToScanThisOpen(ip->info.scan_args.full_pn, flags, inode))
    {
        long timeout = scan_timeout_HZ;
        DPRINTK(2, "%d: %s: start to scan this open\n", cur_pid, cpsMethod);
        ip->info.candid = TRUE;
        DPRINTK(2, "%d: %s: wake up vsapi_chldn_wq\n", cur_pid, cpsMethod);
        WAKE_UP_INTERRUPTIBLE(&vsapi_chldn_wq);
        DPRINTK(3, "%d: %s: sleep on ip->info.other_proc_wq with scan_timeout_HZ\n", cur_pid, cpsMethod);
        spin_unlock(&my_lock);
        if (timeout > 0)
        {
            sleep_on_timeout(
                &(ip->info.other_proc_wq),
                timeout);
        }
        else
        {
            sleep_on(
                &(ip->info.other_proc_wq));
        }
        spin_lock(&my_lock);
        DPRINTK(3, "%d: %s: other pid %d, fd %d, filename %s\n", cur_pid, cpsMethod, ip->info.scan_args.comm_pid, ip->info.scan_args.fd, ip->info.scan_args.full_pn);

        if (ip->info.vsapi_busy == FALSE)
        {
            vsapi_ret = ip->info.scan_args.vsapi_ret;
            action = ip->info.scan_args.action;
            if (vsapi_ret == VIRUS_FOUND)
            {
                /* should be revised here */
                removeCache(inode);
                switch (action)
                {
                case CLEAN:
                    DPRINTK(3, "%d: %s: action CLEAN\n", cur_pid, cpsMethod);
                    if ((flags & O_ACCMODE) == O_RDONLY)
                    {
                        found = deleteList(ip);
                        if (!found)
                            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                        ip = NULL;
                    }
                    break;
                case DELETE:
                    DPRINTK(3, "%d: %s: action DELETE\n", cur_pid, cpsMethod);
                    break;
                case MOVE:
                    DPRINTK(3, "%d: %s: action MOVE\n", cur_pid, cpsMethod);
                    break;
                case RENAME:
                    DPRINTK(3, "%d: %s: action RENAME\n", cur_pid, cpsMethod);
                    break;
                case BYPASS:
                    DPRINTK(3, "%d: %s: action BYPASS\n", cur_pid, cpsMethod);
                    if ((flags & O_ACCMODE) == O_RDONLY)
                    {
                        found = deleteList(ip);
                        if (!found)
                            DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                        ip = NULL;
                    }
                    break;
                case DENYACCESS:
                    DPRINTK(3, "%d: %s: action DENYACCESS\n", cur_pid, cpsMethod);
                    found = deleteList(ip);
                    if (!found)
                        DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                    ip = NULL;
                    /* permission denied */
                    ret = -EACCES;
                    spin_unlock(&my_lock);
                    return (ret);
                default:
                    DPRINTK(3, "%d: %s: action UNKNOWN\n", cur_pid, cpsMethod);
                    found = deleteList(ip);
                    if (!found)
                        DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                    ip = NULL;
                    /* permission denied */
                    ret = -EACCES;
                    spin_unlock(&my_lock);
                    return (ret);
                }
            }
            else if (vsapi_ret == NO_VIRUS)
            {
                /*
                * only perfectly clean files can be
                * added to the cache
                */
                if ((flags & O_ACCMODE) == O_RDONLY)
                {
                    addCache(inode);
                    found = deleteList(ip);
                    if (!found)
                        DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                    ip = NULL;
                }
            }
            else
            {
                DPRINTK(3, "%d: %s: vsapi_ret UNKNOWN\n", cur_pid, cpsMethod);
                found = deleteList(ip);
                if (!found)
                    DPRINTK(3, "%d: %s: deleteList not found\n", cur_pid, cpsMethod);
                ip = NULL;
                /* permission denied */
                ret = -EACCES;
                spin_unlock(&my_lock);
                return (ret);
            }
        }
        else     /* VSAPI time out */
        {
            vsapi_chld_pid = ip->info.scan_args.vsapi_chld_pid;
            if (vsapi_chld_pid > 0)
            {
                /* from kernel/signal.c { */
                struct    siginfo info;
                info.si_signo = SIGTERM;
                info.si_errno = 0;
                info.si_code = SI_USER;
                info.si_pid = ip->
                              info.scan_args.vsapi_chld_pid;
                info.si_uid = 0;
                splx_kill_proc(info.si_pid, info.si_signo, (int)&info);
                /* from kernel/signal.c } */
            }
        }
    }

jump:
    spin_unlock(&my_lock);
    /* SPNT 5.x scans first and then open here if RDONLY */
    ret = orig_nfsd_open(rqstp, fhp, type, access, filp);
    spin_lock(&my_lock);

    DPRINTK(2, "%d: %s: ret %d\n", cur_pid, cpsMethod, ret);
    if (ret != 0)
    {
        if (ip != NULL)
        {
            deleteList(ip);
            ip = NULL;
        }
    }

    DPRINTK(1, "%d: %s: get outta nfsdOpenHook\n", cur_pid, cpsMethod);
    spin_unlock(&my_lock);
    return (ret);
}
#endif
#endif
#endif


#ifdef KHM_PART("API")

#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER

asmlinkage long openHook(const struct pt_regs *regs)
{
    return open_hook_comm(FALSE,OPEN_HOOK, orig_open, regs);
}
asmlinkage long openatHook(const struct pt_regs *regs)
{
    return open_hook_comm(FALSE,OPENAT_HOOK, orig_openat, regs);
}

asmlinkage long closeHook(const struct pt_regs *regs)
{
    return close_hook_comm(FALSE,CLOSE_HOOK, orig_close, regs);
}
asmlinkage long dup2Hook(const struct pt_regs *regs)
{
    return close_hook_comm(FALSE,DUP2_HOOK, orig_dup2, regs);
}

asmlinkage long dup3Hook(const struct pt_regs *regs)
{
    return close_hook_comm(FALSE,DUP3_HOOK, orig_dup3, regs);
}

asmlinkage long writeHook(const struct pt_regs *regs)
{
    return write_hook_comm(FALSE, WRITE_HOOK, orig_write, regs);
}


asmlinkage long pwrite64Hook(const struct pt_regs *regs)
{
    return write_hook_comm(FALSE,PWRITE64_HOOK, orig_pwrite64, regs);
}
asmlinkage long writevHook(const struct pt_regs *regs)
{
    return write_hook_comm(FALSE,WRITEV_HOOK, orig_writev, regs);
}
asmlinkage long pwritevHook(const struct pt_regs *regs)
{
    return write_hook_comm(FALSE,PWRITEV_HOOK, orig_pwritev, regs);
}
asmlinkage long exitHook(const struct pt_regs *regs)
{
    return exit_hook_comm(EXIT_HOOK, orig_exit, regs);
}
asmlinkage long exitgroupHook(const struct pt_regs *regs)
{
    return exit_hook_comm(EXIT_GROUP_HOOK, orig_exit_group, regs);
}

asmlinkage long umountHook(const struct pt_regs *regs)
{
    return umount_hook_comm(FALSE,UMOUNT_HOOK, orig_umount, regs);
}

asmlinkage long umount2Hook(const struct pt_regs *regs)
{
    return umount_hook_comm(FALSE,UMOUNT2_HOOK, orig_umount2, regs);
}

asmlinkage long unlinkHook(const struct pt_regs *regs)
{
    return unlink_hook_comm(FALSE,UNLINK_HOOK, orig_unlink, regs);
}

asmlinkage long unlinkatHook(const struct pt_regs *regs)
{
    return unlink_hook_comm(FALSE,UNLINKAT_HOOK, orig_unlinkat, regs);
}

#ifdef IA32_HOOK //32bit system call in 64bit platform
asmlinkage long IA32_openHook(const struct pt_regs *regs)
{
    return open_hook_comm(TRUE, OPEN_HOOK, IA32_orig_open, regs);
}
asmlinkage long IA32_openatHook(const struct pt_regs *regs)
{
    return open_hook_comm(TRUE, OPENAT_HOOK, IA32_orig_openat, regs);
}

asmlinkage long IA32_closeHook(const struct pt_regs *regs)
{
    return close_hook_comm(TRUE,CLOSE_HOOK, IA32_orig_close, regs);
}
asmlinkage long IA32_dup2Hook(const struct pt_regs *regs)
{
    return close_hook_comm(TRUE,DUP2_HOOK, IA32_orig_dup2, regs);
}

asmlinkage long IA32_dup3Hook(const struct pt_regs *regs)
{
    return close_hook_comm(TRUE,DUP3_HOOK, IA32_orig_dup3, regs);
}
asmlinkage long IA32_writeHook(const struct pt_regs *regs)
{
    return write_hook_comm(TRUE,WRITE_HOOK, IA32_orig_write, regs);
}
asmlinkage long IA32_pwriteHook(const struct pt_regs *regs)
{
    return write_hook_comm(TRUE,PWRITE_HOOK, IA32_orig_pwrite, regs);
}

asmlinkage long IA32_pwrite64Hook(const struct pt_regs *regs)
{
    return write_hook_comm(TRUE,PWRITE64_HOOK, IA32_orig_pwrite64, regs);
}
asmlinkage long IA32_writevHook(const struct pt_regs *regs)
{
    return write_hook_comm(TRUE,WRITEV_HOOK, IA32_orig_writev, regs);
}
asmlinkage long IA32_pwritevHook(const struct pt_regs *regs)
{
    return write_hook_comm(TRUE,PWRITEV_HOOK, IA32_orig_pwritev, regs);
}
asmlinkage long IA32_exitHook(const struct pt_regs *regs)
{
    return exit_hook_comm(EXIT_HOOK, IA32_orig_exit, regs);
}
asmlinkage long IA32_exitgroupHook(const struct pt_regs *regs)
{
    return exit_hook_comm(EXIT_GROUP_HOOK, IA32_orig_exit_group, regs);
}
asmlinkage long IA32_umountHook(const struct pt_regs *regs)
{
    return umount_hook_comm(TRUE,UMOUNT_HOOK,IA32_orig_umount, regs);
}
asmlinkage long IA32_umount2Hook(const struct pt_regs *regs)
{
    return umount_hook_comm(TRUE,UMOUNT2_HOOK, IA32_orig_umount2, regs);
}
asmlinkage long IA32_unlinkHook(const struct pt_regs *regs)
{
    return unlink_hook_comm(TRUE,UNLINK_HOOK, IA32_orig_unlink, regs);
}

asmlinkage long IA32_unlinkatHook(const struct pt_regs *regs)
{
    return unlink_hook_comm(TRUE,UNLINKAT_HOOK, IA32_orig_unlinkat, regs);
}

#endif//IA32_HOOK


#else//(HAS_NO_SYSCALL_WRAPPER)

asmlinkage long openHook(const char *filename, int flags, mode_t mode)
{
    return open_hook_comm(FALSE, OPEN_HOOK, orig_open, AT_FDCWD, filename,flags,mode);
}
asmlinkage long openatHook(int dirfd,const char *filename, int flags, mode_t mode)
{
    return open_hook_comm(FALSE, OPENAT_HOOK, orig_openat, dirfd, filename, flags,mode);
}

asmlinkage long closeHook(int fd)
{
    return close_hook_comm(FALSE,CLOSE_HOOK, orig_close, fd, 0, 0);
}
asmlinkage long dup2Hook(int oldfd, int newfd)
{
    return close_hook_comm(FALSE,DUP2_HOOK, orig_dup2, oldfd, newfd, 0);
}
asmlinkage long dup3Hook(int oldfd,int newfd,int flag)
{
    return close_hook_comm(FALSE,DUP3_HOOK, orig_dup3, oldfd, newfd, flag);
}
asmlinkage long writeHook(int fd, const void *buf, size_t count)
{
    return write_hook_comm(FALSE,WRITE_HOOK, orig_write, fd, (void*)buf, count,0);
}

asmlinkage long pwrite64Hook(int fd, const void * buf, size_t count, off_t offset)
{
    return write_hook_comm(FALSE,PWRITE64_HOOK, orig_pwrite64, fd, (void*)buf, count,offset);
}
asmlinkage long writevHook(int fd, const struct iovec *iov, int iovcnt)
{
    return write_hook_comm(FALSE,WRITEV_HOOK, orig_writev, fd, (void*)iov, iovcnt,0);
}
asmlinkage long pwritevHook(int fd, const struct iovec *iov, int iovcnt, off_t offset)
{
    return write_hook_comm(FALSE,PWRITEV_HOOK, orig_pwritev,  fd, (void*)iov, iovcnt, offset);
}
asmlinkage long exitHook(int errcode)
{
    return exit_hook_comm(EXIT_HOOK, orig_exit, errcode);
}
asmlinkage long exitgroupHook(int errcode)
{
    return exit_hook_comm(EXIT_GROUP_HOOK, orig_exit_group, errcode);
}

asmlinkage long umountHook(const char *name)
{
    return umount_hook_comm(FALSE,UMOUNT_HOOK, orig_umount, name,0);
}

asmlinkage long umount2Hook(const char *name, int flags)
{
    return umount_hook_comm(FALSE,UMOUNT2_HOOK, orig_umount2, name,flags);
}

asmlinkage long unlinkHook(const char *pathname)
{
    return unlink_hook_comm(FALSE,UNLINK_HOOK, orig_unlink, 0,pathname,0);
}

asmlinkage long unlinkatHook(int dirfd,const char *pathname, int flags)
{
    return unlink_hook_comm(FALSE,UNLINKAT_HOOK, orig_unlinkat, dirfd,pathname,flags);
}

#ifdef IA32_HOOK//32bit system call in 64bit platform
asmlinkage long IA32_openHook(const char *filename, int flags, mode_t mode)
{
    return open_hook_comm(TRUE,OPEN_HOOK, IA32_orig_open, AT_FDCWD, filename,flags,mode);
}
asmlinkage long IA32_openatHook(int dirfd,const char *filename, int flags, mode_t mode)
{
    return open_hook_comm(TRUE,OPENAT_HOOK, IA32_orig_openat, dirfd, filename, flags,mode);
}
asmlinkage long IA32_closeHook(int fd)
{
    return close_hook_comm(TRUE,CLOSE_HOOK, IA32_orig_close, fd, 0, 0);
}
asmlinkage long IA32_dup2Hook(int oldfd, int newfd)
{
    return close_hook_comm(TRUE,DUP2_HOOK, IA32_orig_dup2, oldfd, newfd, 0);
}
asmlinkage long IA32_dup3Hook(int oldfd,int newfd,int flag)
{
    return close_hook_comm(TRUE,DUP3_HOOK, IA32_orig_dup3, oldfd, newfd, flag);
}
asmlinkage long IA32_writeHook(int fd, const void *buf, size_t count)
{
    return write_hook_comm(TRUE,WRITE_HOOK, IA32_orig_write, fd, (void*)buf, count,0);
}
asmlinkage long IA32_pwriteHook(int fd, const void *buf, size_t count, off_t offset)
{
    return write_hook_comm(TRUE,PWRITE_HOOK, IA32_orig_pwrite, fd, (void*)buf, count, offset);
}
asmlinkage long IA32_pwrite64Hook(int fd, const void * buf, size_t count, off_t offset)
{
    return write_hook_comm(TRUE,PWRITE64_HOOK, IA32_orig_pwrite64, fd, (void*)buf, count,offset);
}
asmlinkage long IA32_writevHook(int fd, const struct iovec *iov, int iovcnt)
{
    return write_hook_comm(TRUE,WRITEV_HOOK, IA32_orig_writev, fd, (void*)iov, iovcnt,0);
}
asmlinkage long IA32_pwritevHook(int fd, const struct iovec *iov, int iovcnt, off_t offset)
{
    return write_hook_comm(TRUE,PWRITEV_HOOK, IA32_orig_pwritev,  fd, (void*)iov, iovcnt, offset);
}

asmlinkage long IA32_exitHook(int errcode)
{
    return exit_hook_comm(EXIT_HOOK, IA32_orig_exit, errcode);
}
asmlinkage long IA32_exitgroupHook(int errcode)
{
    return exit_hook_comm(EXIT_GROUP_HOOK, IA32_orig_exit_group, errcode);
}
asmlinkage long IA32_umountHook(const char *name)
{
    return umount_hook_comm(TRUE,UMOUNT_HOOK, IA32_orig_umount, name,0);
}

asmlinkage long IA32_umount2Hook(const char *name, int flags)
{
    return umount_hook_comm(TRUE,UMOUNT2_HOOK, IA32_orig_umount2, name,flags);
}
asmlinkage long IA32_unlinkHook(const char *pathname)
{
    return unlink_hook_comm(TRUE,UNLINK_HOOK, IA32_orig_unlink, 0,pathname,0);
}

asmlinkage long IA32_unlinkatHook(int dirfd,const char *pathname, int flags)
{
    return unlink_hook_comm(TRUE,UNLINKAT_HOOK, IA32_orig_unlinkat, dirfd,pathname,flags);
}

#endif//IA32_HOOK
#endif//#ifdef CONFIG_ARCH_HAS_SYSCALL_WRAPPER
#endif
