/* Kernel Hook Module for Trend Micro ServerProtect for Linux  */
/* Copyright (C) 2012 Trend Micro Incorporated.                */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/************************Change History*****************************/
/**
** Modify to support SLES kernel version 2.6.27
** Modify Date: 2009/10/30
** Modify By:   errik_zhang
**/

/**
** Modify to Add support for command bypass
** Proc entry: /proc/splx/comm_exc_list
** Modify Date: 2010/01/15
** Modify By: errik_zhang@trendmicro.com
**/

/**
** Modify to resolve the confliction with auditd
** Function ClearAuditContext() To clear audit flag
** Function SetAuditContext() to setback audit flag
** Modify Date: 2010/02/01
**/

/**
** Modify to support kernel version 3.0
** Modify Date: 2012/06/21
** Modify By:   samir_bai@trendmicro.com.cn
**/

/**
** Modify to fix some compile warning caused by strlen(NULL)
** Modify Date: 2013/12/04
** Modify By:   fred_chen@trendmicro.com.cn 
**/

/**
** Modify to support kernel version 4.x
** Modify Date: 2017/05/22
** Modify By:   subin_qu@trendmicro.com.cn
**/

#include    <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)
#include    <linux/config.h> /* retrieve the CONFIG_* macros */
#if defined(CONFIG_SMP)
#define __SMP__
#endif
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS /* force it on */
#endif

#ifdef MODVERSIONS
#include    <linux/modversions.h>
#endif
#endif
#include    <linux/module.h>

#include    <linux/kernel.h>
#include    <linux/unistd.h>



#include    <linux/fs.h>
#include    <linux/mm.h>
#include    <linux/slab.h>
#include    <linux/ptrace.h>
#include    <linux/string.h>
#include    <linux/spinlock.h>
#include    <asm/atomic.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include    <linux/init.h>
#include    <linux/moduleparam.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
#include    <linux/smp_lock.h>
#endif
#include    <linux/sys.h>
#include    <linux/delay.h>
#include    <linux/sched.h>
#ifdef X86_64
#include    <linux/ioctl32.h>
#include    <linux/syscalls.h>
#include    <linux/ioctl32.h>
#include    <linux/compat.h>
#include    <linux/cdev.h>
#endif

#include    <asm/uaccess.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
#include    <linux/uaccess.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
#include    <asm/page.h>
#include    <asm/cacheflush.h>
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
#include <linux/seq_file.h>
#endif

#include        <splxmod.h>

// For RHEL/CentOS and SUSE
#if defined(CONFIG_KALLSYMS)
// For the kernel 3.10.0-862.el7.x86_64 introduced in RHEL/CentOS 7.5 GA, we cannot use the old 
// way to get the address of sys call table, when we access the table, kernel oops. So we have 
// to use another way to get the address of sys call table.
// Refer to https://stackoverflow.com/questions/2103315/linux-kernel-system-call-hooking-example, 
// use kallsyms to get the address of sys call table. Because we cannot get the concrete minor release
// in <linux/version>, we just let all RHEL/CentOS 7 use this new method. In fact, the concrete minor
// release is defined in <generated/uapi/linux/version.h>, I don't want to include this file because
// we don't know what side effect may occur.
#if !defined(CONFIG_SUSE_KERNEL)
#if defined(RHEL_RELEASE_CODE) && defined(RHEL_RELEASE_VERSION)
#if RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(7,0)
#define USE_KALLSYMS
#include <linux/kallsyms.h>
#endif
#endif
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
#define USE_KALLSYMS
#include <linux/kallsyms.h>
#endif
#endif
#endif

#ifdef  NFSD
#include    <linux/sunrpc/svc.h>
#include    <linux/nfsd/nfsd.h>
#define __NR_nfsd_open  1
#define __NR_nfsd_close 2
#endif

#ifdef IA32_HOOK
#include    <splx_ia32_unistd.h>
#endif

#include "hooks.h"
#include "hook_lsm.h"

// add by errik for /proc
#include    <linux/proc_fs.h>

//PDE_DATA() replaced by pde_data()in linux kernel version 5.17.
//https://github.com/openzfs/zfs/issues/13004
//https://lore.kernel.org/all/20211124081956.87711-2-songmuchun@bytedance.com/T/#u
#if defined(RHEL_RELEASE_CODE)&& defined(RHEL_RELEASE_VERSION)
    #if LINUX_VERSION_CODE >= KERNEL_VERSION(5,14,0)&& RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(9,1)
        #define PDE_DATA(i) pde_data(i)
    #elif LINUX_VERSION_CODE >= KERNEL_VERSION(5,17,0)
	    #define PDE_DATA(i) pde_data(i)
    #endif
#endif

/***********************************************************************************************************************
                                                Function Declare
************************************************************************************************************************/
#define PROC_STR_SIZE     8

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0)
int khm_read_proc(char *, char **, off_t , int , int *, void *);
int khm_write_proc(struct file *, const char __user *, unsigned long , void *);
#else
int khm_proc_show(struct seq_file *m, void *v);
ssize_t khm_write_proc(struct file *, const char __user *, size_t, loff_t*);
#endif

void unhook_module(void);
void hook_module(void);
// add end

// Proc entry read/write function.
// 150 for at least 10 command name.
#define PROC_COMMS_LENGTH 150
char g_exc_comms[PROC_COMMS_LENGTH+1]={0};
char exc_comms[PROC_COMMS_LENGTH+1]={0};

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0)
int khm_comms_read_proc(char *, char **, off_t, int, int*, void *);
int khm_comms_write_proc(struct file *, const char __user *, unsigned long , void *);
#else
int khm_comms_proc_show(struct seq_file *m, void *v);
ssize_t khm_comms_write_proc(struct file *, const char __user *, size_t, loff_t*);
#endif

extern void parseAddExcComms(const char *comms);
extern void PrintExcComms(void);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)

static int proc_khm_open(struct inode *inode, struct file *file)
{
   return single_open(file, khm_proc_show, PDE_DATA(inode));
}

static int proc_khm_comms_open(struct inode *inode, struct file *file)
{ 
   return single_open(file, khm_comms_proc_show, PDE_DATA(inode));
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,6,0)
// proc_ops operation
static const struct proc_ops proc_khm_fops = 
{
    .proc_open       = proc_khm_open,
    .proc_read       = seq_read,
    .proc_write      = khm_write_proc,
    .proc_lseek      = seq_lseek,
    .proc_release    = single_release,
};

static const struct proc_ops proc_khm_comms_fops = 
{
    .proc_open       = proc_khm_comms_open,
    .proc_read       = seq_read,
    .proc_write      = khm_comms_write_proc,
    .proc_lseek     = seq_lseek,
    .proc_release    = single_release,
};
#else
//file_operations version
static const struct file_operations proc_khm_fops = 
{
    .owner      = THIS_MODULE,
    .open       = proc_khm_open,
    .read       = seq_read,
    .write      = khm_write_proc,
    .llseek     = seq_lseek,
    .release    = single_release,
};

static const struct file_operations proc_khm_comms_fops = 
{
    .owner      = THIS_MODULE,
    .open       = proc_khm_comms_open,
    .read       = seq_read,
    .write      = khm_comms_write_proc,
    .llseek     = seq_lseek,
    .release    = single_release,
};
#endif
#endif
extern rwlock_t hook_init_lock;
// Add end
extern asmlinkage int sys_execve_glue_hook(struct pt_regs);
#ifdef X86_64
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
extern long ioctlMod(struct file *, unsigned int , unsigned long);
extern long splxmod_compat_ioctl(struct file *, unsigned int , unsigned long);
#else
extern long sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);
extern int ioctlMod(struct inode *, struct file *, unsigned int, unsigned long);
#endif
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
extern int ioctlMod(struct inode *, struct file *, unsigned int, unsigned long);
#else
extern long ioctlMod(struct file *, unsigned int, unsigned long);
#endif

extern int openMod(struct inode *, struct file *);
extern int releaseMod(struct inode *, struct file *);
extern long busy_timeout_HZ;
extern long scan_timeout_HZ;
extern int hook_init;
extern pid_t  *exc_pid_ary;
extern pid_t  *vsc_pid_ary;

extern void parseAddDirs(char *);
extern void parseAddExts(char *);
extern void parseAddExcDirs(char *);
extern void parseAddExcFils(char *);
extern void parseAddExcExts(char *);
extern int REGISTER_CHRDEV(unsigned int, const char *, struct file_operations *);
extern Boolean deleteListAll(void);
extern void delDirList(void);
extern void delExtList(void);
extern void delExcDirList(void);
extern void delExcFilList(void);
extern void delExcExtList(void);
extern void removeCacheAll(void);
extern void delDenyWriteList(DENYWRITE_TYPE type);
extern void delExcCommList(void);

/***************************************************************************************************************
                                              Global Variable Declare
****************************************************************************************************************/


extern atomic_t ref_cnt;

//SLES11 support
unsigned long orig_cr0;
//End
unsigned int major;
int splxmod_debug = 0;
char    *splxmod_addr = NULL;
int g_iDbgLevel = 0;

char *splxmod_execve_addr = NULL;

#ifdef IA32_HOOK
char *splxmod_ia32_addr = NULL;
char *splxmod_compat_do_execve_addr = NULL;
#endif
char *splxmod_ret_addr = NULL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
module_param(splxmod_debug, int, 0);
module_param(splxmod_addr, charp, 0);
module_param(splxmod_execve_addr, charp, 0);
#ifdef X86_64
module_param(splxmod_compat_do_execve_addr, charp, 0);
module_param(splxmod_ia32_addr, charp, 0);
module_param(splxmod_ret_addr, charp, 0);
#endif
#else
MODULE_PARM(splxmod_debug, "i");
MODULE_PARM(splxmod_addr, "s");
MODULE_PARM(splxmod_execve_addr, "s");
#ifdef X86_64
MODULE_PARM(splxmod_compat_do_execve_addr, "s");
MODULE_PARM(splxmod_ia32_addr, "s");
MODULE_PARM(splxmod_ret_addr, "s");
#endif
#endif
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16) && defined(X86_64)
struct file_operations splxmod_fops =
{
    .owner              =    THIS_MODULE,
    .unlocked_ioctl     =    ioctlMod,
    .compat_ioctl       =    splxmod_compat_ioctl,
    .open               =    openMod,
    .release            =    releaseMod,
};
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
struct file_operations splxmod_fops =
{
    .owner              =    THIS_MODULE,
    .unlocked_ioctl     =    ioctlMod,
    .open               =    openMod,
    .release            =    releaseMod,
};
#else
struct file_operations splxmod_fops =
{
    .owner              =    THIS_MODULE,
    .ioctl              =    ioctlMod,
    .open               =    openMod,
    .release            =    releaseMod,
};
#endif

INIT_ARGS kini =
{
    .incoming           =      INCOMING_DEF,
    .outgoing           =      OUTGOING_DEF,
    .running            =      RUNNING_DEF,
    .dirs               =      DIRS_DEF,
    .exts               =      EXTS_DEF,
    .exc_dirs           =      EXC_DIRS_DEF,
    .exc_fils           =      EXC_FILS_DEF,
    .exc_exts           =      EXC_EXTS_DEF,
    .debug_level        =      DEBUG_LEVEL_DEF,
    .max_cache_item     =      MAX_CACHE_ITEM_DEF,
    .max_list_item      =      MAX_LIST_ITEM_DEF,
    .max_dir_item       =      MAX_DIR_ITEM_DEF,
    .max_ext_item       =      MAX_EXT_ITEM_DEF,
    .max_exc_dir_item   =      MAX_EXC_DIR_ITEM_DEF,
    .max_exc_fil_item   =      MAX_EXC_FIL_ITEM_DEF,
    .max_exc_ext_item   =      MAX_EXC_EXT_ITEM_DEF,
    .waitq_timeout      =      WAITQ_TIMEOUT_DEF,
    .vsapi_timeout      =      VSAPI_TIMEOUT_DEF,
    .max_exc_pid        =      MAX_EXC_PID_DEF,
    .max_vsc_pid        =      MAX_VSC_PID_DEF,
    .max_path_len       =      MAX_PATH_LEN_DEF,
    .max_cmd_len        =      MAX_CMD_LEN_DEF
};

HOOK_FLAGS hook_flag =
{
    .open_hooked        =      0,
    .close_hooked       =      0,
    .execve_hooked      =      0
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
rwlock_t kini_lock __cacheline_aligned = RW_LOCK_UNLOCKED;
spinlock_t dbg_lock = SPIN_LOCK_UNLOCKED;
//rwlock to protect the command list
rwlock_t comm_list_lock __cacheline_aligned = RW_LOCK_UNLOCKED;
#else
DEFINE_RWLOCK(kini_lock);
DEFINE_SPINLOCK(dbg_lock);
DEFINE_RWLOCK(comm_list_lock);
#endif

void **p_sys_call_table = NULL;
#ifdef IA32_HOOK
void **p_ia32_sys_call_table = NULL;
void **p_sys32_execve = NULL;
#endif
void **p_do_execve = NULL;
void **p_int_ret_from_sys_call = NULL;


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)

#ifdef X86_64
asmlinkage long (*splx_change_page_attr_addr)(unsigned long address, int numpages, pgprot_t prot);
void **orig_p_sys_call_table = NULL;
#ifdef IA32_HOOK
void **orig_p_ia32_sys_call_table = NULL;
#endif
#define virt_addr_valid_symbol(kaddr)   pfn_valid(__pa_symbol(kaddr) >> PAGE_SHIFT)
#define virt_to_page_symbol(kaddr)  pfn_to_page(__pa_symbol(kaddr) >> PAGE_SHIFT)
#define PHYS_BASE 0xffffffff803097f8
#define CHANGE_PAGE_ATTR_ADDR 0xffffffff8027d5b7
#endif


/***********************************************************************************************************************
                                                Function Definition
************************************************************************************************************************/

/*
* clear WP bit of CR0, and return the original value
* A new way to modify the attribute of memory address
* Only apply on SLES11
*/

unsigned long clear_and_return_cr0(void)
{
    unsigned long cr0 = 0;
    unsigned long ret;
#ifdef X86_64
    __asm__ __volatile("movq %%cr0, %%rax" : "=a"(cr0));
    ret = cr0;
    cr0 &= 0xfffffffffffeffff;
    __asm__ __volatile__("movq %%rax, %%cr0" : : "a"(cr0));
#else
    __asm__ __volatile("mov %%cr0, %%eax" : "=a"(cr0));
    ret = cr0;
    cr0 &= 0xfffeffff;
    __asm__ __volatile__("mov %%eax, %%cr0" : : "a"(cr0));

#endif
    return ret;
}

/** set CR0 with new value
*
* @val : new value to set in cr0

*/
void setback_cr0(unsigned long val)
{
#ifdef X86_64
    __asm__ __volatile__("movq %%rax, %%cr0" : : "a"(val));
#else
    __asm__ __volatile__("mov %%eax, %%cr0" : : "a"(val));
#endif
}
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
void change_page_to_rw(void **p_addr)
{
#ifdef X86_64
    unsigned long addr;
#ifndef CONFIG_XEN
    unsigned long phys_base;
    unsigned long *p_phys_base ;

    p_phys_base = (unsigned long * ) _AC(PHYS_BASE, UL);

    phys_base = *p_phys_base;
#endif
    splx_change_page_attr_addr = (void *)_AC(CHANGE_PAGE_ATTR_ADDR, UL);
    addr = (unsigned long)__va(__pa_symbol(p_addr));
    p_addr = (void **)addr;
    splx_change_page_attr_addr(addr, 1, PAGE_KERNEL);
    global_flush_tlb();
#else

    change_page_attr(virt_to_page(p_addr), 1, PAGE_KERNEL);
    global_flush_tlb();
#endif
}

void change_page_to_ro(void **p_addr)
{
#ifdef X86_64
    unsigned long addr;
#ifndef CONFIG_XEN

    unsigned long phys_base;
    unsigned long *p_phys_base ;

    p_phys_base = (unsigned long *)_AC(PHYS_BASE, UL);
    phys_base = *p_phys_base;
#endif
    splx_change_page_attr_addr = (void *)_AC(CHANGE_PAGE_ATTR_ADDR, UL);
    addr = (unsigned long)__va(__pa_symbol(p_addr));
    splx_change_page_attr_addr(addr, 1, PAGE_KERNEL_RO);
    global_flush_tlb();
    p_addr = (void **)addr;
#else
    change_page_attr(virt_to_page(p_addr), 1, PAGE_KERNEL_RO);
    global_flush_tlb();
#endif
}
#endif
//End

//modify by derrick for support suse12
//The function enabled on RHEL6 and suse12
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,32) && !defined(CONFIG_SUSE_KERNEL)
#define USE_SPLX_SET_MEMORY_ARRT
#elif defined(CONFIG_SUSE_KERNEL) && LINUX_VERSION_CODE >= KERNEL_VERSION(3,12,28)
#define USE_SPLX_SET_MEMORY_ARRT
#endif 

//The function only enabled on RHEL6
#if defined(USE_SPLX_SET_MEMORY_ARRT)
#include <asm/tlbflush.h>

static int is_table_RW(unsigned long address)
{
    unsigned int level;
    pte_t *kpte, cur_pte;
    pgprot_t cur_prot;
    address = (unsigned long)(address) & PAGE_MASK;
    kpte = lookup_address(address, &level);

    if (!kpte)
    {
        return 0;
    }
    
    cur_pte = *kpte;
    cur_prot = pte_pgprot(cur_pte);
    if ( (pgprot_val(cur_prot) & pgprot_val(__pgprot(_PAGE_RW))) )
    {
        return 1;
    }
    return 0;
}

static int splx_set_memory_attr(unsigned long address, bool b_rw)
{
    unsigned int level;
    pte_t *kpte, old_pte, new_pte;
    pgprot_t new_prot;
    unsigned long pfn;
    int i = 0;
    //Set two pages
    for (; i < 2; i++)
    {
        address += i * PAGE_SIZE;
        address = (unsigned long)(address) & PAGE_MASK;
        kpte = lookup_address(address, &level);

        if (!kpte)
        {
            return 1;
        }
        old_pte = *kpte;

        new_prot = pte_pgprot(old_pte);
        pfn = pte_pfn(old_pte);
        if (b_rw)
            pgprot_val(new_prot) |= pgprot_val(__pgprot(_PAGE_RW));
        else
            pgprot_val(new_prot) &= ~pgprot_val(__pgprot(_PAGE_RW));

        new_pte = pfn_pte(pfn, canon_pgprot(new_prot));
        set_pte_atomic(kpte, new_pte);

    }
    __flush_tlb_all();
    return 0;
}

static int splx_set_syscall_attr_rw(void)
{
    const char *cpsMethod = "splx_set_syscall_attr";
    CP_DBG_LVL;
    if (splx_set_memory_attr((unsigned long)p_sys_call_table, true))
    {
        DPRINTK(0, "%s: Failed to change the attribute of syscall table, please contact TrendMicro\n", cpsMethod);
        return 1;
    }
#ifdef IA32_HOOK
    if (splx_set_memory_attr((unsigned long)p_ia32_sys_call_table, true))
    {
        DPRINTK(0, "%s: Failed to change the attribute of IA32 syscall table, please contact TrendMicro\n", cpsMethod);
        splx_set_memory_attr((unsigned long)p_sys_call_table, false);
        return 1;
    }
#endif
    return 0;
}

static void splx_set_syscall_attr_ro(void)
{
    const char *cpsMethod = "splx_set_syscall_attr_ro";
    CP_DBG_LVL;
    if (splx_set_memory_attr((unsigned long)p_sys_call_table, false))
    {
        DPRINTK(0, "%s: Failed to change the attribute of syscall table, please contact TrendMicro\n", cpsMethod);
    }
#ifdef IA32_HOOK
    if (splx_set_memory_attr((unsigned long)p_ia32_sys_call_table, false))
    {
        DPRINTK(0, "%s: Failed to change the attribute of IA32 syscall table, please contact TrendMicro\n", cpsMethod);
    }
#endif
}

#endif
//End
/*
** Main entry to change page attribute
** For SLES11, call the kernel API
** SuSE kernel bug 439348
** The API will fail on 2.6.27.19-5
*/
void change_page_to_RW(void **p_addr)
{
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
    change_page_to_rw(p_addr);
    //SuSE 11
#elif defined(CONFIG_SUSE_KERNEL) && !defined(USE_SPLX_SET_MEMORY_ARRT)
    mark_rodata_rw();
    //RHEL6
#else
    return;
#endif
}

void change_page_to_RO(void **p_addr)
{
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
    change_page_to_ro(p_addr);
#elif defined(CONFIG_SUSE_KERNEL) && !defined(USE_SPLX_SET_MEMORY_ARRT)
    mark_rodata_ro();
#else
    return;
#endif
}

#else
//Empty function for kernel below 2.6.18 no need to modify
//page attributes
inline void change_page_to_RW(void **p_addr)
{

}
inline void change_page_to_RO(void **p_addr)
{

}

#endif
// End of #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
/**
*** add for dynamically enable debug log
*** khm_read_proc
*** khm_write_proc
*** Date: 2008.10.25
**/

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
static DECLARE_MUTEX(mutex);
#else
static DEFINE_SEMAPHORE(mutex);
#endif

int my_snprintf(char *str, size_t size, const char *fmt, ...)
{
    int len;

    va_list ap;
    va_start(ap, fmt);
    len = vsnprintf(str, size, fmt, ap);
    va_end(ap);
    return (len <= size) ? len : size;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
int khm_comms_proc_show(struct seq_file *m, void *v)
{
    const char* cpsMethod = "khm_comms_proc_show";
    CP_DBG_LVL;
    DPRINTK(LOG_COMMON, "%s: Get into read proc khm comms:\n", cpsMethod);
    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%s: mutex was interrupted\n",cpsMethod);
        return -ERESTARTSYS;
    }
    seq_printf(m, "%s\n", g_exc_comms);
    up(&mutex);
    DPRINTK(LOG_COMMON, "%s: Get out of read proc khm comms:\n", cpsMethod);
    return 0;
}
#else
int khm_comms_read_proc(char *page, char **start, off_t off, int count,
                        int *eof, void *data)
{
    int len;
    const char *cpsMethod = "khm_comms_read_proc";
    char *p = page;
    char *p_tail = page + count;
    CP_DBG_LVL;

    DPRINTK(LOG_COMMON, "%s: Get into read proc debug level:\n", cpsMethod);

    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%s: mutex was interrupted\n", cpsMethod);
        return -ERESTARTSYS;
    }

    p += my_snprintf(p, p_tail - p, "%s\n", g_exc_comms);

    up(&mutex);

    len = (p - page) - off;
    if (len < 0)
        len = 0;
    *eof = (len <= count) ? 1 : 0;
    *start = page + off;
    DPRINTK(LOG_COMMON, "%s: Get out of read prco:\n", cpsMethod);
    return len;
}

#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)
int khm_comms_write_proc(struct file *file, const char __user *userbuf,
        unsigned long count, void *data)
#else
ssize_t khm_comms_write_proc(struct file *file, const char __user *userbuf
        , size_t count, loff_t* offset)
#endif
{
    const char *cpsMethod = "khm_comms_write_proc";
    char proc_str[PROC_COMMS_LENGTH + 1] = {0};
    pid_t cur_pid = current->pid;
    CP_DBG_LVL;

    DPRINTK(LOG_COMMON, "%s: Get into write proc\n", cpsMethod);

    if (count > PROC_COMMS_LENGTH)
    {
        DPRINTK(LOG_WARNING, "WARNING: %d:%s: User's input length exceeds the max length %d\n", cur_pid, cpsMethod, PROC_COMMS_LENGTH);
        return -EINVAL;
    }
    if (count == 0)
    {
        DPRINTK(LOG_COMMON, "WARNING: %d:%s: Clear the command exclusion list\n", cur_pid, cpsMethod);
    }
    else if (copy_from_user(proc_str, userbuf, count))
    {
        DPRINTK(LOG_WARNING, "WARNING:%d:%s: Unable to copy date from user\n", cur_pid, cpsMethod);
        return -EPERM;
    }
    proc_str[count] = '\0';
    if (count > 0 && proc_str[count - 1] == '\n')
        proc_str[count - 1 ] = '\0';
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0)
    DPRINTK(LOG_WARNING, "%d:%s: Write %s to %s\n", cur_pid, cpsMethod, proc_str, file->f_dentry->d_iname);
#else
    DPRINTK(LOG_WARNING, "%d:%s: Write %s to %s\n", cur_pid, cpsMethod, proc_str, file->f_path.dentry->d_iname);
#endif
    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%d: %s: mutex was interrupted\n", cur_pid, cpsMethod);
        return -ERESTARTSYS;
    }
    strncpy(g_exc_comms, proc_str, strlen(proc_str));
    g_exc_comms[strlen(proc_str)] = '\0';
    up(&mutex);
    //Initial the command list
    strncpy(exc_comms, proc_str, strlen(proc_str));
    exc_comms[strlen(proc_str)] = '\0';
    parseAddExcComms(exc_comms);
    //PrintExcComms();
    DPRINTK(LOG_COMMON, "%s: Get out of write proc\n", cpsMethod);
    return count;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
int khm_proc_show(struct seq_file *m, void *v)
{
    const char* cpsMethod = "khm_proc_show";
    CP_DBG_LVL;
    DPRINTK(LOG_COMMON, "%s: Get into read prco:\n", cpsMethod);
    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%s: mutex was interrupted\n",cpsMethod);
        return -ERESTARTSYS;
    }
    seq_printf(m, "%d\n", g_iDbgLevel);
    up(&mutex);

    return 0;
}
#else
int khm_read_proc(char *page, char **start, off_t off, int count,
                  int *eof, void *data)
{
    int len;
    const char *cpsMethod = "khm_read_proc";
    char *p = page;
    char *p_tail = page + count;
    CP_DBG_LVL;

    DPRINTK(LOG_COMMON, "%s: Get into read prco:\n", cpsMethod);

    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%s: mutex was interrupted\n", cpsMethod);
        return -ERESTARTSYS;
    }

    p += my_snprintf(p, p_tail - p, "%d\n", g_iDbgLevel);

    up(&mutex);

    len = (p - page) - off;
    if (len < 0)
        len = 0;
    *eof = (len <= count) ? 1 : 0;
    *start = page + off;
    DPRINTK(LOG_COMMON, "%s: Get out of read prco:\n", cpsMethod);
    return len;
}
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0)
int khm_write_proc(struct file *file, const char __user *userbuf,
        unsigned long count, void *data)
#else
ssize_t khm_write_proc(struct file * file, const char __user *userbuf
       , size_t count, loff_t* offset)
#endif
{
    const char *cpsMethod = "khm_write_prco";
    char proc_str[PROC_STR_SIZE] = "";
    pid_t cur_pid = current->pid;
    int val;
    CP_DBG_LVL;

    DPRINTK(LOG_COMMON, "%s: Get into write proc\n", cpsMethod);

    if (count > sizeof(proc_str) - 1)
    {
        DPRINTK(LOG_WARNING, "WARNING:%d:%s: User's input length exceeds %d\n", cur_pid, cpsMethod, PROC_STR_SIZE);
        return -EINVAL;
    }

    if (copy_from_user(proc_str, userbuf, count))
    {
        DPRINTK(LOG_WARNING, "WARNING:%d:%s: Unable to copy date from user\n", cur_pid, cpsMethod);
        return -EPERM;
    }

    proc_str[count] = '\0';
    if (count > 0 && proc_str[count - 1] == '\n')
        proc_str[count - 1 ] = '\0';
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0)
    DPRINTK(LOG_WARNING, "%d:%s: Write %s to %s\n", cur_pid, cpsMethod, proc_str, file->f_dentry->d_iname);
#else
    DPRINTK(LOG_DEBUG, "%d:%s: Write %s to %s\n", cur_pid, cpsMethod, proc_str, file->f_path.dentry->d_iname);
#endif

    if (sscanf(proc_str, "%d", &val) != 1)
    {
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0)
		DPRINTK(LOG_WARNING, "%d:%s: Write %s to %s\n", cur_pid, cpsMethod, proc_str, file->f_dentry->d_iname);
#else
        DPRINTK(LOG_WARNING, "WARNING:%d:%s Invalid value %s for %s\n", cur_pid, cpsMethod, proc_str, file->f_path.dentry->d_iname);
#endif
        return -EINVAL;
    }
    if (down_interruptible(&mutex))
    {
        DPRINTK(LOG_WARNING, "WARNING:%d: %s: mutex was interrupted\n", cur_pid, cpsMethod);
        return -ERESTARTSYS;
    }
    if (val > 3)
        g_iDbgLevel = 3;
    else if (val < 0)
        g_iDbgLevel = 0;
    else g_iDbgLevel = val;
    up(&mutex);
    DPRINTK(LOG_COMMON, "%s: Get out of write proc\n", cpsMethod);
    return count;
}
//add end

/* 2010/10/28
** To make code easily read, re-construct the code for hooking
*/

#define HOOK_SYSCALL(index, orig_sys_call, wrapper) \
    do { \
        if(p_sys_call_table[index]) \
        { \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18))\
            { change_page_to_RW(&(p_sys_call_table[index]));}\
            orig_sys_call = p_sys_call_table[index];\
            p_sys_call_table[index] = wrapper;\
            DPRINTK(1, "hook: hooked %s\n", #index);\
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18))\
            {change_page_to_RO(&(p_sys_call_table[index]));} \
        } \
        else \
        {\
            DPRINTK(1, "hook: warning: didn't hook %s \n", #index); \
        } \
    }while(0);

#ifdef IA32_HOOK

#define HOOK_SYSCALL_IA32(index, orig_sys_call, wrapper) \
    do{ \
        if(p_ia32_sys_call_table[index]) \
        { \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RW(&(p_ia32_sys_call_table[index]));} \
            orig_sys_call = p_ia32_sys_call_table[index]; \
            p_ia32_sys_call_table[index] = wrapper; \
            DPRINTK(1, "IA32 hook: hooked %s\n", #orig_sys_call); \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RO(&(p_ia32_sys_call_table[index]));} \
        }\
        else \
        {\
            DPRINTK(1, "IA32 hook: warning: didn't hook %s \n", #orig_sys_call); \
        }\
    }while(0);

#endif

#define UNHOOK_SYSCALL(index, orig_sys_call, wrapper) \
    do{ \
        if (orig_sys_call != NULL) \
        { \
            if (p_sys_call_table[index] != wrapper) \
            { \
                DPRINTK(1,"unhook_module: warning: Somebody else also played with the %s\n", #orig_sys_call); \
            } \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RW(&(p_sys_call_table[index]));}    \
            p_sys_call_table[index] = orig_sys_call; \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RO(&(p_sys_call_table[index]));} \
        }\
    }while(0);

#ifdef IA32_HOOK
#define UNHOOK_SYSCALL_IA32(index, orig_sys_call, wrapper) \
    do{\
        if (orig_sys_call != NULL) \
        { \
            if (p_ia32_sys_call_table[index] != wrapper) \
            { \
                DPRINTK(1,"unhook_module: warning: Somebody else also played with the %s\n", #orig_sys_call); \
            } \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RW(&(p_ia32_sys_call_table[index]));} \
            p_ia32_sys_call_table[index] = orig_sys_call; \
            if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)) \
            {change_page_to_RO(&(p_ia32_sys_call_table[index]));} \
        }\
    }while(0);
#endif
//2010/10/28 End


//TT224111: Security hook_module

void security_hook(void)
{
    write_lock(&hook_init_lock);
    if (hook_init != UN_HOOKED)
    {
        write_unlock(&hook_init_lock);
        return;
    }

    hook_init = IN_HOOK;
    write_unlock(&hook_init_lock);

    //Never add spin lock when hook
    hook_module();

    write_lock(&hook_init_lock);
    hook_init = HOOKED;
    write_unlock(&hook_init_lock);
}

void hook_module(void)
{
    const char *cpsMethod = "hook_module";
    pid_t cur_pid = 0;
#if defined(USE_SPLX_SET_MEMORY_ARRT)
    int bPageRW = 0;
#endif
    CP_DBG_LVL;

    cur_pid = current->pid;

    memset(&hook_flag, 0, sizeof(HOOK_FLAGS));
    DPRINTK(1, "%d: %s: get into hook_module\n", cur_pid, cpsMethod);

    //2011.3.16   For RHEL6, first change the syscall table attribute
#if defined(USE_SPLX_SET_MEMORY_ARRT)
    bPageRW = is_table_RW((unsigned long)p_sys_call_table);
    if (!bPageRW && splx_set_syscall_attr_rw())
    {
        return;
    }
#endif
    //End
    if (kini.outgoing)
    {
		HOOK_SYSCALL(__NR_open, orig_open, openHook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
		HOOK_SYSCALL(__NR_openat, orig_openat, openatHook);
        #endif
        hook_flag.open_hooked = 1;
    }
    if (kini.incoming)
    {
		HOOK_SYSCALL(__NR_close, orig_close, closeHook);
        
        HOOK_SYSCALL(__NR_dup2, orig_dup2, dup2Hook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,27)
        HOOK_SYSCALL(__NR_dup3, orig_dup3, dup3Hook);
        #endif
        
        hook_flag.close_hooked = 1;
    }

    HOOK_SYSCALL(__NR_exit, orig_exit, exitHook); 

    /*
     * For 2.6.32 64 bits no_xen kernel, use LSM to do the execve hook
     */
    if (kini.running)
    {
#ifdef USE_LSM_HOOK
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,2,0)
		if (hook_lsm())
#else
        if (hook_lsm_new())
#endif
        {
            hook_flag.execve_hooked = 1;
        }
#else
        HOOK_SYSCALL(__NR_execve, orig_execve, execveHook);
        hook_flag.execve_hooked = 1;
#endif

    }

    /*get getpid*/
    orig_getpgid = p_sys_call_table[__NR_getpgid];

    /*just for hook 32bit systemcall on x86_64*/
#ifdef IA32_HOOK

    if (kini.outgoing)
    {
        HOOK_SYSCALL_IA32(__NR_ia32_open, IA32_orig_open, IA32_openHook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
		HOOK_SYSCALL_IA32(__NR_ia32_openat, IA32_orig_openat, IA32_openatHook);
		#endif
    }
    if (kini.incoming)
    {
        HOOK_SYSCALL_IA32(__NR_ia32_close, IA32_orig_close, IA32_closeHook);
        
        HOOK_SYSCALL_IA32(__NR_ia32_dup2, IA32_orig_dup2, IA32_dup2Hook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,27)
        HOOK_SYSCALL_IA32(__NR_ia32_dup3, IA32_orig_dup3, IA32_dup3Hook);
        #endif
        
    }

    HOOK_SYSCALL_IA32(__NR_ia32_exit, IA32_orig_exit, IA32_exitHook);
   
    if (kini.running)
    {
#ifndef USE_LSM_HOOK
        HOOK_SYSCALL_IA32(__NR_ia32_execve, IA32_orig_execve, stub32_execveHook);
#endif
    }
#endif

#if defined(USE_SPLX_SET_MEMORY_ARRT)
    if (!bPageRW)
    {
        splx_set_syscall_attr_ro();
    }
#endif

    DPRINTK(1, "%d: %s: get outta hook_module, hook open [%d], hook close [%d], hook execve [%d]\n", cur_pid, cpsMethod
            , hook_flag.open_hooked, hook_flag.close_hooked, hook_flag.execve_hooked);
}

void security_unhook(void)
{
    write_lock(&hook_init_lock);
    if (hook_init != HOOKED)
    {
        write_unlock(&hook_init_lock);
        return;
    }

    hook_init = IN_HOOK;
    write_unlock(&hook_init_lock);

    //Never add spin lock when hook/unhook
    unhook_module();

    write_lock(&hook_init_lock);
    hook_init = UN_HOOKED;
    write_unlock(&hook_init_lock);
}

void unhook_module(void)
{
    const char *cpsMethod = "unhook_module";
#if defined(USE_SPLX_SET_MEMORY_ARRT)
    int bPageRW = 0;
#endif
    pid_t cur_pid = 0;
    CP_DBG_LVL;

    cur_pid = current->pid;
    DPRINTK(1, "%d: %s: get into unhook_module\n", cur_pid, cpsMethod);
#if defined(USE_SPLX_SET_MEMORY_ARRT)
    bPageRW = is_table_RW((unsigned long)p_sys_call_table);
    if (!bPageRW && splx_set_syscall_attr_rw())
    {
        DPRINTK(0, "[FATAL]%d: %s: splx_set_syscall_attr_rw failed, can't do unhook\n", cur_pid, cpsMethod);
        return;
    }
#endif
    if (hook_flag.open_hooked)
    {
        UNHOOK_SYSCALL(__NR_open, orig_open, openHook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
		UNHOOK_SYSCALL(__NR_openat, orig_openat, openatHook);
        #endif
    }
    if (hook_flag.close_hooked)
    {
        UNHOOK_SYSCALL(__NR_close, orig_close, closeHook);
       
        UNHOOK_SYSCALL(__NR_dup2, orig_dup2, dup2Hook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,27)
        UNHOOK_SYSCALL(__NR_dup3, orig_dup3, dup3Hook);
        #endif
    }

    UNHOOK_SYSCALL(__NR_exit, orig_exit, exitHook);

    /*
     * Kernel > 2.6.32 with Arch x86_64, use LSM to do execve Hook
     */
    if (hook_flag.execve_hooked)
    {
#ifdef USE_LSM_HOOK
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,2,0)
        unhook_lsm();
#else
		unhook_lsm_new();
#endif
#else
        UNHOOK_SYSCALL(__NR_execve, orig_execve, execveHook);
#endif
    }
    /*just for unhook 32bit systemcall on x86_64*/
#ifdef IA32_HOOK

    if (hook_flag.open_hooked)
    {
        UNHOOK_SYSCALL_IA32(__NR_ia32_open, IA32_orig_open, IA32_openHook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
        UNHOOK_SYSCALL_IA32(__NR_ia32_openat, IA32_orig_openat, IA32_openatHook);
        #endif
    }
    if (hook_flag.close_hooked)
    {
        UNHOOK_SYSCALL_IA32(__NR_ia32_close, IA32_orig_close, IA32_closeHook);
        
        UNHOOK_SYSCALL_IA32(__NR_ia32_dup2, IA32_orig_dup2, IA32_dup2Hook);
        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,27)
        UNHOOK_SYSCALL_IA32(__NR_ia32_dup3, IA32_orig_dup3, IA32_dup3Hook);
        #endif 
    }
    
    
    UNHOOK_SYSCALL_IA32(__NR_ia32_exit, IA32_orig_exit, IA32_exitHook);
 
    if (hook_flag.execve_hooked)
    {
#ifndef USE_LSM_HOOK
        UNHOOK_SYSCALL_IA32(__NR_ia32_execve, IA32_orig_execve, stub32_execveHook);
#endif
    }
#endif
#if defined(USE_SPLX_SET_MEMORY_ARRT)
    if (!bPageRW)
    {
        splx_set_syscall_attr_ro();
    }
#endif

    DPRINTK(1, "%d: %s: get outta unhook_module\n", cur_pid, cpsMethod);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
int can_unload(void)
{
    DECLARE_WAIT_QUEUE_HEAD(cleanup_wq);
    Boolean count;

    count = GET_USE_COUNT(THIS_MODULE) ||
            atomic_read(&ref_cnt);

    return (count ? -EBUSY : 0);
}
#endif

#ifdef X86_64
#include "ioctl_trans_x86_64.c"
#endif
//Add proc entry for khm debug log
static struct proc_dir_entry *proc_splx_base;
char *exc_cmds = NULL;

static int __init splxmod_init(void)
{
    const char *cpsMethod = "init_module";
    pid_t cur_pid = 0;
    struct proc_dir_entry *entry;

    char *execve_strend;
    char *tmp = NULL;
    int len = 0;
    //change it to the global, free it at splxmod_exit 
    //char *exc_cmds;
#ifdef X86_64
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,6,0)
    char *ret_strend;
#endif
#endif
#ifdef IA32_HOOK
    char *sys32_execve_strend;
#endif
    CP_DBG_LVL;

    if (splxmod_debug > 0)
    {
        g_iDbgLevel = splxmod_debug;
        l_iDbgLevel = splxmod_debug;
    }

    cur_pid = current->pid;
    DPRINTK(1, "%d: %s: get into init_module\n", cur_pid, cpsMethod);

    if (!splxmod_addr)
        return -EFAULT;

#ifdef KPROBE_LOOKUP
    typedef unsigned long (*kallsyms_lookup_name_t)(const char *name);
    kallsyms_lookup_name_t kallsyms_lookup_name;
    register_kprobe(&kp);
    kallsyms_lookup_name = (kallsyms_lookup_name_t) kp.addr;
    unregister_kprobe(&kp);
#endif

#ifdef USE_KALLSYMS
	p_sys_call_table = (void **) kallsyms_lookup_name("sys_call_table");
#else
	char *strend;
    p_sys_call_table = (void **) simple_strtoul(splxmod_addr, &strend, 0);
#endif
    if (p_sys_call_table == 0)
        return -EFAULT;

#ifdef IA32_HOOK


    if (!splxmod_ia32_addr)
        return -EFAULT;

#ifdef USE_KALLSYMS
	p_ia32_sys_call_table = (void **) kallsyms_lookup_name("ia32_sys_call_table");
#else
	char *ia32_strend;
    p_ia32_sys_call_table = (void **) simple_strtoul(splxmod_ia32_addr, &ia32_strend, 0);
#endif
	
    if (p_ia32_sys_call_table == 0)
        return -EFAULT;
#endif

    /*get do_execve address*/
    if (!splxmod_execve_addr)
        return -EFAULT;

    DPRINTK(1, "%d: splxmod_execve_addr = 0x%s:\n", cur_pid, splxmod_execve_addr);
    p_do_execve = (void **) simple_strtoul(splxmod_execve_addr, &execve_strend, 0);
#ifndef X86_64
    DPRINTK(1, "%d: p_do_execve = 0x%x:\n", cur_pid, (int) p_do_execve);
#else
    DPRINTK(1, "%d: p_do_execve = 0x%lx:\n", cur_pid, (long unsigned int)p_do_execve);
#endif
    orig_do_execve = (void *) p_do_execve;

#ifndef X86_64
    DPRINTK(1, "%d: orig_do_execve = 0x%x:\n", cur_pid, (int) orig_do_execve);
#else
    DPRINTK(1, "%d: orig_do_execve = 0x%lx:\n", cur_pid, (long unsigned int)orig_do_execve);
#endif

    if (p_do_execve == 0)
    {
        return -EFAULT;
    }
#ifdef IA32_HOOK
    /*get compat_do_execve address*/

//failed to load khm when splxmod_compat_do_execve_addr is equal to 0. In fact, we use the lsm hook.
#ifndef USE_LSM_HOOK
    if (!splxmod_compat_do_execve_addr)
    {
        return -EFAULT;
    }
#endif      
    DPRINTK(1, "%d: splxmod_compat_do_execve_addr = 0x%s:\n", cur_pid, splxmod_compat_do_execve_addr);

    p_sys32_execve = (void **) simple_strtoul(splxmod_compat_do_execve_addr, &sys32_execve_strend, 0);

    DPRINTK(1, "%d: p_sys32_execve = 0x%lx:\n", cur_pid, (long unsigned int)p_sys32_execve);

    IA32_orig_compat_do_execve = (void *) p_sys32_execve;


    DPRINTK(1, "%d: IA32_orig_compat_do_execve = 0x%lx:\n", cur_pid, (long unsigned int)IA32_orig_compat_do_execve);

#ifndef USE_LSM_HOOK
    if (p_sys32_execve == 0)
    {
        DPRINTK(1, "Quit init mod because  splxmod_compat_do_execve_addr is 0");
        return -EFAULT;
    }
#endif

#endif

#ifdef X86_64
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,6,0)

    if (!splxmod_ret_addr)
        return -EFAULT;
    p_int_ret_from_sys_call = (void **) simple_strtoul(splxmod_ret_addr, &ret_strend, 0);
    DPRINTK(1, "%d: p_int_ret_from_sys_call = 0x%lx:\n", cur_pid, (long unsigned int)p_int_ret_from_sys_call);
    if (p_int_ret_from_sys_call == 0)
        return -EFAULT;
#endif
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    if (!mod_member_present(&__this_module, can_unload))
        return -EBUSY;

    __this_module.can_unload = &can_unload;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0) && LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    EXPORT_NO_SYMBOLS;
#endif

    /* setting initializations */
    if (kini.dirs)
    {
        tmp = DIRS_DEF;
        if(NULL == tmp)
        {
            len = 0;
        }
        else
        {
            len = strlen(tmp);
        }
        kini.dirs = (char *)kmalloc(sizeof(char) * (len + 1), GFP_KERNEL);
        strncpy(kini.dirs, tmp, len+1);
        kini.dirs[len] = '\0';
    }
    if (kini.exts)
    {
        tmp = EXTS_DEF;
        if(NULL == tmp)
        {
            len = 0;
        }
        else
        {
            len = strlen(tmp);
        }
        kini.exts = (char *)kmalloc(sizeof(char) * (len + 1), GFP_KERNEL);
        strncpy(kini.exts, tmp, len + 1);
        kini.exts[len] = '\0';
    }
    if (kini.exc_dirs)
    {
        tmp = EXC_DIRS_DEF;
        if(NULL == tmp)
        {
            len = 0;
        }
        else
        {
            len = strlen(tmp);
        }
        kini.exc_dirs = (char *)kmalloc(sizeof(char) * (len + 1), GFP_KERNEL);
        strncpy(kini.exc_dirs, tmp, len + 1);
        kini.exc_dirs[len] = '\0';
    }
    if (kini.exc_fils)
    {
        tmp = EXC_FILS_DEF;
        if(NULL == tmp)
        {
            len = 0;
        }
        else
        {
            len = strlen(tmp);
        }
        kini.exc_fils = (char *)kmalloc(sizeof(char) * (len + 1), GFP_KERNEL);
        strncpy(kini.exc_fils, tmp, len + 1);
        kini.exc_fils[len] = '\0';
    }
    if (kini.exc_exts)
    {
        tmp = EXC_EXTS_DEF;
        if(NULL == tmp)
        {
            len = 0;
        }
        else
        {
            len = strlen(tmp);
        }
        kini.exc_exts = (char *)kmalloc(sizeof(char) * (len + 1), GFP_KERNEL);
        strncpy(kini.exc_exts, tmp, len + 1);
        kini.exc_exts[len] = '\0';
    }

    busy_timeout_HZ = HZ * (kini.waitq_timeout / 1000);
    scan_timeout_HZ = HZ * kini.vsapi_timeout;
    exc_pid_ary = (pid_t *)kmalloc(sizeof(pid_t) * MAX_EXC_PID_DEF,
                                   GFP_KERNEL);
    vsc_pid_ary = (pid_t *)kmalloc(sizeof(pid_t) * MAX_VSC_PID_DEF,
                                   GFP_KERNEL);
    parseAddDirs(kini.dirs);
    parseAddExts(kini.exts);
    parseAddExcDirs(kini.exc_dirs);
    parseAddExcFils(kini.exc_fils);
    parseAddExcExts(kini.exc_exts);

    //Add default exclude command list
    len = strlen(DEF_EXC_COMM);
    exc_cmds = (char *)kmalloc(len + 1, GFP_KERNEL);
    strncpy(exc_cmds, DEF_EXC_COMM, len + 1);
    exc_cmds[len] = '\0';
    parseAddExcComms(exc_cmds);



    /* dynamic allocation of major number */
    major = REGISTER_CHRDEV (0, DEVICE_NAME, &splxmod_fops);
    if (major < 0)
    {
        WPRINTK("%d: %s: can't get major %d\n", cur_pid, cpsMethod, major);
        return (major);
    }
#ifdef X86_64

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16)

    {
        int ret;
        ret = 0;
        /* First compatible ones */
        ret = register_ioctl32_conversion(SIOCSETINIFIL_32, handle_SIOCSETINIFIL);
        ret |= register_ioctl32_conversion(SIOCGETNXTFIL_32, handle_SIOCGETNXTFIL);
        ret |= register_ioctl32_conversion(SIOCPUTEXCPID, (void *)sys_ioctl);
        ret |= register_ioctl32_conversion(SIOCPUTVSCPID, (void *)sys_ioctl);
        ret |= register_ioctl32_conversion(SIOCPUTLSTRES_32, handle_SIOCPUTLSTRES);
        ret |= register_ioctl32_conversion(SIOCGETKHMINFO_32, handle_SIOCGETKHMINFO);
        ret |= register_ioctl32_conversion(SIOCADDONEDENYWRITEFILE_32, handle_SIOCADDONEDENYWRITEFILE);
        ret |= register_ioctl32_conversion(SIOCSETDENYACCESSFILELIST_32, handle_SIOCSETDENYACCESSFILELIST);
        ret |= register_ioctl32_conversion(SIOCADDONEDENYWRITEDIR_32, handle_SIOCADDONEDENYWRITEDIR);
        ret |= register_ioctl32_conversion(SIOCSETDENYACCESSDIRLIST_32, handle_SIOCSETDENYACCESSDIRLIST);
        ret |= register_ioctl32_conversion(SIOCSETFILTEREXTINDENYACCESSDIR_32, handle_SIOCSETFILTEREXTINDENYACCESSDIR);
        ret |= register_ioctl32_conversion(SIOCSETEXCEPTIONEXTENSION_32, handle_SIOCSETEXCEPTIONEXTENSION);
        ret |= register_ioctl32_conversion(SIOCCONTINUECWD, (void *)sys_ioctl);

        //Add by errik 2010-9-29
        ret |= register_ioctl32_conversion(SIOCGETKHMVERSION, (void *)sys_ioctl);
        ret |= register_ioctl32_conversion(SIOCCLEARKHMLIST, (void *)sys_ioctl);
        ret |= register_ioctl32_conversion(SIOCGETFIRSTITEM_32, handle_SIOCGETFIRSTITEM);
        ret |= register_ioctl32_conversion(SIOCSETCOMMEXCLUSION_32, handle_SIOCSETCOMMEXCLUSION);
        //Add end


        /* These need to be handled by translation */
        if (ret)
            printk(KERN_ERR "SPLXMod:  Error registering ioctl32 translations\n");
    }

#endif

#endif

    proc_splx_base = proc_mkdir(PROC_NAME, NULL);
    if (NULL == proc_splx_base)
    {
        DPRINTK(LOG_WARNING, "%s: can't create  directory /prco/%s for splx kernel module\n", cpsMethod, PROC_NAME);
        return -EPERM;
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0)
    entry = create_proc_entry(KHM_ENTRY, 0600, proc_splx_base);
    if (!entry)
    {
        remove_proc_entry(PROC_NAME, 0);
        DPRINTK(LOG_WARNING, "%s: can't create KHM  entry /prco/%s/%s to access khm\n", cpsMethod, PROC_NAME, KHM_ENTRY);
        return -EPERM;
    }
    entry->read_proc = (read_proc_t *)khm_read_proc;
    entry->write_proc = (write_proc_t *)khm_write_proc;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30)
    entry->owner = THIS_MODULE;
#endif
    // add end;

    // Add for commads exclusion list
    // Creat the proc entry for command exclusion list
    entry = create_proc_entry(KHM_COMMS_ENTRY, 0600, proc_splx_base);
    if (!entry)
    {
        remove_proc_entry(KHM_ENTRY, proc_splx_base);
        remove_proc_entry(PROC_NAME, 0);
        DPRINTK(LOG_WARNING, "%s: can't create KHM commands exclusion list entry /prco/%s/%s to access khm\n", cpsMethod, PROC_NAME, KHM_COMMS_ENTRY);
        return -EPERM;
    }
    entry->read_proc = (read_proc_t *)khm_comms_read_proc;
    entry->write_proc = (write_proc_t *)khm_comms_write_proc;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30)
    entry->owner = THIS_MODULE;
#endif

#else
    entry = proc_create(KHM_ENTRY, 0600, proc_splx_base, &proc_khm_fops);
    if(!entry)
    {
        remove_proc_entry(PROC_NAME, 0);
        DPRINTK(LOG_WARNING, "%s: can't create KHM commands exclusion list entry /prco/%s/%s to access khm\n", cpsMethod, PROC_NAME, KHM_COMMS_ENTRY);
        return -EPERM;
    }

    entry = proc_create(KHM_COMMS_ENTRY, 0600, proc_splx_base, &proc_khm_comms_fops);
    if(!entry)
    {
        remove_proc_entry(KHM_ENTRY, proc_splx_base);
        remove_proc_entry(PROC_NAME, 0);
        DPRINTK(LOG_WARNING, "%s: can't create KHM commands exclusion list entry /prco/%s/%s to access khm\n", cpsMethod, PROC_NAME, KHM_COMMS_ENTRY);
        return -EPERM;
    }

#endif

    //Add end
    DPRINTK(LOG_CLOSE, "SPLX 3.0: KHM loaded. Version [%d]\n", KHM_VERSION);
    DPRINTK(1, "%d: %s: get outta init_module\n", cur_pid, cpsMethod);
    return 0; /* success */
}

static void __exit splxmod_exit(void)
{
    const char *cpsMethod = "cleanup_module";
    pid_t cur_pid = 0;
    CP_DBG_LVL;

    cur_pid = current->pid;
    DPRINTK(1, "%d: %s: get into cleanup_module\n", cur_pid, cpsMethod);
    // add by errik for debug log
    DPRINTK(LOG_DEBUG, "%d: %s: start to clean proc entry\n", cur_pid, cpsMethod);
    remove_proc_entry(KHM_ENTRY, proc_splx_base);
    remove_proc_entry(KHM_COMMS_ENTRY, proc_splx_base);
    remove_proc_entry(PROC_NAME, 0);
    DPRINTK(LOG_DEBUG, "%d: %s: clean proc entry sucess\n", cur_pid, cpsMethod);
    // add end
    delExcCommList();
    // free exc_coms
    if (exc_cmds)  kfree(exc_cmds);
    deleteListAll();
    delDirList();
    delExtList();
    delExcDirList();
    delExcFilList();
    delExcExtList();
    delDenyWriteList(DENYWRITE_FILE);
    delDenyWriteList(DENYWRITE_DIR);
    delDenyWriteList(DENYWRITE_FILTER_EXT);
    removeCacheAll();
    if (exc_pid_ary) kfree(exc_pid_ary);
    if (vsc_pid_ary) kfree(vsc_pid_ary);
    if (kini.dirs) kfree(kini.dirs);
    if (kini.exts) kfree(kini.exts);
    if (kini.exc_dirs) kfree(kini.exc_dirs);
    if (kini.exc_fils) kfree(kini.exc_fils);
    if (kini.exc_exts) kfree(kini.exc_exts);

    unregister_chrdev(major, DEVICE_NAME);

#ifdef X86_64
    {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16)

        int ret;
        ret = 0;
        /* First compatible ones */
        ret = unregister_ioctl32_conversion(SIOCSETINIFIL_32);
        ret |= unregister_ioctl32_conversion(SIOCGETNXTFIL_32);
        ret |= unregister_ioctl32_conversion(SIOCPUTEXCPID);
        ret |= unregister_ioctl32_conversion(SIOCPUTVSCPID);
        ret |= unregister_ioctl32_conversion(SIOCPUTLSTRES_32);
        ret |= unregister_ioctl32_conversion(SIOCGETKHMINFO_32);
        ret |= unregister_ioctl32_conversion(SIOCADDONEDENYWRITEFILE_32);
        ret |= unregister_ioctl32_conversion(SIOCSETDENYACCESSFILELIST_32);
        ret |= unregister_ioctl32_conversion(SIOCADDONEDENYWRITEDIR_32);
        ret |= unregister_ioctl32_conversion(SIOCSETDENYACCESSDIRLIST_32);
        ret |= unregister_ioctl32_conversion(SIOCSETFILTEREXTINDENYACCESSDIR_32);
        ret |= unregister_ioctl32_conversion(SIOCSETEXCEPTIONEXTENSION_32);
        ret |= unregister_ioctl32_conversion(SIOCCONTINUECWD);
        //Add by errik 2010-9-29
        ret |= unregister_ioctl32_conversion(SIOCGETKHMVERSION);
        ret |= unregister_ioctl32_conversion(SIOCCLEARKHMLIST);
        ret |= unregister_ioctl32_conversion(SIOCGETFIRSTITEM_32);
        ret |= unregister_ioctl32_conversion(SIOCSETCOMMEXCLUSION_32);
        //Add end

        /* These need to be handled by translation */
        //      ret |= register_ioctl32_conversion(DV1394_IOC32_GET_STATUS, handle_dv1394_get_status);
        if (ret)
            printk(KERN_ERR "SPLXMod:  Error unregistering ioctl32 translations\n");
#endif
    }
#endif
    DPRINTK(1, "%d: %s: get outta cleanup_module\n", cur_pid, cpsMethod);
    return;
}

module_init(splxmod_init);
module_exit(splxmod_exit);
